/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.ejb.mdb;

import java.sql.Timestamp;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;

import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.jms.JMSException;
import javax.jms.ObjectMessage;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.NotifyTypeConstant;
import com.clustercontrol.bean.OutputNotifyGroupInfo;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.commons.util.SendQueue;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyInfoContext;
import com.clustercontrol.notify.bean.QueueConstant;
import com.clustercontrol.notify.ejb.entity.NotifyRelationInfo;
import com.clustercontrol.notify.ejb.entity.NotifyRelationInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyRelationInfoUtil;
import com.clustercontrol.notify.factory.SelectNotify;

/**
 *
 * 各種通知の振り分けを行う Message-Driven Bean <BR>
 * <p>監視結果のステータス／イベントへの通知 及び メールの送信、ジョブの実行、リモートのsyslogへのログ出力を行います。<BR>
 * イベントは抑制機能があり通知された順序が重要となるため、Singletonにし直列で処理を行います。
 *
 * <!-- begin-xdoclet-definition -->
 * @ejb.bean name="NotifyControlBean" 
 *     acknowledge-mode="Auto-acknowledge"
 *     destination-type="javax.jms.Queue"
 *     
 *     transaction-type="Container"
 *     destination-jndi-name="queue/clustercontrolNotifyControl"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=AccessController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=RepositoryController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyInfo"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=EventLog"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=QuartzManager"
 * 
 * @jboss.container-configuration
 *  name="Singleton Message Driven Bean"
 * 
 *--
 * Server Runtime Specific Tags
 * If you are not using a specific runtime, you can safely remove the tags below.
 * @jonas.message-driven-destination jndi-name="queue/clustercontrolNotifyControl"
 * @jboss.destination-jndi-name name="queue/clustercontrolNotifyControl"
 *
 *--
 * <!-- end-xdoclet-definition -->
 **/
public class NotifyControlBean implements javax.ejb.MessageDrivenBean, javax.jms.MessageListener {
	
	private static final long serialVersionUID = -4596765959814807231L;

	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( NotifyControlBean.class );
	
	/** 通知情報キャッシュ */
	private static HashMap<String, NotifyInfo> m_notifyMap = null;
	
	/** コンテキスト情報 */
	private javax.ejb.MessageDrivenContext messageContext = null;
	
	/**
	 * コンテキスト情報を設定します。<BR>
	 * Message-Driven Bean がインスタンスプールに格納される際に行う処理を実装します。
	 * @see javax.ejb.MessageDrivenBean#setMessageDrivenContext(javax.ejb.MessageDrivenContext)
	 */
	public void setMessageDrivenContext(
			javax.ejb.MessageDrivenContext messageContext)
	throws javax.ejb.EJBException {
		this.messageContext = messageContext;
	}
	
	/** 
	 * 未処理。Message-Driven Bean が生成される際に行う処理を実装します。
	 * @ejb.create-method 
	 */
	public void ejbCreate() {
		//no specific action required for message-driven beans 
	}
	
	/** 
	 * 未処理。Message-Driven Bean が削除される際に行う処理を実装します。
	 * @see javax.ejb.MessageDrivenBean#ejbRemove()
	 */
	public void ejbRemove() {
		messageContext = null;
	}
	
	/** 
	 * 引数で指定された受信メッセージを元に、ログ出力をおこないます。<BR>
	 * Message-Driven Bean で行うビジネスロジックを実装します。特定のメッセージを受信した場合のみ処理を行います。
	 * 受信メッセージは下記の通りです。
	 * 
	 * <p><li>{@link com.clustercontrol.notify.message.OutputNotifyGroupInfo} : 含まれている通知グループIDを元にキャッシュしている通知情報グループを取得します。取得した通知情報に従って、受信メッセージに含まれている通知出力基本情報を基に各種通知を行います。
	 * <p><li>{@link com.clustercontrol.notify.bean.NotifyInfoContext} : 含まれている通知情報を、キャッシュしている通知情報に反映します。
	 *
	 * @see javax.jms.MessageListener#onMessage(javax.jms.Message)
	 *
	 * @see com.clustercontrol.notify.message.OutputNotifyGroupInfo
	 * @see com.clustercontrol.notify.bean.NotifyInfoContext
	 */
	@SuppressWarnings("unchecked")
	public void onMessage(javax.jms.Message message) {
		if(m_log.isDebugEnabled()){
			m_log.debug("onMessage start : Message Driven Bean got message " + message);
		}
		
		// 通知情報キャッシュ取得
		if(m_notifyMap == null){
			try{
				m_notifyMap = new SelectNotify().getNotifyMap();
				
			} catch (Exception e) {
				m_log.error("onMessage(): 通知情報キャッシュ取得時にエラーが発生しました。 " + e.getMessage());
			}
		}
		
		if(message instanceof ObjectMessage)
		{
			ObjectMessage msg = (ObjectMessage)message;
			
			try
			{
				Object objMsg = msg.getObject();
				if(objMsg instanceof OutputNotifyGroupInfo){
					OutputNotifyGroupInfo info = (OutputNotifyGroupInfo)objMsg;
					
					if(info != null){
						String notifyGroupId = info.getNotifyGroupId();
						if(notifyGroupId != null && !"".equals(notifyGroupId.trim())){
							
							notifyAction(info);
							
						}
					}
				}
				// 通知情報キャッシュ更新
				else if(objMsg instanceof NotifyInfoContext) {
					
					try{
						NotifyInfoContext notifyInfoContext = (NotifyInfoContext)objMsg;
						
						// 通知情報登録 または 更新時 
						if(NotifyInfoContext.TYPE_ADD == notifyInfoContext.getType() || 
								NotifyInfoContext.TYPE_UPDATE == notifyInfoContext.getType()){
							
							m_notifyMap.put(notifyInfoContext.getNotifyId(), notifyInfoContext.getNotifyInfo());
						}
						// 通知情報削除時
						else if(NotifyInfoContext.TYPE_DELETE == notifyInfoContext.getType()){
							m_notifyMap.remove(notifyInfoContext.getNotifyId());
						}
					}
					catch(Exception e){
						m_log.error("onMessage(): 通知情報キャッシュ更新時にエラーが発生しました。" + e.getMessage());
					}
				}
				else
				{
					m_log.debug("onMessage(): ObjectMessage is not an expected instance. " + objMsg.toString());
				}
			}
			catch(JMSException e){
				m_log.error("onMessage():" + e.getMessage(), e);
			}
			catch(EJBException e){
				m_log.error("onMessage():" + e.getMessage(), e);
			}
			catch(Exception e){
				m_log.error("onMessage():" + e.getMessage(), e);
			}	
		}
		
		if(m_log.isDebugEnabled()){
			m_log.debug("onMessage end   : Message Driven Bean got message " + message);
		}
	}
	
	
	/**
	 * 引数で指定された情報を各種通知に出力します。<BR>
	 * イベント通知、ステータス通知、メール通知、ジョブ通知、ログエスカレーション通知を行います。
	 * 
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyEventBean#onMessage(javax.jms.Message)
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyStatusBean#onMessage(javax.jms.Message)
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyMailBean#onMessage(javax.jms.Message)
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyJobBean#onMessage(javax.jms.Message)
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyLogEscalationBean#onMessage(javax.jms.Message)
	 * 
	 * @param info 通知出力情報
	 */
	@SuppressWarnings("unchecked")
	public void notifyAction(OutputNotifyGroupInfo info) {
		
		Date now = new Date();
		Timestamp outputDate = new Timestamp(now.getTime());
		info.setOutputDate(outputDate);
		
		String notifyGroupId = info.getNotifyGroupId();
		try {
			// 通知関連情報を取得（通知ID順に）
			Collection<NotifyRelationInfo> ct = NotifyRelationInfoUtil.getLocalHome().findByNotifyGroupId(notifyGroupId);
			
			// 通知グループIDに登録されている数分の通知種別を振り分ける。
			if(ct != null && ct.size()>0 ) {
				Iterator<NotifyRelationInfo> itr = ct.iterator();
				while (itr.hasNext()) {
					NotifyRelationInfoLocal notifyRelation = (NotifyRelationInfoLocal) itr.next();
					String notifyId = notifyRelation.getNotifyId();
					NotifyInfo notifyInfo = m_notifyMap.get(notifyId);
					
					m_log.debug("notifyAction() notifyGroupId : " + notifyGroupId + ", notifyId : " + notifyId);
					
					int notifyType = notifyRelation.getNotifyType();
					int notifyFlg = notifyInfo.getValidFlg();
					SendQueue queue = null;
					String message = "";
					
					try {
						// 通知情報が「通知する」となっている場合
						if (notifyFlg == ValidConstant.TYPE_VALID) {
							switch (notifyType) {
							// イベント通知
							case NotifyTypeConstant.TYPE_EVENT:
								message = NotifyTypeConstant.STRING_EVENT;
								queue = new SendQueue(QueueConstant.QUEUE_NAME_EVENT);
								info.setNotifyId(notifyId);
								queue.put(info);
								break;
							// ステータス通知
							case NotifyTypeConstant.TYPE_STATUS:
								message = NotifyTypeConstant.STRING_STATUS;
								queue = new SendQueue(QueueConstant.QUEUE_NAME_STATUS);
								info.setNotifyId(notifyId);
								queue.put(info);
								break;
							// メール通知
							case NotifyTypeConstant.TYPE_MAIL:
								message = NotifyTypeConstant.STRING_MAIL;
								queue = new SendQueue(QueueConstant.QUEUE_NAME_MAIL);
								info.setNotifyId(notifyId);
								queue.put(info);
								break;
							// ジョブ通知
							case NotifyTypeConstant.TYPE_JOB:
								message = NotifyTypeConstant.STRING_JOB;
								queue = new SendQueue(QueueConstant.QUEUE_NAME_JOB);
								info.setNotifyId(notifyId);
								queue.put(info);
								break;
							// ログエスカレーション通知
							case NotifyTypeConstant.TYPE_LOG_ESCALATE:
								message = NotifyTypeConstant.STRING_LOG_ESCALATE;
								queue = new SendQueue(QueueConstant.QUEUE_NAME_LOGESCALATION);
								info.setNotifyId(notifyId);
								queue.put(info);
								break;
							default:
								break;
							}
						}
						else {
							// TODO:通知しない場合は、キャッシュ情報をアップデートしたほうが良い
						}
					} catch (JMSException e) {
//						m_log.debug("notifyAction() : メール送信メッセージ送信エラー : " + e.getMessage(), e);
						m_log.warn("notifyAction() : notifyId = " + notifyId + ", " + message + " : " + e.getMessage());
					} catch (NamingException e) {
						m_log.warn("notifyAction() : notifyId = " + notifyId + ", " + message + " : " + e.getMessage());
					}
					finally{
						if(queue != null){
							try {
								queue.terminate();
							} catch (JMSException e) {
								m_log.debug(e, e);
							}	
						}
					}
				}
			}
		} catch (FinderException e) {
			m_log.warn("notifyAction() : notifyGroupId = " + notifyGroupId +" not found.", e);
		} catch (NamingException e) {
			m_log.warn("notifyAction() : catch NamingExcepiton" + e.getMessage(), e);
		} 	
	}
}