//
//This file is part of the joeSNMP Java Library.
//
//joeSNMP is Copyright (C) 2002-2003 Blast Internet Services, Inc.  All rights reserved.
//joeSNMP is a derivative work, containing both original code, included code and modified
//code that was published under the GNU Lesser General Public License. Copyrights for modified 
//and included code are below.
//
//Copyright (C) 1999-2001 Oculan Corp. All rights reserved.
//
//This library is free software; you can redistribute it and/or
//modify it under the terms of the GNU Lesser General Public
//License as published by the Free Software Foundation; either
//version 2.1 of the License, or (at your option) any later version.
//
//This library is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//Lesser General Public License for more details.
//
//You should have received a copy of the GNU Lesser General Public
//License along with this library; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//See: http://www.fsf.org/copyleft/lesser.html
//
//For more information contact:
//   joeSNMP Licensing       <joesnmp-license@lists.sourceforge.net>
//   http://sourceforge.net/projects/joesnmp/
//

package org.opennms.protocols.snmp;

import java.util.TreeSet;

/**
 * Provides a simple timer scheduler for use by the
 * internal SnmpSession class. Resolution is provided
 * at the millisecond level.
 *
 * @see SnmpSession
 *
 * @author  <a href="mailto:aozarov@hotmail.com">Arie Ozarov</a>
 * @version 1.1.1.1
 */

class SnmpTimer
{
    private TimerLogic m_timerLogic;
    private Thread m_thread;
    
    public SnmpTimer()
    {
        m_timerLogic = new TimerLogic();
        m_thread = new Thread(m_timerLogic, "SnmpTimer");
        m_thread.setDaemon(true);
        m_thread.start();
    }
    
    /**
     * Schedules the runnable to be run after AT LEAST ms milliseconds
     * of time has expired. The runnable may be invoked in a delayed
     * manner, but will not be run BEFORE ms milliseconds have expired.
     *
     * @param runner    The runnable object
     * @param ms        The number of milliseconds to wait
     *
     */    
    public void schedule(Runnable runnable, long delay)
    {
        m_timerLogic.schedule(runnable, delay);
    }
    
    /**
     * Cancel pending tasks and call interrupt on the thread of the current running task (if such exists) 
     * @param waitForAll true if you want to wait until the current task is completed
     */
    public void cancel(boolean waitForAll)
    {
        m_timerLogic.cancel(waitForAll);
    }
    
    private static class TimerElement implements Comparable 
    {
        private Runnable m_runnable;
        private long m_when;
        private long m_sequence;
        
        private static int ms_nextInSequence;
        
        TimerElement(Runnable runnable, long delay)
        {
            m_runnable = runnable;
            m_when = System.currentTimeMillis() + delay;
            m_sequence = getNextInSequence();
        }

        public String toString()
        {
            return "TimerElement[ID=" + m_sequence + ", When=" + m_when + ", Runnable=" + m_runnable + "]";
        }
        
        private static synchronized long getNextInSequence()
        {
            return ms_nextInSequence++;
        }

        public int compareTo(Object o)
        {
            TimerElement compareTo = (TimerElement) o;
            if (m_when < compareTo.m_when)
                return -1;
            else if (m_when > compareTo.m_when)
                return 1;
            else
                return (m_sequence < compareTo.m_sequence) ? -1 : ((m_sequence == compareTo.m_sequence) ? 0 : 1);
        }
    }
    
    private static class TimerLogic implements Runnable
    {
        private boolean m_isActive = true;
        private Thread m_thread; 
        private TreeSet m_items = new TreeSet();
        
        public void run()
        {
            m_thread = Thread.currentThread();
            
            while (m_isActive)
            {
                try
                {
                    TimerElement timerElement = null;
                    
                    synchronized (m_items)
                    {
                        while (m_items.isEmpty())
                            m_items.wait();
                        
                        while (true)
                        {
                            timerElement = (TimerElement) m_items.first();
                            
                            long timeToWait = timerElement.m_when - System.currentTimeMillis();
                            
                            if (timeToWait > 0)
                                m_items.wait(timeToWait); 
                            
                            if (m_items.first() == timerElement && timerElement.m_when - System.currentTimeMillis() <= 0)
                            {
                                m_items.remove(timerElement);
                                break;
                            }
                        }
                    }
                    
                    timerElement.m_runnable.run();
                    
                }
                catch (InterruptedException ex)
                {
                    // was called by cancle
                }
                catch (Throwable te)
                {
                    // ignore;
                }
            }
        }

		public void cancel(boolean waitForAll) {
			synchronized (this) {
				if (m_isActive) {
					m_isActive = false;

					if (m_thread != null) {
						m_thread.interrupt();
					}
				}
			}

			if (waitForAll) {
				while (true) {
					try {
						if (m_thread != null) {
							m_thread.join();
						}
						break;
					} catch (InterruptedException e) {
						// ignore
					}
				}
			}
		}

        public synchronized void schedule(Runnable runnable, long delay)
        {
            if (m_isActive == false)
                throw new IllegalStateException("Timer is not active");
            
            TimerElement timerElement = new TimerElement(runnable, delay);
            
            synchronized (m_items)
            {
                m_items.add(timerElement);
                if (m_items.size() == 1 || m_items.first() == timerElement)
                    m_items.notify();
            }
        }
    }
}
