/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.accesscontrol.dialog;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.SizeConstant;
import com.clustercontrol.composite.StringVerifyListener;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.util.LoginManager;
import com.clustercontrol.util.Messages;

/**
 * アクセス[ログイン]ダイアログクラスです。
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class LoginDialog extends Dialog {
	
	
	/** ユーザID用テキスト */
    protected Text m_uidText = null;
    /** パスワード用テキスト */
    protected Text m_passwordText = null;
    /** ユーザID */
    protected String m_uid = null;
    /** パスワード */
    protected String m_password = null;
    
    /**ログイン先用テキスト**/
    protected Text m_urlText = null;
    /** ユーザID */
    protected String m_url = null;
    
    
    /** シェル */
    protected Shell m_shell = null;

    /**
     * コンストラクタ
     * 
     * @param parent 親シェル
     */
    public LoginDialog(Shell parent) {
        super(parent);
    }
    
    /**
     * ダイアログのコンテンツを配置します。
     * 
     * @see org.eclipse.jface.window.Window#createContents(org.eclipse.swt.widgets.Composite)
     */
    protected Control createContents(Composite parent) {
        Composite composite = (Composite) super.createContents(parent);

        // ダイアログエリアのカスタマイズ
        this.customizeDialog((Composite) this.getDialogArea());

        // ＯＫボタンのテキスト変更
        String okText = this.getOkButtonText();
        if (okText != null) {
            Button button = this.getButton(IDialogConstants.OK_ID);
            if (button != null) {
                button.setText(okText);
            }
        }

        // キャンセルボタンのテキスト変更
        String cancelText = this.getCancelButtonText();
        if (cancelText != null) {
            Button button = this.getButton(IDialogConstants.CANCEL_ID);
            if (button != null) {
                button.setText(cancelText);
            }
        }

        return composite;
    }

    /**
     * ダイアログエリアを生成します。
     * 
     * @param parent 親コンポジット
     */
    protected void customizeDialog(Composite parent) {
        m_shell = this.getShell();
        
        m_shell.setText(Messages
                .getString("dialog.accesscontrol.login"));

        RowLayout layout = new RowLayout();
        layout.type = SWT.VERTICAL;
        layout.spacing = 10;
        layout.marginWidth = 12;
        layout.marginHeight = 10;
        layout.fill = true;
        parent.setLayout(layout);

        Composite composite = null;
        RowLayout rowLayout = null;

        composite = new Composite(parent, SWT.NONE);
        rowLayout = new RowLayout();
        rowLayout.type = SWT.HORIZONTAL;
        rowLayout.spacing = 10;
        composite.setLayout(rowLayout);
        Label uidTitle = new Label(composite, SWT.NONE);
        uidTitle.setText(Messages.getString("user.id") + " : ");
        uidTitle.setLayoutData(new RowData(80, SizeConstant.SIZE_LABEL_HEIGHT));
        m_uidText = new Text(composite, SWT.BORDER);
        m_uidText.setLayoutData(new RowData(150, SizeConstant.SIZE_TEXT_HEIGHT));
        m_uidText.addVerifyListener(
        		new StringVerifyListener(DataRangeConstant.VARCHAR_64));

        composite = new Composite(parent, SWT.NONE);
        rowLayout = new RowLayout();
        rowLayout.type = SWT.HORIZONTAL;
        rowLayout.spacing = 10;
        composite.setLayout(rowLayout);
        Label passwordTitle = new Label(composite, SWT.NONE);
        passwordTitle.setText(Messages.getString("password") + " : ");
        passwordTitle.setLayoutData(new RowData(80, SizeConstant.SIZE_LABEL_HEIGHT));
        m_passwordText = new Text(composite, SWT.BORDER | SWT.PASSWORD);
        m_passwordText.setLayoutData(new RowData(150, SizeConstant.SIZE_TEXT_HEIGHT));
        m_passwordText.addVerifyListener(
        		new StringVerifyListener(DataRangeConstant.VARCHAR_64));
        
        
        composite = new Composite(parent, SWT.NONE);
        rowLayout = new RowLayout();
        rowLayout.type = SWT.HORIZONTAL;
        rowLayout.spacing = 10;
        composite.setLayout(rowLayout);
        Label urlTitle = new Label(composite, SWT.NONE);
        urlTitle.setText(Messages.getString("connection.url") + " : ");
        urlTitle.setLayoutData(new RowData(80, SizeConstant.SIZE_LABEL_HEIGHT));
        m_urlText = new Text(composite, SWT.BORDER);
        m_urlText.setLayoutData(new RowData(150, SizeConstant.SIZE_TEXT_HEIGHT));
        m_urlText.addVerifyListener(
        		new StringVerifyListener(DataRangeConstant.VARCHAR_256));
        
        
        new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);

        // 画面中央に
        Display display = m_shell.getDisplay();
        m_shell.setLocation(
                (display.getBounds().width - m_shell.getSize().x) / 2, (display
                        .getBounds().height - m_shell.getSize().y) / 2);

        //ユーザID反映
        reflectData();
        
        //通常はパスワードのみの入力のためパスワードにフォーカス
        m_passwordText.setFocus();
    }
    
    /**
     * ＯＫボタンのテキストを返します。
     * 
     * @return ＯＫボタンのテキスト
     */
    protected String getOkButtonText() {
        return Messages.getString("login");
    }

    /**
     * キャンセルボタンのテキストを返します。
     * 
     * @return キャンセルボタンのテキスト
     */
    protected String getCancelButtonText() {
        return Messages.getString("cancel");
    }

    /**
     * プレファレンスページに保存されているデータを読み出します。
     */
    private void reflectData() {
    	
    	
    	m_urlText.setText(readPrefURL()); 
    	m_uidText.setText(readPrefUID());

    }
    
    /**
     * ＯＫボタンが押された場合に呼ばれるメソッドで、入力値チェックを実施します。
     * <p>
     * 
     * エラーの場合、ダイアログを閉じずにエラー内容を通知します。
     */
    protected void okPressed() {
        ValidateResult result = this.validate();

        if (result == null || result.isValid()) {
            super.okPressed();
        } else {
            this.displayError(result);
        }
    }

    /**
     * ダイアログの入力値チェックを行います。
     * 
     * 必要に応じて、入力値チェックを実装して下さい。
     * 
     * @return ValidateResultオブジェクト
     */
    protected ValidateResult validate() {
        ValidateResult result = null;

        //ユーザID取得
        if (m_uidText.getText().length() > 0) {
        	setUserid(m_uidText.getText());
        } else {
            result = new ValidateResult();
            result.setValid(false);
            result.setID(Messages.getString("message.hinemos.1"));
            result.setMessage(Messages.getString("message.accesscontrol.1"));
            return result;
        }
        
        //パスワード取得
        if (m_passwordText.getText().length() > 0) {
        	setPassword(m_passwordText.getText());
        } else {
            result = new ValidateResult();
            result.setValid(false);
            result.setID(Messages.getString("message.hinemos.1"));
            result.setMessage(Messages.getString("message.accesscontrol.2"));
            return result;
        }

        //接続先URL取得
        if (m_urlText.getText().length() > 0) {
        	setUrl(m_urlText.getText());
        } else {
            result = new ValidateResult();
            result.setValid(false);
            result.setID(Messages.getString("message.hinemos.1"));
            result.setMessage(Messages.getString("message.hinemos.9"));
            return result;
        }
        
        return null;
    }
    
    /**
     * エラー内容を通知します。
     * <p>
     * 
     * 警告メッセージボックスにて、クライアントに通知します。
     * 
     * @param result
     *            ValidateResultオブジェクト
     */
    protected void displayError(ValidateResult result) {
    	MessageDialog.openWarning(
    			null, 
    			result.getID(), 
    			result.getMessage());
    }

	/**
	 * パスワードを返します。
	 * 
	 * @return パスワード
	 */
	public String getPassword() {
		return m_password;
	}

	/**
	 * パスワードを設定します。
	 * 
	 * @param password パスワード
	 */
	public void setPassword(String password) {
		m_password = password;
	}

	/**
	 * ユーザIDを返します。
	 * @return
	 */
	public String getUserid() {
		return m_uid;
	}

	/**
	 * 接続先URLを返します。
	 * @return
	 */
	public String getUrl() {
		return m_url;
	}

	
	/**
	 * ユーザIDを設定します。
	 * 
	 * @param uid ユーザID
	 */
	public void setUserid(String uid) {
		m_uid = uid;
	}
	
	/**
	 * 接続先URLを設定します。
	 * 
	 * @param uid ユーザID
	 */
	public void setUrl(String url) {
		m_url = url;
	}
	
	
	/**
	 * 接続先URLをプレファレンスから取得します。
	 * @return
	 */
    private String readPrefURL() {
        //リソースストアから接続先URLを取得
        IPreferenceStore store = ClusterControlPlugin.getDefault()
                .getPreferenceStore();
        String url = store.getString(LoginManager.KEY_EJB_URL);
        if (url.compareTo("") == 0) {
            url = LoginManager.VALUE_EJB_URL;
        }
        
   
        
        return url;
    }
	/**
	 * ユーザIDをプレファレンスから取得します。
	 * 
	 */
	private String readPrefUID(){
        //リソースストアから接続先URLを取得
        IPreferenceStore store = ClusterControlPlugin.getDefault()
                .getPreferenceStore();
        String uid = store.getString(LoginManager.KEY_EJB_UID);
        if (uid.compareTo("") == 0) {
            uid = LoginManager.VALUE_EJB_UID;
        }
        return uid;
		
	}
    
}