/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.run.dialog;

import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ProcessConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.composite.StringVerifyListener;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.run.bean.MonitorStringValueInfo;
import com.clustercontrol.monitor.run.composite.StringRunJobComposite;
import com.clustercontrol.monitor.run.util.StringValueInfoManager;
import com.clustercontrol.notify.composite.NotifyIdListComposite;
import com.clustercontrol.util.Messages;

/**
 * 文字列監視[作成・変更]ダイアログクラス<BR>
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
public class StringValueInfoCreateDialog extends CommonDialog {
	
	/** カラム数（タイトル）。 */
	public static final int WIDTH_TITLE = 5;
	
	/** カラム数（値）。 */
	public static final int WIDTH_VALUE = 2;

    /** 入力値を保持するオブジェクト。 */
    private MonitorStringValueInfo m_inputData = null;

    /** 入力値の正当性を保持するオブジェクト。 */
    private ValidateResult m_validateResult = null;

    /** 変更対象の識別キー。 */
    private String m_identifier = null;

    /** 変更対象の順序。 */
    private int m_order = Integer.MAX_VALUE;

    /** 説明。 */
    private Text m_textDescription = null;

    /** パターンマッチ表現。 */
    private Text m_textPattern = null;

    /** 条件に一致したら処理しない。 */
    private Button m_radioNotProcess = null;

    /** 条件に一致したら処理する。 */
    private Button m_radioProcess = null;
    
    /** 通知ID。 */
    private NotifyIdListComposite m_notifyId = null;

    /** 重要度。 */
    private Combo m_comboPriority = null;

    /** メッセージID。 */
    private Text m_textMessageId = null;

    /** メッセージ。 */
    private Text m_textMessage = null;
    
    /** ジョブ実行。 */
    private Button m_checkJobRun = null;
    
    /** ジョブ実行情報。 */
    private StringRunJobComposite m_jobRun = null;

    /** この設定を有効にする。 */
    private Button m_buttonValid = null;
    
    /**
     * 作成用ダイアログのインスタンスを返します。
     * 
     * @param parent 親のシェルオブジェクト
     */
    public StringValueInfoCreateDialog(Shell parent) {
        super(parent);
    }

    /**
     * 変更用ダイアログのインスタンスを返します。
     * 
     * @param parent 親のシェルオブジェクト
     * @param identifier 変更する文字列監視の判定情報の識別キー
     */
    public StringValueInfoCreateDialog(Shell parent, String identifier) {
        super(parent);

        this.m_identifier = identifier;
    }

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.notify.composite.NotifyIdListComposite#NotifyIdListComposite(Composite, int, boolean)
	 * @see com.clustercontrol.monitor.run.composite.StringRunJobComposite#StringRunJobComposite(Composite, int)
	 * @see com.clustercontrol.monitor.run.util.StringValueInfoManager#get(String)
	 * @see com.clustercontrol.monitor.run.bean.MonitorStringValueInfo
	 * @see #setInputData(MonitorStringValueInfo)
	 */
    protected void customizeDialog(Composite parent) {
        Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages.getString("dialog.monitor.run.create.modify.string"));

        // 変数として利用されるラベル
        Label label = null;
        // 変数として利用されるグリッドデータ
        GridData gridData = null;

        // レイアウト
        GridLayout layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 15;
        parent.setLayout(layout);
        
        /*
         * 説明
         */
        // ラベル
        label = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_TITLE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("description") + " : ");
        // テキスト
        this.m_textDescription = new Text(parent, SWT.BORDER | SWT.LEFT);
        gridData = new GridData();
        gridData.horizontalSpan = 10;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_textDescription.setLayoutData(gridData);
        this.m_textDescription.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_256));
        
        /*
         * 監視条件グループ
         */
        // ラベル
        label = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("monitor.rule") + " : ");

        // グループ
        Group monitorRuleGroup = new Group(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 15;
        monitorRuleGroup.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        monitorRuleGroup.setLayoutData(gridData);
        
        /*
         * パターンマッチ表現
         */
        // ラベル
        label = new Label(monitorRuleGroup, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_TITLE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("pattern.matching.expression") + " : ");
        // テキスト
        this.m_textPattern = new Text(monitorRuleGroup, SWT.BORDER | SWT.LEFT);
        gridData = new GridData();
        gridData.horizontalSpan = 10;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_textPattern.setLayoutData(gridData);
        this.m_textPattern.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_1024));
        
        /*
         * 出力の条件
         */
        // 条件に一致したら処理しない
        this.m_radioNotProcess = new Button(parent, SWT.RADIO);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_radioNotProcess.setLayoutData(gridData);
        this.m_radioNotProcess.setText(Messages.getString("don't.process.if.matched"));

        // 条件に一致したら処理する
        this.m_radioProcess = new Button(parent, SWT.RADIO);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_radioProcess.setLayoutData(gridData);
        this.m_radioProcess.setText(Messages.getString("process.if.matched"));
        this.m_radioProcess.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                setEnabled(m_radioProcess.getSelection());
            }
        });

        /*
         * 条件グループ
         */
        // 大グループ
        Group group = new Group(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 15;
        group.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        group.setLayoutData(gridData);
        
        /*
         * 通知
         */
		// 通知情報
		this.m_notifyId = new NotifyIdListComposite(group, SWT.NONE, true);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_notifyId.setLayoutData(gridData);

        /*
         * 重要度
         */
        // ラベル
		label = new Label(group, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_TITLE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("priority") + " : ");
        // コンボボックス
        this.m_comboPriority = new Combo(group, SWT.DROP_DOWN | SWT.READ_ONLY);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_comboPriority.setLayoutData(gridData);
        this.m_comboPriority.add(PriorityConstant.STRING_CRITICAL);
        this.m_comboPriority.add(PriorityConstant.STRING_WARNING);
        this.m_comboPriority.add(PriorityConstant.STRING_INFO);
        this.m_comboPriority.add(PriorityConstant.STRING_UNKNOWN);
        this.m_comboPriority.setText(PriorityConstant.STRING_CRITICAL);
        // 空白
        label = new Label(group, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 8;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);

        /*
         * メッセージID
         */
        // ラベル
        label = new Label(group, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_TITLE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("message.id") + " : ");
        // テキスト
        this.m_textMessageId = new Text(group, SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalSpan = 10;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_textMessageId.setLayoutData(gridData);
        this.m_textMessageId.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_64));

        /*
         * メッセージ
         */
        // ラベル
        label = new Label(group, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_TITLE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("message") + " : ");
        // テキスト
        this.m_textMessage = new Text(group, SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalSpan = 10;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_textMessage.setLayoutData(gridData);
        this.m_textMessage.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_256));
        
        // ジョブ実行
        this.m_checkJobRun = new Button(group, SWT.CHECK);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        m_checkJobRun.setLayoutData(gridData);
        m_checkJobRun.setText(Messages.getString("run") + " : ");
        m_checkJobRun.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
            	m_jobRun.setEnabled(m_checkJobRun.getSelection());
            }
        });
		
		// ジョブ実行グループ
        Group groupRunJob = new Group(group, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 1;
        groupRunJob.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupRunJob.setLayoutData(gridData);
        
        // 実行（実行，ジョブID，通知抑制と連動，呼出失敗時）
		this.m_jobRun = new StringRunJobComposite(
				groupRunJob, 
				SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 1;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_jobRun.setLayoutData(gridData);

        /*
         * 有効／無効
         */
        this.m_buttonValid = new Button(parent, SWT.CHECK);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_buttonValid.setLayoutData(gridData);
        this.m_buttonValid.setText(Messages.getString("setting.valid.confirmed"));

        // ラインを引く
        Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 15;
        line.setLayoutData(gridData);

        // サイズを最適化
        // グリッドレイアウトを用いた場合、こうしないと横幅が画面いっぱいになります。
        shell.pack();
        shell.setSize(new Point(550, shell.getSize().y));

        // 画面中央に
        Display display = shell.getDisplay();
        shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
                (display.getBounds().height - shell.getSize().y) / 2);

        // 識別子が指定されている場合、その情報を初期表示する。
        MonitorStringValueInfo info = StringValueInfoManager.getInstance().get(this.m_identifier);
        if (info == null) {
        	info = new MonitorStringValueInfo();
        }
        this.setInputData(info);
    }

	/**
     * 入力値を保持した文字列監視の判定情報を返します。
     * 
     * @return 判定情報
     */
    public MonitorStringValueInfo getInputData() {
        return this.m_inputData;
    }

	/**
	 * 引数で指定された判定情報の値を、各項目に設定します。
	 * 
	 * @param info 設定値として用いる判定情報
	 */
    protected void setInputData(MonitorStringValueInfo info) {

        this.m_inputData = info;

        // 各項目に反映
        this.m_identifier = info.getIdentifier();
        this.m_order = info.getOrderNo();

        // 説明
        if (info.getDescription() != null) {
            this.m_textDescription.setText(info.getDescription());
        }
        
        // パターンマッチ表現
        if (info.getPattern() != null) {
            this.m_textPattern.setText(info.getPattern());
        }
        
        // 処理する／しない
        if (info.getProcessType() == ProcessConstant.TYPE_YES) {
            this.m_radioProcess.setSelection(true);
        } else {
            this.m_radioNotProcess.setSelection(true);
        }
        
        // 通知
        if (info.getNotifyId() != null) {
            this.m_notifyId.setText(info.getNotifyId());
        }
        
        // 重要度
        this.m_comboPriority.setText(PriorityConstant.typeToString(info.getPriority()));
        
        // メッセージID
        if (info.getMessageId() != null) {
            this.m_textMessageId.setText(info.getMessageId());
        }
        
        // メッセージ
        if (info.getMessage() != null) {
            this.m_textMessage.setText(info.getMessage());
        }
        
        // ジョブ実行
        m_checkJobRun.setSelection(YesNoConstant.typeToBoolean(info.getJobRun()));
        
        // 各重要度のジョブ実行
        m_jobRun.setInputData(info);
        
        // 有効／無効
        if (info.isValidFlg() == ValidConstant.BOOLEAN_VALID) {
            this.m_buttonValid.setSelection(true);
        }
        
        // 入力制御
        this.setEnabled(this.m_radioProcess.getSelection());
    }

	/**
	 * 引数で指定された判定情報に、入力値を設定します。
	 * <p>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return 判定情報
	 * 
	 * @see #setValidateResult(String, String)
	 */
    protected MonitorStringValueInfo createInputData() {
    	MonitorStringValueInfo info = new MonitorStringValueInfo();

        info.setIdentifier(this.m_identifier);
        info.setOrderNo(this.m_order);

        // 説明
        if (this.m_textDescription.getText() != null
                && !"".equals((this.m_textDescription.getText()).trim())) {
            info.setDescription(this.m_textDescription.getText());
        } else {
            this.setValidateResult(Messages.getString("message.hinemos.1"),
                    Messages.getString("message.monitor.26"));
            return null;
        }
        
        // パターンマッチ表現
        if (this.m_textPattern.getText() != null
                && !"".equals((this.m_textPattern.getText()).trim())) {
        	
        	try{
    			Pattern.compile(this.m_textPattern.getText());
    			info.setPattern(this.m_textPattern.getText());
    		}
    		catch(PatternSyntaxException e){
    			this.setValidateResult(Messages.getString("message.hinemos.1"),
                        Messages.getString("message.monitor.27"));
                return null;
    		}
            
        } else {
            this.setValidateResult(Messages.getString("message.hinemos.1"),
                    Messages.getString("message.monitor.27"));
            return null;
        }

        // 処理する／しない
        if (this.m_radioProcess.getSelection()) {
        	info.setProcessType(ProcessConstant.TYPE_YES);
        } else {
            info.setProcessType(ProcessConstant.TYPE_NO);
        }
        String priorityText = this.m_comboPriority.getText();
        info.setPriority(PriorityConstant.stringToType(priorityText));
        
        // 通知ID
        String notifyId = this.m_notifyId.getText();
        if (notifyId != null && !"".equals(notifyId.trim())) {
        	info.setNotifyId(notifyId);
        }

        // メッセージID
        if (this.m_textMessageId.getText() != null
                && !"".equals((this.m_textMessageId.getText()).trim())) {
            info.setMessageId(this.m_textMessageId.getText());
        } else {
            if (this.m_radioProcess.getSelection()) {
        		this.setValidateResult(Messages.getString("message.hinemos.1"),
                        Messages.getString("message.monitor.28"));
                return null;	
            }
        }
        
        // メッセージ
        if (this.m_textMessage.getText() != null
                && !"".equals((this.m_textMessage.getText()).trim())) {
            info.setMessage(this.m_textMessage.getText());
        } else {
            if (this.m_radioProcess.getSelection()) {
        		this.setValidateResult(Messages.getString("message.hinemos.1"),
                        Messages.getString("message.monitor.29"));
                return null;	
            }
        }
        
        // ジョブ実行
        info.setJobRun(YesNoConstant.booleanToType(m_checkJobRun.getSelection()));
        
        // 各重要度のジョブ実行
        m_validateResult = m_jobRun.createInputData(info);
        if(m_validateResult != null){
        	return null;
        }
        
        // 有効
        if (this.m_buttonValid.getSelection()) {
            info.setValidFlg(ValidConstant.BOOLEAN_VALID);
        } else {
            info.setValidFlg(ValidConstant.BOOLEAN_INVALID);
        }
        
        return info;
    }

    /**
     * 無効な入力値をチェックをします。
     * 
     * @return 検証結果
     * 
     * @see #createInputData()
     */
    protected ValidateResult validate() {
        // 入力値生成
        this.m_inputData = this.createInputData();

        if (this.m_inputData != null) {
            return super.validate();
        } else {
            return m_validateResult;
        }
    }

    /**
     * ＯＫボタンのテキストを返します。
     * 
     * @return ＯＫボタンのテキスト
     */
    protected String getOkButtonText() {
        return Messages.getString("ok");
    }

    /**
     * キャンセルボタンのテキストを返します。
     * 
     * @return キャンセルボタンのテキスト
     */
    protected String getCancelButtonText() {
        return Messages.getString("cancel");
    }

    /**
     * コンポジットの選択可/不可を設定します。
     * 
     * @param enable 選択可の場合、<code> true </code>
     */
    private void setEnabled(boolean enable) {

    	if(enable){
        	this.m_jobRun.setEnabled(this.m_checkJobRun.getSelection());
        }
        else{
        	this.m_jobRun.setEnabled(enable);
        }
    	this.m_notifyId.setEnabled(enable);
    	this.m_comboPriority.setEnabled(enable);
        this.m_textMessageId.setEnabled(enable);
        this.m_textMessage.setEnabled(enable);
        this.m_checkJobRun.setEnabled(enable);
    }
    
	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 */
    private void setValidateResult(String id, String message) {

        this.m_validateResult = new ValidateResult();
        this.m_validateResult.setValid(false);
        this.m_validateResult.setID(id);
        this.m_validateResult.setMessage(message);
    }
}