/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */
package com.clustercontrol.monitor.util;

import java.io.IOException;
import java.net.URL;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;

import com.clustercontrol.accesscontrol.action.GetUserInfo;
import com.clustercontrol.bean.OutputFormConstant;
import com.clustercontrol.bean.Property;
import com.clustercontrol.monitor.MonitorPlugin;
import com.clustercontrol.monitor.factory.EventFilterProperty;
import com.clustercontrol.monitor.factory.StatusFilterProperty;
import com.clustercontrol.util.PropertyUtil;
import com.clustercontrol.util.ReportGenerator;

/**
 * イベント情報の帳票を出力するクラス<BR>
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
public class EventReportGenerator {
	
	/** JasperReports定義ファイルパス（PDF用）。 */
	public static final String JASPER_FILE_PDF = "conf/EventLogReportPdf.jasper";
	/** JasperReports定義ファイルパス（CSV用）。 */
	public static final String JASPER_FILE_CSV = "conf/EventLogReportCsv.jasper";
	
	/** パラメータ名（レポート出力日時）。 */
	public static final String REPORT_OUTPUT_DATE = "REPORT_OUTPUT_DATE";
	/** パラメータ名（レポート出力ユーザ）。 */
	public static final String REPORT_OUTPUT_USER = "REPORT_OUTPUT_USER";
	
	/** パラメータ名（重要度）。 */
	public static final String PRIORITY = "PRIORITY";
	/** パラメータ名（受信日時）。 */
	public static final String OUTPUT_DATE = "OUTPUT_DATE";
	/** パラメータ名（出力日時）。 */
	public static final String GENERATION_DATE = "GENERATION_DATE";
	/** パラメータ名（ファシリティID）。 */
	public static final String FACILITY_ID = "FACILITY_ID";
	/** パラメータ名（アプリケーション）。 */
	public static final String APPLICATION = "APPLICATION";
	/** パラメータ名（確認）。 */
	public static final String CONFIRM_FLG = "CONFIRM_FLG";
	/** パラメータ名（メッセージ）。 */
	public static final String MESSAGE = "MESSAGE";
	
	/** パラメータ名（確認ユーザ） */
	public static final String CONFIRM_USER = "CONFIRM_USER";
	
	/**
	 * 帳票を出力します。
	 * 
	 * @param facilityId ファシリティID
	 * @param condition 検索条件
	 * @param list 帳票出力用イベント情報一覧（{@link com.clustercontrol.monitor.bean.ReportEventInfo}のリスト）
	 * @param outputForm 帳票出力形式
	 * @param redirect 帳票出力先
	 * @return 出力に成功した場合、</code> true </code>
	 * 
	 * @see com.clustercontrol.util.ReportGenerator
	 */
	@SuppressWarnings("unchecked")
	public boolean run(String facilityId, Property condition, ArrayList list, int outputForm, String redirect) {
		
		ArrayList values = null;
		
		String priority = null;
		Date outputFromDate = null;
		Date outputToDate = null;
		Date generationFromDate = null;
		Date generationToDate = null;
		String application = null;
		String confirmFlg = null;
		String message = null;
		String confirmUser = null;
		
		// 重要度取得
		values = PropertyUtil.getPropertyValue(condition, EventFilterProperty.PRIORITY);
		if (!"".equals((String) values.get(0))) {
			priority = (String) values.get(0);
		}
		
		//更新日時（自）取得
		values = PropertyUtil.getPropertyValue(condition, EventFilterProperty.OUTPUT_FROM_DATE);
		if (values.get(0) instanceof Date) {
			outputFromDate = (Date) values.get(0);
		}
		
		//更新日時（至）取得
		values = PropertyUtil.getPropertyValue(condition, EventFilterProperty.OUTPUT_TO_DATE);
		if (values.get(0) instanceof Date) {
			outputToDate = (Date)values.get(0);
		}
		
		//出力日時（自）取得
		values = PropertyUtil.getPropertyValue(condition, EventFilterProperty.GENERATION_FROM_DATE);
		if (values.get(0) instanceof Date) {
			generationFromDate = (Date) values.get(0);
		}
		
		//出力日時（至）取得
		values = PropertyUtil.getPropertyValue(condition, EventFilterProperty.GENERATION_TO_DATE);
		if (values.get(0) instanceof Date) {
			generationToDate = (Date) values.get(0);
		}
		
		//アプリケーション取得
		values = PropertyUtil.getPropertyValue(condition, StatusFilterProperty.APPLICATION);
		if (!"".equals((String) values.get(0))) {
			application = (String) values.get(0);
		}
		
		//確認有無取得
		values = PropertyUtil.getPropertyValue(condition, EventFilterProperty.CONFIRMED);
		if (!"".equals((String) values.get(0))) {
			confirmFlg = (String) values.get(0);
		}
		
		//メッセージ取得
		values = PropertyUtil.getPropertyValue(condition, StatusFilterProperty.MESSAGE);
		if (!"".equals((String) values.get(0))) {
			message = (String) values.get(0);
		}
		
		//確認ユーザ取得
		values = PropertyUtil.getPropertyValue(condition, EventFilterProperty.CONFIRM_USER);
		if (!"".equals((String) values.get(0))) {
			confirmUser = (String) values.get(0);
		}
		
		//ユーザ名取得
		String userName = new GetUserInfo().getName();
		
		// パラメータの生成
		Map paramMap = new HashMap();
		paramMap.put(REPORT_OUTPUT_DATE, DateFormat.getDateTimeInstance().format(new Date()));
		paramMap.put(REPORT_OUTPUT_USER, userName);
		
		paramMap.put(PRIORITY, priority);
		paramMap.put(OUTPUT_DATE, getDateText(outputFromDate, outputToDate));
		paramMap.put(GENERATION_DATE, getDateText(generationFromDate, generationToDate));
		paramMap.put(FACILITY_ID, facilityId);
		paramMap.put(APPLICATION, application);
		paramMap.put(CONFIRM_FLG, confirmFlg);
		paramMap.put(MESSAGE, message);
		paramMap.put(CONFIRM_USER, confirmUser);
		
		// jasperファイルパス取得
		String jasperPath = null;
		if(outputForm == OutputFormConstant.TYPE_PDF){
			jasperPath = JASPER_FILE_PDF;
		}
		else if(outputForm == OutputFormConstant.TYPE_CSV){
			jasperPath = JASPER_FILE_CSV;
		}
		URL url = MonitorPlugin.getDefault().find(new Path(jasperPath));
		try {
			url = Platform.asLocalURL(url);
			jasperPath = url.getPath();
		} catch (IOException e) {
			e.printStackTrace();
			return false;
		}
		
		ReportGenerator report = new ReportGenerator(jasperPath, outputForm);
		return report.run(redirect, paramMap, list);
	}
	
	/**
	 * 日付文字列を返します。
	 * 
	 * @param fromDate 開始日時 
	 * @param toDate 終了日時
	 * @return
	 */
	private String getDateText(Date fromDate, Date toDate) {
		
		StringBuffer test = new StringBuffer();
		
		if(fromDate != null){
			test.append(DateFormat.getDateTimeInstance().format(fromDate));
		}
		if(fromDate != null || toDate != null){
			test.append(" - ");
		}
		if(toDate != null){
			test.append(DateFormat.getDateTimeInstance().format(toDate));
		}
		
		return test.toString();
	}
}