/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.notify.ejb.session;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionContext;
import javax.jms.JMSException;
import javax.naming.NamingException;

import org.eclipse.swt.SWT;

import com.clustercontrol.bean.TableColumnInfo;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyTabelDefine;
import com.clustercontrol.notify.factory.AddNotify;
import com.clustercontrol.notify.factory.AddSystemNotify;
import com.clustercontrol.notify.factory.DeleteNotify;
import com.clustercontrol.notify.factory.DeleteSystemNotify;
import com.clustercontrol.notify.factory.ModifyNotify;
import com.clustercontrol.notify.factory.ModifySystemNotify;
import com.clustercontrol.notify.factory.SelectNotify;
import com.clustercontrol.notify.factory.SelectSystemNotify;
import com.clustercontrol.util.Messages;

/**
 * 通知機能の管理を行う Session Bean <BR>
 * クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。
 *
 * @ejb.bean name="NotifyController"	
 *           jndi-name="NotifyController"
 *           type="Stateless"
 *           transaction-type="Container"
 *           
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=AccessController"
 *           
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 */
public abstract class NotifyControllerBean implements javax.ejb.SessionBean {
	
	/** コンテキスト情報。 */
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean の生成時に行う処理を実装します。
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * 通知情報をマネージャに登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyWrite"
	 *     method-intf="Remote"
	 * 
	 * @param info 作成対象の通知情報
	 * @return 作成に成功した場合、<code> true </code>
	 * @throws CreateException
	 * @throws NamingException
	 * @throws JMSException
	 * 
	 * @see com.clustercontrol.notify.factory.AddNotify#add(NotifyInfo)
	 */
	public boolean addNotify(NotifyInfo info) throws CreateException, NamingException, JMSException {
		
		// 通知情報を登録
		AddNotify notify = new AddNotify();
		return  notify.add(info);
	}
	
	/**
	 * マネージャ上の通知情報を変更します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyWrite"
	 *     method-intf="Remote"
	 * 
	 * @param info 変更対象の通知情報
	 * @return 変更に成功した場合、<code> true </code>
	 * @throws CreateException
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws JMSException
	 * 
	 * @see com.clustercontrol.notify.factory.ModifyNotify#modify(NotifyInfo)
	 * @see com.clustercontrol.notify.bean.NotifyInfo
	 */
	public boolean modifyNotify(NotifyInfo info) throws CreateException, FinderException, RemoveException, NamingException, JMSException {
		
		// 通知情報を更新
		ModifyNotify notify = new ModifyNotify();
		return  notify.modify(info);
	}
	
	/**
	 * 通知情報をマネージャから削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyWrite"
	 *     method-intf="Remote"
	 * 
	 * @param notifyId 削除対象の通知ID
	 * @return 削除に成功した場合、<code> true </code>
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws JMSException
	 * 
	 * @see com.clustercontrol.notify.factory.DeleteNotify#delete(String)
	 * @see com.clustercontrol.notify.bean.NotifyInfo
	 */
	public boolean deleteNotify(String notifyId) throws FinderException, RemoveException, NamingException, JMSException {
		
		// 通知情報を削除
		DeleteNotify notify = new DeleteNotify();
		return  notify.delete(notifyId);
	}
	
	/**
	 * 引数で指定された通知情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyRead"
	 *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param notifyId 取得対象の通知ID
	 * @return 通知情報
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.factory.SelectNotify#getNotify(String)
	 */
	public NotifyInfo getNotify(String notifyId) throws FinderException, NamingException {
		
		// 通知情報を取得
		SelectNotify notify = new SelectNotify();
		NotifyInfo info = notify.getNotify(notifyId);
		
		return info;
	}
	
	/**
	 * 通知ID一覧を取得します。<BR>
	 * 
	 * 戻り値のArrayListにはNotifyId(String)が順に
	 * 格納されています。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return 通知ID一覧
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.factory.SelectNotify#getNotifyIdList()
	 */
	public ArrayList getNotifyIdList() throws CreateException, FinderException, NamingException {
		
		// 通知ID一覧を取得
		SelectNotify notify = new SelectNotify();
		ArrayList list = notify.getNotifyIdList();
		
		return list;
	}
	
	/**
	 * 通知情報一覧を取得します。<BR>
	 * 
	 * 戻り値はArrayListのArrayListであり以下のように格納される。
	 * 
	 * <Pre>
	 * ArrayList info = new ArrayList();
	 *  info.add(notify.getNotifyId());
	 *  info.add(notify.getDescription());
	 *  info.add(new Boolean(ValidConstant.typeToBoolean(notify.getStatusFlg().intValue())));
	 *  info.add(new Boolean(ValidConstant.typeToBoolean(notify.getEventFlg().intValue())));
	 *  info.add(notify.getUpdateDate() == null ? null:new Date(notify.getUpdateDate().getTime()));
	 * 
	 *  list.add(info);
	 * </Pre>
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyRead"
	 *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return 通知情報一覧（Objectの2次元配列）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.factory.SelectNotify#getNotifyList()
	 */
	public ArrayList getNotifyList() throws CreateException, FinderException, NamingException {
		
		// 通知一覧を取得
		SelectNotify notify = new SelectNotify();
		ArrayList list = notify.getNotifyList();
		
		return list;
	}
	
	/**
	 * 通知情報のマップを取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyRead"
	 *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return 通知情報マップ
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.factory.SelectNotify#getNotifyMap()
	 */
	public HashMap<String, NotifyInfo> getNotifyMap() throws FinderException, NamingException {
		
		// 通知一覧を取得
		SelectNotify notify = new SelectNotify();
		HashMap<String, NotifyInfo> map = notify.getNotifyMap();
		
		return map;
	}
	
	/**
	 * 通知情報一覧のテーブル定義情報を返します。<BR>
	 * リストに、カラム毎にテーブルカラム情報をセットします。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @param locale ロケール情報
	 * @return テーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo#TableColumnInfo(java.lang.String, int, int, int)
	 * @see com.clustercontrol.notify.bean.NotifyTabelDefine
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getNotifyListTableDefine(Locale locale) {
		
		/** テーブル情報定義配列 */
		ArrayList tableDefine = new ArrayList();
		
		tableDefine.add(NotifyTabelDefine.NOTIFY_ID, 
				new TableColumnInfo(Messages.getString("notify.id", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(NotifyTabelDefine.DESCRIPTION, 
				new TableColumnInfo(Messages.getString("description", locale), TableColumnInfo.NONE, 300, SWT.LEFT));
		tableDefine.add(NotifyTabelDefine.STATUS, 
				new TableColumnInfo(Messages.getString("status.katakana", locale), TableColumnInfo.CHECKBOX, 60, SWT.LEFT));
		tableDefine.add(NotifyTabelDefine.EVENT, 
				new TableColumnInfo(Messages.getString("event.katakana", locale), TableColumnInfo.CHECKBOX, 60, SWT.LEFT));
		tableDefine.add(NotifyTabelDefine.UPDATE_TIME, 
				new TableColumnInfo(Messages.getString("update.time", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		
		return tableDefine;
	}
	
	/**
	 * システム通知情報を作成します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyWrite"
	 *     method-intf="Remote"
	 * 
	 * @param info 作成対象のシステム通知情報
	 * @return 作成に成功した場合、<code> true </code>
	 * @throws CreateException
	 * @throws NamingException
	 * @throws JMSException
	 * 
	 * @see com.clustercontrol.notify.factory.AddSystemNotify#add(NotifyInfo)
	 */
	public boolean addSystemNotify(NotifyInfo info) throws CreateException, NamingException, JMSException {
		
		// システム通知情報を登録
		AddSystemNotify notify = new AddSystemNotify();
		return  notify.add(info);
	}
	
	/**
	 * システム通知情報を変更します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyWrite"
	 *     method-intf="Remote"
	 * 
	 * @param info 変更対象のシステム通知情報
	 * @return 変更に成功した場合、<code> true </code>
	 * @throws CreateException
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws JMSException
	 * 
	 * @see com.clustercontrol.notify.factory.ModifySystemNotify#modify(NotifyInfo)
	 */
	public boolean modifySystemNotify(NotifyInfo info) throws CreateException, FinderException, RemoveException, NamingException, JMSException {
		
		// システム通知情報を更新
		ModifySystemNotify notify = new ModifySystemNotify();
		return  notify.modify(info);
	}
	
	/**
	 * システム通知情報を削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyWrite"
	 *     method-intf="Remote"
	 * 
	 * @param notifyId 削除対象の通知ID
	 * @return 削除に成功した場合、<code> true </code>
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws JMSException
	 * 
	 * @see com.clustercontrol.notify.factory.DeleteSystemNotify#delete(String)
	 */
	public boolean deleteSystemNotify(String notifyId) throws FinderException, RemoveException, NamingException, JMSException {
		
		// システム通知情報を削除
		DeleteSystemNotify notify = new DeleteSystemNotify();
		return  notify.delete(notifyId);
	}
	
	/**
	 * 引数で指定されたシステム通知情報を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyRead"
	 *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param notifyId 取得対象の通知ID
	 * @return システム通知情報
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.factory.SelectSystemNotify#getNotify(String)
	 */
	public NotifyInfo getSystemNotify(String notifyId) throws FinderException, NamingException {
		
		// システム通知情報を取得
		SelectSystemNotify notify = new SelectSystemNotify();
		NotifyInfo info = notify.getNotify(notifyId);
		
		return info;
	}
}
