/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.factory;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.jms.JMSException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.factory.AccessLock;
import com.clustercontrol.commons.util.SendQueue;
import com.clustercontrol.monitor.bean.QueueConstant;
import com.clustercontrol.notify.bean.NotifyEventInfo;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyInfoContext;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 通知情報を変更するクラス<BR>
 *
 * @version 1.0.0
 * @since 1.0.0
 */
public class ModifyNotify {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( ModifyNotify.class );
	
	/**
	 * 通知情報を変更します。
	 * <p>
	 * <ol>
	 *  <li>通知IDより、通知情報を取得します。</li>
	 *  <li>通知情報を変更します。</li>
	 *  <li>通知情報に設定されている通知イベント情報を削除します。</li>
	 *  <li>通知イベント情報を作成し、通知情報に設定します。</li>
	 *  <li>キャッシュ更新用の通知情報を生成し、ログ出力キューへ送信します。
	 *      監視管理機能で、監視管理機能で保持している通知情報キャッシュが更新されます。</li>
	 * </ol>
	 * 
	 * @param info 変更対象の通知情報
	 * @return 変更に成功した場合、<code> true </code>
	 * @throws CreateException
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws JMSException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyInfoBean
	 * @see com.clustercontrol.notify.ejb.entity.NotifyEventInfoBean
	 * @see com.clustercontrol.notify.factory.DeleteNotify#deleteEvents(Collection)
	 * @see com.clustercontrol.notify.bean.NotifyInfoContext#NotifyInfoContext(int, java.lang.String, com.clustercontrol.notify.bean.NotifyInfo)
	 * @see com.clustercontrol.notify.util.SendQueue
	 */
	@SuppressWarnings("unchecked")
	public boolean modify(NotifyInfo info) throws CreateException, FinderException, RemoveException, NamingException, JMSException {
    	//アクセスロック
    	AccessLock.lock(AccessLock.NOTIFY);
    	
		SendQueue queue = null;
		try 
		{
			// 通知情報を取得
			NotifyInfoLocal notify = NotifyInfoUtil.getLocalHome().findByPrimaryKey(info.getNotifyId());
			
			// 通知情報を更新
			if(notify != null){
				notify.setDescription(info.getDescription());
				notify.setStatusFlg(info.getStatusFlg());
				notify.setStatusValidPeriod(info.getStatusValidPeriod());
				notify.setStatusInvalidFlg(info.getStatusInvalidFlg());
				notify.setStatusUpdatePriority(info.getStatusUpdatePriority());
				notify.setEventFlg(info.getEventFlg());
				notify.setInhibitionFlg(info.getInhibitionFlg());
				notify.setInhibitionFrequency(info.getInhibitionFrequency());
				notify.setInhibitionPeriod(info.getInhibitionPeriod());
				notify.setUpdateDate(new Timestamp(new Date().getTime()));
				notify.setMailTemplateId(info.getMailTemplateId());
			}
			Collection ct = notify.getNotifyEventInfo();
			
			// 通知イベント情報を削除
			DeleteNotify delete = new DeleteNotify();
			delete.deleteEvents(ct);
			
			// 通知イベント情報を挿入
			ArrayList<NotifyEventInfo> eventList = info.getNotifyEventInfo();
			ArrayList<NotifyEventInfoLocal> eventLocalList = new ArrayList<NotifyEventInfoLocal>();
			
			NotifyEventInfo event = null;
			for(int index=0; index<eventList.size(); index++){
				event = (NotifyEventInfo)eventList.get(index);
				if(event != null){
					NotifyEventInfoLocal local = NotifyEventInfoUtil.getLocalHome().create(
							info.getNotifyId(),
							event.getPriority(),
							event.getEventNormalFlg(),
							event.getEventNormalState(),
							event.getEventInhibitionFlg(),
							event.getEventInhibitionState(),
							event.getMailFlg(),
							event.getMailAddress());
					eventLocalList.add(local);
				}
			}
			if(eventLocalList != null){
				ct.addAll(eventLocalList);
				notify.setNotifyEventInfo(ct);
			}
			
			NotifyInfoContext context = new NotifyInfoContext(NotifyInfoContext.TYPE_UPDATE, info.getNotifyId(), info);
			
			// 通知情報キャッシュを更新
			queue = new SendQueue(QueueConstant.QUEUE_NAME_LOG);
			queue.put(context);
			queue.terminate();
			
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { info.getNotifyId() };
            apllog.put("SYS", "002", args);
            m_log.debug("modify():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { info.getNotifyId() };
            apllog.put("SYS", "002", args);
            m_log.debug("modify():" + e.getMessage());
			throw e;
		} catch (RemoveException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { info.getNotifyId() };
            apllog.put("SYS", "002", args);
           m_log.debug("modify():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { info.getNotifyId() };
            apllog.put("SYS", "002", args);
            m_log.debug("modify():" + e.getMessage());
			throw e;
		} catch (JMSException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { info.getNotifyId() };
            apllog.put("SYS", "002", args);
            m_log.debug("modify():" + e.getMessage());
			throw e;
		}
		finally{
			if(queue != null){
				queue.terminate();	
			}
		}
		return true;
	}
	
}
