/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performanceMGR.bean;

import java.io.Serializable;
import java.util.Collection;
import java.util.HashMap;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * ある収集設定に含まれるファシリティのデバイス情報を集めたDTOクラス<P>
 * 
 * 収集プロパティ情報をクライアント側で表示する際には、収集時点のデバイス情報
 * が必要となるので、その情報をセットでやり取りするためのクラス
 * 
 * @version 1.0
 * @since 1.0
 */
public class DeviceDataSet implements Serializable {
	private HashMap deviceMap;
	
	/**
	 * コンストラクター
	 *
	 */
	public DeviceDataSet(){
		deviceMap = new HashMap();
	}
	
	/**
	 * デバイス情報を登録します。
	 * 
	 * @param deviceData　個々のデバイス情報
	 */
	public void putDeviceList(DeviceData deviceData){
		String facilitID = deviceData.getFacilityId();
		String deviceType = deviceData.getDeviceType();
		
		HashMap facilityDeviceMap = getFacilityDeviceMap(facilitID);
		// 指定のファシリティIDでの登録が初めての場合
		if(facilityDeviceMap == null){
			// ファシリティIDをキーに登録
			facilityDeviceMap = new HashMap();
			deviceMap.put(facilitID, facilityDeviceMap);
		}
		
		ArrayList deviceList = (ArrayList)facilityDeviceMap.get(deviceType);
		// 指定のデバイス種別での登録が初めての場合
		if(deviceList == null){
			// デバイス種別をキーに登録
			deviceList = new ArrayList();
			facilityDeviceMap.put(deviceType, deviceList);
		}
		// デバイス情報を追加
		deviceList.add(deviceData);
	}
	
	/**
	 * ファシリティが保持していた
	 * デバイス情報のリストを取得します。
	 * 
	 * 存在しない場合はnullを返す。
	 * 
	 * @param facilityID 取得したいファシリティ
	 * @param deviceType　デバイス種別 "Disk" or "Nic"
	 * @return　list Disk または Nic のデバイス情報のリスト
	 */
	public List getDeviceList(String facilityID, String deviceType){
		HashMap facilityDeviceMap = getFacilityDeviceMap(facilityID);
		if(facilityDeviceMap != null){
			return getDeviceList(facilityDeviceMap, deviceType);
		} else {
			return null;
		}
	}
	/**
	 * あるファシリティ（ノード）が保持していたデバイスで
	 * のデバイス種別がdiskのもののリストを取得します。
	 * 
	 * 存在しない場合はnullを返す。
	 * @param  facilityID 取得したいファイシリティのID
	 * @return　list　Diskに関するデバイス情報のリスト
	 */
	public List getDisk(String facilityID){
		return getDeviceList(facilityID, "disk");
	}
	/**
	 * あるファシリティ（ノード）が保持していたデバイスで
	 * のデバイス種別がnicのもののリストを取得します。
	 * 
	 * 存在しない場合はnullを返す。
	 * 
	 * @param  facilityID 取得したいファイシリティのID
	 * @return　list　NICに関するデバイス情報のリスト
	 */
	public List getNic(String facilityID){
		return getDeviceList(facilityID, "nic");
	}
	
	/**
	 * このインスタンスが保持している全てのデバイス情報を返します。
	 * 
	 * @return　list　デバイス情報のリスト
	 */
	public List getAll(){
		ArrayList deviceList = new ArrayList();
		
		Set keys = deviceMap.keySet();
		
		Iterator itr = keys.iterator();
		
		while(itr.hasNext()){
			String facilitID = (String)itr.next();
			HashMap facilityDeviceMap = getFacilityDeviceMap(facilitID);
			Collection dl = facilityDeviceMap.values();
			Iterator itrDl = dl.iterator();
			
			while(itrDl.hasNext()){
				deviceList.addAll((ArrayList)(itrDl.next()));
			}
		}
		
		return deviceList;
	}
	
	/**
	 * ファシリティ毎のデバイス情報を保持するハッシュマップを返します。
	 * 
	 * @param facilityID　取得したいファシリティのID
	 * @return　HashMap ファイシリティに対応するデバイス全体のHashMap
	 */
	private HashMap getFacilityDeviceMap(String facilityID){
		return (HashMap)deviceMap.get(facilityID);
	}
	
	/**
	 * デバイスマップから指定のデバイスタイプに合致するもののリストを返します。
	 * 
	 * @param facilityDeviceMap
	 * @param deviceType
	 * @return
	 */
	private ArrayList getDeviceList(HashMap facilityDeviceMap, String deviceType){
		return (ArrayList)facilityDeviceMap.get(deviceType);
	}
}
