/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performanceMGR.ejb.session;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.swt.SWT;

import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.bean.PluginConstant;
import com.clustercontrol.bean.TableColumnInfo;
import com.clustercontrol.commons.ejb.UsedFacilityException;
import com.clustercontrol.commons.ejb.session.CheckFacility;
import com.clustercontrol.performanceMGR.bean.CollectedDataInfo;
import com.clustercontrol.performanceMGR.bean.CollectedDataSet;
import com.clustercontrol.performanceMGR.bean.CollectionTabelDefine;
import com.clustercontrol.performanceMGR.bean.CollectorItemInfo;
import com.clustercontrol.performanceMGR.bean.CollectorLockKey;
import com.clustercontrol.performanceMGR.bean.CollectorProperty;
import com.clustercontrol.performanceMGR.bean.CollectorRunStatus;
import com.clustercontrol.performanceMGR.bean.CollectorType;
import com.clustercontrol.performanceMGR.bean.DeviceData;
import com.clustercontrol.performanceMGR.bean.DeviceDataSet;
import com.clustercontrol.performanceMGR.dao.CalculatedDataDAO;
import com.clustercontrol.performanceMGR.dao.CollectorItemDAO;
import com.clustercontrol.performanceMGR.dao.DeviceDAO;
import com.clustercontrol.performanceMGR.ejb.bmp.Collector;
import com.clustercontrol.performanceMGR.ejb.bmp.CollectorHome;
import com.clustercontrol.performanceMGR.ejb.bmp.CollectorPK;
import com.clustercontrol.performanceMGR.ejb.bmp.CollectorUtil;
import com.clustercontrol.performanceMGR.ejb.bmp.RecordCollectorData;
import com.clustercontrol.performanceMGR.util.FacilityTreeDB;
import com.clustercontrol.performanceMGR.util.JNDIConnectionManager;
import com.clustercontrol.performanceMGR.util.LockManager;
import com.clustercontrol.performanceMGR.util.Messages;
import com.clustercontrol.performanceMGR.util.Repository;
import com.clustercontrol.performanceMGR.util.code.CollectorItemCodeTable;
import com.clustercontrol.performanceMGR.util.code.CollectorItemTreeItem;

/**
 *　性能管理機能の管理を行うコントローラクラス
 * クライアントからの Entity Bean へのアクセスは、このSession Bean を介して行います。
 * 
 * @version 1.0
 * @since 1.0
 *
 * <!-- begin-xdoclet-definition --> 
 * @ejb.bean name="CollectorController"	
 *           jndi-name="CollectorController"
 *           type="Stateful" 
 *           transaction-type="Container"
 *           
 * <!-- @jboss.clustered -->
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=AccessController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=RepositoryController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=QuartzManager"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorInfo"
 *           
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 * <!-- end-xdoclet-definition --> 
 * @generated
 */
public abstract class CollectorControllerBean implements javax.ejb.SessionBean, CheckFacility {
	
	//	ログ出力
	protected static Log m_log = LogFactory.getLog(CollectorControllerBean.class);
	
	/** 性能値の取得対象がスコープ　*/
	public static final int CURRENT_SCOPE = 0;
	/** 性能値が取得対象がサブスコープ*/
	public static final int SUB_SCOPE = 1;
	
	private String m_collectorID;  // 収集ID
	private int m_collectorType;   // 収集種別（リアルタイム もしくは 実績）
	
	/**
	 * SessionBean生成時にコンテナから呼ばれます。
	 *
	 * <!-- begin-user-doc -->
	 * The  ejbCreate method.
	 * <!-- end-user-doc -->
	 *
	 * <!-- begin-xdoclet-definition --> 
	 * @ejb.create-method 
	 * <!-- end-xdoclet-definition --> 
	 * @generated
	 */
	public void ejbCreate() throws CreateException {
		// 収集のステータスを設定
		this.m_collectorType = CollectorType.UNKNOWN;		
	}
	
	/**
	 * SessionBean削除時にコンテナから呼ばれます。
	 **/
	public void ejbRemove() {
		m_log.debug("ejbRemove() start : " + m_collectorID);
		
		// リアルタイム収集中の場合
		if(this.m_collectorType == CollectorType.REALTIME){
			try {			
				CollectorHome home = JNDIConnectionManager.getCollectorHome();
				Collector myBean = home.findByPrimaryKey(new CollectorPK(m_collectorID));
				
				// 収集を停止
				myBean.stopCollect();
				
				// DBの収集設定を削除
				myBean.remove();
			} catch (FinderException e) {
				throw new EJBException(e.getMessage(), e);
			} catch (NamingException e) {
				throw new EJBException(e.getMessage(), e);
			} catch (EJBException e) {
				throw new EJBException(e.getMessage(), e);
			} catch (RemoveException e) {
				throw new EJBException(e.getMessage(), e);
			} catch (RemoteException e) {
				throw new EJBException(e.getMessage(), e);
			}
		}
		m_log.debug("ejbRemove() end : " + m_collectorID);
	}
	
	/**
	 * SessionBeanインスタンスが「非活性化」状態になる前にコンテナから呼ばれます。
	 **/
	public void ejbPassivate(){
	}
	
	/**
	 * リアルタイム収集用のEntityBeanを生成します。
	 * 
	 * @param facilityID ファシリティID
	 * @param interval 収集間隔
	 * @param colletorItemList　収集項目設定のリスト
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceExecute"
     *  method-intf="Remote"
     *  
	 **/
	public void createRealtimeColletor(String facilityID, int interval, List colletorItemList) {
		m_collectorType = CollectorType.REALTIME;
		
		// 収集IDを生成して設定
		m_collectorID = (CollectorUtil.generateGUID(this));
		
		// デバッグ出力
		m_log.debug("createRealtimeColletor() start : " + 
				Thread.currentThread().getName() + "\t" + m_collectorID);
		
		try {
			RecordCollectorData data = new RecordCollectorData();
			data.setCollectorID(m_collectorID);
			data.setCollectorType(CollectorType.REALTIME);
			data.setLabel(Thread.currentThread().getName());  // スレッド名を説明項目に入力
			data.setFacilityID(facilityID);
			data.setInterval(interval);
			data.setPeriod(-1);  // 収集期間は無期限を指定

			// スコープツリーを取得
			Repository repository = new Repository();
			FacilityTreeItem facilityTree = repository.getFacilityTree(facilityID);
			data.setFacilityTree(facilityTree);

			// 対象スコープが存在しない場合はエラー
			if(facilityTree == null){
				// エラー処理
				return;				
			}
			
			// RealtimeCollector Entity Bean を生成
			CollectorHome home = JNDIConnectionManager.getCollectorHome();
			Collector myBean = home.create(data, colletorItemList);
			
			// 収集の開始を指示
			myBean.startCollect();
		} catch (CreateException e) {
			throw new EJBException(e);
		} catch (NamingException e){
			throw new EJBException(e);
		} catch (RemoteException e) {
			throw new EJBException(e);
		}
		
		m_log.debug("createRealtimeColletor() end : " + Thread.currentThread().getName() + "\t" + m_collectorID);
	}
	
	/**
	 * リアルタイム収集の収集済みデータを取得します。
	 * @param facilityID ファシリティID
	 * @param itemID　収集項目ID
	 * @param option　取得の対象を指示（指定スコープ（CURRENT_SCOPE）もしくは指定スコープのサブスコープ（SUB_SCOPE））
	 * @return　収集済み性能データ（CollectedDataInfo）のリスト
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceExecute"
     *  method-intf="Remote"
     *  
	 */
	public Collection getRealtimeCollectedData(String facilityID, int itemID, int option) { 
		try {
			CollectorHome home = JNDIConnectionManager.getCollectorHome();
			Collector myBean = home.findByPrimaryKey(new CollectorPK(m_collectorID));
			
			// 性能値を取得
			if(option == CURRENT_SCOPE){
				CollectedDataInfo data = myBean.getValue(facilityID, itemID);
				ArrayList<CollectedDataInfo> ret = new ArrayList<CollectedDataInfo>();
				ret.add(data);
				return ret;
			} else if(option == SUB_SCOPE){
				return myBean.getSubScopeValues(facilityID, itemID);
			} else {
				return null;
			}
		} catch (FinderException e) {
			throw new EJBException(e);
		} catch (NamingException e){
			throw new EJBException(e);
		} catch (RemoteException e) {
			throw new EJBException(e);
		}
	}
	
	/**
	 * リアルタイム収集の収集済みデータを取得します。 	 
	 * @param facilityID　ファシリティID
	 * @param itemIDs 収集項目の配列
	 * @return 収集済み性能データ（CollectedDataInfo）のリスト
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceExecute"
     *  method-intf="Remote"
	 *
	 */
	public Collection getRealtimeCollectedData(String facilityID, int[] itemIDs) {
		try{
			ArrayList<CollectedDataInfo> ret = new ArrayList<CollectedDataInfo>();

			CollectorHome home = JNDIConnectionManager.getCollectorHome();
			Collector myBean = home.findByPrimaryKey(new CollectorPK(m_collectorID));
			
			for(int i=0; i<itemIDs.length; i++){
				// 性能値を取得
				ret.add(myBean.getValue(facilityID, itemIDs[i]));
			}
			
			return ret;
		} catch (FinderException e) {
			throw new EJBException(e);
		} catch (NamingException e){
			throw new EJBException(e);
		} catch (RemoteException e) {
			throw new EJBException(e);
		}
	}
	
	/**
	 * リアルタイム収集の収集済みデータを取得します。
	 * 指定のファシリティで収集されている全ての収集項目の性能値データを取得します。
	 * 
	 * @param facilityID　ファシリティID
	 * @return 収集済み性能データ（CollectedDataInfo）のリスト
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceExecute"
     *  method-intf="Remote"
	 *
	 **/
	public Collection getRealtimeCollectedDataAll(String facilityID) {
		try {
			CollectedDataInfo[] returnData = null;
			
			CollectorHome home = JNDIConnectionManager.getCollectorHome();
			Collector myBean = home.findByPrimaryKey(new CollectorPK(m_collectorID));
			
			// 性能値を取得
			returnData = myBean.getValueAll(facilityID);
			return Arrays.asList(returnData); 
		} catch (FinderException e) {
			throw new EJBException(e);
		} catch (NamingException e){
			throw new EJBException(e);
		} catch (RemoteException e) {
			throw new EJBException(e);
		}
	}
	
	/**
	 * 収集設定のプロパティ情報を取得します。
	 * @param collectorID 収集ID
	 * @return 収集設定のプロパティ情報
	 * 
	 * @ejb.interface-method
	 *	view-type="remote"
	 *
	 *	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
	 *
	 **/
	public CollectorProperty getCollectorProperty(String collectorID) { 
		// 収集IDを設定
		m_collectorID = collectorID;
		
		CollectorProperty property = new CollectorProperty();
		
		RecordCollectorData collectorData = null;
		
		try {
			// Entity Bean を取得する
			CollectorHome home = JNDIConnectionManager.getCollectorHome();
			Collector myBean = home.findByPrimaryKey(new CollectorPK(collectorID));
			
			// 収集定義を取得する
			collectorData = myBean.getRecordColletorData();
			
			// デバイス情報を設定
			property.setDeviceSet(getDeviceSet(collectorID));
			
			// スコープツリー情報を設定
			collectorData.setFacilityTree(this.getFacilityTree(collectorID));
			
			// 収集設定を格納
			property.setCollectorData(collectorData);
			
			// 収集項目用のDAOを生成
			CollectorItemDAO dao = new CollectorItemDAO();
			
			
			ArrayList itemList = new ArrayList(
					dao.selectCollectorItemByCollectorID(collectorID)
			);
			
			property.setItemList(itemList);
						
			return property;
		} catch (FinderException e) {
			throw new EJBException(e);
		} catch (NamingException e){
			throw new EJBException(e);
		} catch (RemoteException e) {
			throw new EJBException(e);
		}
	}
	
	/**
	 * 実績情報収集用のEntityBeanを生成し、収集を開始します。
	 * @param collectorDef　収集の定義情報
	 * 
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceWrite"
     *  method-intf="Remote"
	 *
	 **/
	public void createRecordColletor(CollectorProperty collectorDef)
	throws DuplicateKeyException {
		// デバッグ出力
		m_log.debug("createRecordColletor() start : " + Thread.currentThread().getName() + "\t" + m_collectorID);
		
		// 収集種別を設定
		this.m_collectorType = CollectorType.RECORD;
		
		// このBeanに収集IDを設定
		m_collectorID = collectorDef.getCollectorData().getCollectorID();
		
		RecordCollectorData collectorData = collectorDef.getCollectorData();
		collectorData.setCollectorType(m_collectorType);
		
		// 設定の書き込みとEntity Beanの生成
		try {
			// 収集の開始を指示
			CollectorHome home = JNDIConnectionManager.getCollectorHome();
			Collector collector = home.create(collectorData, collectorDef.getItemList());
			
			collector.startCollect();
		} catch (DuplicateKeyException e) {
			throw e;  // キーが重複した場合はクライアントまで投げる
		} catch (CreateException e) {
			throw new EJBException(e);
		} catch (NamingException e){
			throw new EJBException(e);
		} catch (RemoteException e) {
			throw new EJBException(e);
		}
		
		m_log.debug("createRecordColletor() end : " + Thread.currentThread().getName() + "\t" + m_collectorID);
	}
	
	/**
	 * 実績収集を停止します。
	 * 
	 * @param collectorID 停止したい収集ID
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceExecute"
     *  method-intf="Remote"
	 *
	 */
	public void stopRecordCollector(String collectorID) {
		
		try {
			CollectorHome home = JNDIConnectionManager.getCollectorHome();
			Collector myBean = home.findByPrimaryKey(new CollectorPK(collectorID));
			
			// 収集の停止を指示
			myBean.stopCollect();
		} catch (FinderException e) {
			throw new EJBException(e);
		} catch (NamingException e){
			throw new EJBException(e);
		} catch (RemoteException e) {
			throw new EJBException(e);
		}
	}
	
	/**
	 * 実績収集で収集されたデータを取得します。
	 * @param collectorID 収集ID
	 * @param facilityID　ファシリティID
	 * @param itemID　収集項目ID
	 * @param startDate　取得したい始点の時刻
	 * @param endDate　　取得した終点の時刻
	 * @return　収集済み性能データ（CollectedDataInfo）のリスト
	 *  
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
	 *
	 */
	public List getRecordCollectedData(
			CollectorItemInfo item, String facilityID, Date startDate, Date endDate) {
		m_log.debug("getRecordCollectedData() " + facilityID);
		CalculatedDataDAO dao = new CalculatedDataDAO();
		
		List data = dao.select(
				item.getCollectorID(), 
				item.getCollectorItemCode(), 
				item.getDeviceIndex(),
				item.getDeviceName(),
				facilityID,
				startDate, 
				endDate);
		
		return data;
	}
	
	/**
	 * 実績収集で収集されたデータを取得します。
	 * @param collectorID 収集ID
	 * @param facilityIDs　ファシリティIDの配列
	 * @param itemIDs　収集項目IDの配列
	 * @param startDate　取得したい始点の時刻
	 * @param endDate　　取得した終点の時刻
	 * @return　収集データのデータセット
	 *  
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
	 *
	 */
	public CollectedDataSet getRecordCollectedData
	(CollectorItemInfo[] items, String[] facilityIDs, Date startDate, Date endDate) {
		CollectedDataSet ret = new CollectedDataSet();

		for(int j=0; j<facilityIDs.length; j++){
			for(int i=0; i<items.length; i++){
				List dataList = getRecordCollectedData(
						items[i], 
						facilityIDs[j],
						startDate,
						endDate);
				ret.setCollectedDataList(facilityIDs[j], items[i].getCollectorItemID(), dataList);
			}
		}
		
		return ret;
	}
	
	/**
	 * 実績収集で収集されたデータを取得します。
	 * @param collectorID 収集ID
	 * @param facilityID　ファシリティIDの配列
	 * @param itemID　収集項目IDの配列
	 * @param startDate　取得したい始点の時刻
	 * @param endDate　　取得した終点の時刻
	 * @return　収集データのデータセット
	 *  
	 * @ejb.interface-method
	 *  view-type="remote"
	 *
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
	 *
	 */
	public CollectedDataSet getRecordCollectedData
	(String collectorID, String[] facilityIDs, int[] itemID, Date startDate, Date endDate) {
		CollectedDataSet ret = new CollectedDataSet();

		CollectorItemDAO dao = new CollectorItemDAO();
		
		for(int j=0; j<facilityIDs.length; j++){
			for(int i=0; i<itemID.length; i++){
				// 収集項目をDBから取得します
				CollectorItemInfo item = dao.selectCollectorItem(collectorID, itemID[i]);
				
				List dataList = getRecordCollectedData(
						item, 
						facilityIDs[j],
						startDate,
						endDate);
				ret.setCollectedDataList(facilityIDs[j],item.getCollectorItemID(), dataList);
			}
		}
		
		return ret;
	}
	
	/**
	 * 実績収集設定及び、収集した性能情報を全て削除します。
	 * @param collectorID　削除したい収集ID
	 * 
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceWrite"
     *  method-intf="Remote"
	 *
	 **/
	public void deleteRecordCollector(String collectorID) throws RemoveException {
		// デバッグ出力
		m_log.debug("deleteRecordCollector() start : " + Thread.currentThread().getName() + "\t" + m_collectorID);
		
		// 削除ロックされているか否かをチェック
		if(LockManager.isLocked(collectorID)){
			m_log.debug("deleteRecordCollector()  : " + collectorID + " is locked.");
			throw new RemoveException(collectorID + " is locked.");
		}
		
		try {
			// 指定のCollectorIDのEntityBeanを取得
			CollectorHome home = JNDIConnectionManager.getCollectorHome();
			Collector myBean = home.findByPrimaryKey(new CollectorPK(collectorID));
			
			// まだ動作中の場合
			if(myBean.getStatus() == CollectorRunStatus.RUNNING){
				m_log.debug("deleteRecordCollector()  : " + collectorID + " is running.");
				throw new RemoveException(collectorID + " is running.");
			}
			
			myBean.remove();
		} catch (FinderException e) {
			throw new RemoveException(e.getMessage());
		} catch (NamingException e){
			throw new RemoveException(e.getMessage());
		} catch (RemoveException e) {
			throw e;
		} catch (RemoteException e) {
			throw new RemoveException(e.getMessage());
		}		
		
		m_log.debug("deleteRecordCollector() end : " + Thread.currentThread().getName() + "\t" + m_collectorID);
	}
	
	/**
	 * 収集設定時のファシリティツリー情報を取得します。
	 * @param collectorID 収集ID
	 * @return FacilityTreeItem ファシリティツリー
	 * 
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
	 *
	 **/
	public FacilityTreeItem getFacilityTree(String collectorID){
		return new FacilityTreeDB().getFacilityTree(collectorID);
	}
	
	/**
	 * 現在のリポジトリに登録されているデバイス情報から
	 * 指定のファシリティID以下の階層のノードのデバイスのリストを取得します。
	 * @param ファシリティID
	 * @return デバイス情報のセット
	 * 
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
	 *
	 **/
	public DeviceDataSet getCurrentDeviceSet(String facilityID) {
		DeviceDataSet dataSet = new DeviceDataSet();
		
		Repository repository = new Repository();
		List deviceList = repository.getDeviceList(facilityID);
		Iterator itr = deviceList.iterator();
		
		// デバイス情報を設定
		while(itr.hasNext()){
			dataSet.putDeviceList((DeviceData)itr.next());
		}
		
		return dataSet;
	}

	/**
	 * 収集設定に含まれたファシリティのデバイス情報を取得します。
	 * @param collectorID　収集ID
	 * @return デバイス情報のセット
	 * 
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
	 *
	 **/
	public DeviceDataSet getDeviceSet(String collectorID)
	throws FinderException, NamingException {
		DeviceDataSet dataSet = new DeviceDataSet();
		
		// デバイス情報用のDAOを生成
		DeviceDAO dao = new DeviceDAO();
		
		// デバイス情報をDBから取得する
		Collection devicList = dao.selectDeviceByCollectorID(collectorID);
		
		// デバイス情報を設定
		Iterator itr = devicList.iterator();
		while(itr.hasNext()){
			dataSet.putDeviceList((DeviceData)itr.next());
		}
		
		return dataSet;
	}
	
	/**
	 * 指定の収集IDのデータに対して削除ロックをかけます。
	 * 
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
     *  unchecked="true"
     *  method-intf="Remote"
	 *
	 **/
	public void lock(String collectorID, CollectorLockKey key) {
		LockManager.lock(collectorID, key);
	}

	/**
	 * 指定の収集IDのデータに対する削除ロックを解除します。
	 * 
	 * @ejb.interface-method
	 *	view-type="remote"
	 *
	 * @ejb.permission
     *  unchecked="true"
     *  method-intf="Remote"
	 *
	 **/
	public void unlock(CollectorLockKey key){
		LockManager.unlock(key);
	}
	
	/**
	 * 収集一覧のテーブル定義情報を返します。
	 * 
	 * @param locale ロケール
	 * @return TableColumnInfoのリスト
	 * 
	 * @ejb.interface-method
	 * @generated
	 * 
 	 * @ejb.permission
     *  unchecked="true"
     *  method-intf="Remote"
	 * 
	 */
	public ArrayList getCollectionListTableDefine(Locale locale) {
		
		/** テーブル情報定義配列 */
		ArrayList<TableColumnInfo> tableDefine = new ArrayList<TableColumnInfo>();

		tableDefine.add(CollectionTabelDefine.RUN_STATUS, 
				new TableColumnInfo(Messages.getString("run.status", locale), TableColumnInfo.STATE, 70, SWT.LEFT));
		tableDefine.add(CollectionTabelDefine.ERROR_STATUS, 
				new TableColumnInfo(Messages.getString("error.status", locale), TableColumnInfo.END_STATUS, 70, SWT.LEFT));
		tableDefine.add(CollectionTabelDefine.COLLECTOR_ID, 
				new TableColumnInfo(Messages.getString("collector.id", locale), TableColumnInfo.NONE, 200, SWT.LEFT));
		tableDefine.add(CollectionTabelDefine.DESCRIPTION, 
				new TableColumnInfo(Messages.getString("description", locale), TableColumnInfo.NONE, 220, SWT.LEFT));
		tableDefine.add(CollectionTabelDefine.SCOPE, 
				new TableColumnInfo(Messages.getString("scope", locale), TableColumnInfo.FACILITY, 220, SWT.LEFT));
		tableDefine.add(CollectionTabelDefine.START_DATE, 
				new TableColumnInfo(Messages.getString("start.date", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(CollectionTabelDefine.STOP_DATE, 
				new TableColumnInfo(Messages.getString("stop.date", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(CollectionTabelDefine.INTERVAL, 
				new TableColumnInfo(Messages.getString("interval", locale), TableColumnInfo.NONE, 70, SWT.LEFT));
		tableDefine.add(CollectionTabelDefine.COUNT, 
				new TableColumnInfo(Messages.getString("count", locale), TableColumnInfo.NONE, 70, SWT.LEFT));

		return tableDefine;
	}
	
	/**
	 * 実績収集のリストを返します。
	 * 
	 * 戻り値のArrayListはArrayLisyのArrayListとなる。
	 *　実行状態、ステータス、収集ID、説明、スコープのパス、収集開始時刻、収集終了時刻、収集間隔、収集回数
	 *　が格納されているArrayListのリストとなる。
	 *
	 * 下記のような構造。
	 *　インデックス1:実行状態、ステータス、収集ID、説明、スコープのパス、収集開始時刻、収集終了時刻、収集間隔、収集回数
	 *　インデックス2:実行状態、ステータス、収集ID、説明、スコープのパス、収集開始時刻、収集終了時刻、収集間隔、収集回数
	 *　インデックス3:実行状態、ステータス、収集ID、説明、スコープのパス、収集開始時刻、収集終了時刻、収集間隔、収集回数
	 *　インデックス4:実行状態、ステータス、収集ID、説明、スコープのパス、収集開始時刻、収集終了時刻、収集間隔、収集回数
	 * …
	 * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
	 * 
	 * @return 実績収集の一覧（Objectの2次元配列）
	 * @throws NamingException 
	 * @throws FinderException 
	 */
	public ArrayList getCollectionList() {
		m_log.debug("getCollectionList() start : ");
		
		ArrayList<ArrayList> list = new ArrayList<ArrayList>();
		
		try {
			// 収集種別が実績であるEntityBeanを取得
			CollectorHome home = JNDIConnectionManager.getCollectorHome();
			Collection myBeans = home.findByCollectorType(CollectorType.RECORD);
			
			Iterator itr = myBeans.iterator();
			while(itr.hasNext())
			{
				Collector myBean = (Collector)itr.next();
				
				// 収集設定を取得する
				RecordCollectorData data = myBean.getRecordColletorData();
				
				ArrayList<Object> info = new ArrayList<Object>();
				info.add(new Integer(data.getStatus()));
				info.add(new Integer(data.getEndStatus()));
				info.add(data.getCollectorID());
				info.add(data.getLabel());
				info.add(data.getScopePath());
				info.add(data.getStartDate());
				info.add(data.getStopDate());
				info.add(data.getInterval());
				info.add(new Integer(data.getCount()));
				list.add(info);
			}
		} catch (FinderException e) {
			throw new EJBException(e);
		} catch (NamingException e) {
			throw new EJBException(e);
		} catch (RemoteException e) {
			throw new EJBException(e);
		}
		
		m_log.debug("getCollectionList() end   : ");
		return list;
	}
	
	/**
	 * 収集項目コードの一覧を取得します
	 * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @ejb.permission
     *  unchecked="true"
     *  method-intf="Remote"
     *   
	 * @return 収集項目IDをキーとしCollectorItemTreeItemが格納されているHashMap
	 */
	public HashMap<String, CollectorItemTreeItem> getItemCodeMap(){
		return CollectorItemCodeTable.getItemCodeMap();
	}
	
	/**
	 * 収集項目ツリーを返します
	 * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
	 * 
	 * @param facilityId ファシリティID
	 * @param collectorId 収集項目コード
	 * @return 収集項目ツリー（GUIで収集項目をカテゴリ毎にツリー構造で表示するためのデータ構造）
	 */
    public CollectorItemTreeItem[] getCollectorTree(String facilityId, String collectorId) {
    	return CollectorItemCodeTable.getCollectorTree(facilityId, collectorId);
    }
    
    /**
     * 指定のファシリティで収集可能な項目のリストを返します
     * デバイス別の収集項目があり、ノード毎に登録されているデバイス情報が異なるため、
     * 取得可能な収集項目はファシリティ毎に異なる。
     * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
     *
     * @param facilityId ファシリティID
     * @return 指定のファシリティで収集可能な項目のリスト
     */
    public List getAvailableCollectorItemList(String facilityId) {
    	return CollectorItemCodeTable.getAvailableCollectorItemList(facilityId);
    }

    /**
     * アクセス可能かどうかをチェックするためのメソッド
     * 読み込み権限のあるroleで呼び出すとtrueを返す
     * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
     *
     * @return true
     */
    public boolean checkPermissionRead() {
    	return true;
    }
    
    /**
     * アクセス可能かどうかをチェックするためのメソッド
     * 書き込み権限のあるroleで呼び出すとtrueを返す
     * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @ejb.permission
     *  role-name="PerformanceWrite"
     *  method-intf="Remote"
     *
     * @return true
     */
    public boolean checkPerformanceWrite() {
    	return true;
    }
    
    /**
     * アクセス可能かどうかをチェックするためのメソッド
     * 書き込み権限のあるroleで呼び出すとtrueを返す
     * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @ejb.permission
     *  role-name="PerformanceExecute"
     *  method-intf="Remote"
     *
     * @return true
     */
    public boolean checkPermissionExecute() {
    	return true;
    }
    
	/**
	 * 指定のファシリティIDが収集設定で使用されているかチェックします。
	 * <P>
	 * 使用されていたら、UsedFacilityException がスローされる。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="PerformanceRead"
	 *     method-intf="Remote"
	 *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param facilityId ファシリティ
	 * @throws UsedFacilityException
	 * 
	 * @see com.clustercontrol.commons.ejb.session.CheckFacility#isUseFacilityId(java.lang.String)
	 * @see com.clustercontrol.bean.PluginConstant;
	 */
	public void isUseFacilityId(String facilityId) throws UsedFacilityException {
		try {
			Collection ct = null;
			
			// ファシリティIDが使用されている、収集種別がリアルタイムであるEntityBeanを取得する。
			CollectorHome home = JNDIConnectionManager.getCollectorHome();
			ct = home.findByCollectorTypeAndFacilityId(CollectorType.REALTIME, facilityId);
			if(ct != null && ct.size() > 0) {
				throw new UsedFacilityException(PluginConstant.TYPE_PERFORMANCE);
			}
			
			// ファシリティIDが使用されている、収集種別が実績であるEntityBeanを取得する。
			ct = null;
			ct = home.findByCollectorTypeAndFacilityId(CollectorType.RECORD, facilityId);
			if(ct != null && ct.size() > 0) {
				Iterator itr = ct.iterator();
				while(itr.hasNext()){
					Collector collector = (Collector)itr.next();
					// 収集中の場合、UsedFacilityExceptionをスロー
					if(collector.getStatus() == CollectorRunStatus.RUNNING){
						throw new UsedFacilityException(PluginConstant.TYPE_PERFORMANCE);
					}
				}
			}
			
		} catch (FinderException e) {
			m_log.debug("isUseFacilityId() : " + e.getMessage());
		} catch (NamingException e) {
			m_log.debug("isUseFacilityId() : " + e.getMessage());
		} catch (RemoteException e) {
			m_log.debug("isUseFacilityId() : " + e.getMessage());
		}
	}
}