/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.repository.dao.cache;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.repository.dao.DeviceInfoDAOImpl;
import com.clustercontrol.repository.ejb.entity.DeviceInfoBean;
import com.clustercontrol.repository.ejb.entity.DeviceInfoPK;
import com.clustercontrol.repository.util.RepositoryCacheProperties;

/**
 * DeviceInfo(BMP Entity Bean)のDAO実装クラス
 *
 * @version 1.0.0
 * @since 1.0.0
 */
public class CachedDeviceInfoDAOImpl extends DeviceInfoDAOImpl {
	protected static Log m_log = LogFactory.getLog( CachedDeviceInfoDAOImpl.class );

	// 初期化されているか否かを示すフラグ
	private Boolean m_isInit = false;
	
	// ファシリティIDからの検索キャッシュ
	private static HashMap<String, Collection> m_facilityIdCache = null;
	
	// 保持しているキャッシュ用HashMapのリスト
	private static List<HashMap> m_cacheMapList  = new ArrayList<HashMap>();
	
	// 初期化
	@ Override
	public void init() {
		synchronized (m_isInit) {
			if (m_isInit) {
				// 既に初期化されているため何もしない
				return;
			} else {
				// キャッシュ対象となるもののHashMapを生成
				if (RepositoryCacheProperties
						.getProperties(RepositoryCacheProperties.DEVICE_FACILITYID_KEY) > 0) {
					m_facilityIdCache = new HashMap<String, Collection>();
					m_cacheMapList.add(m_facilityIdCache);
					m_log.info("DeviceInfoDAO findByFacilityId : enable");
				}

				m_isInit = true;
			}
		}
	}
	
	/**
	 * 全てのキャッシュをクリアする。
	 */
	private static void clearCacheAll(){
		synchronized (m_cacheMapList) {
			Iterator<HashMap> itr = m_cacheMapList.iterator();
			while(itr.hasNext()){
				HashMap map = itr.next();
				
				synchronized (map) {
					m_log.debug("clearCache() clear : " + map.size());
					map.clear();
				}
			}
		}
	}
	
//	LDAPアクセス必須
//	public void load(DeviceInfoPK pk, DeviceInfoBean ejb) throws EJBException {
//	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.DeviceInfoDAO#store(com.clustercontrol.repository.ejb.DeviceInfoBean)
	 */
	@Override
	public void store(DeviceInfoBean ejb) throws EJBException {
		m_log.debug("store()");

		super.store(ejb);
		
		// キャッシュをクリアする
		clearCacheAll();
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.DeviceInfoDAO#remove(com.clustercontrol.repository.ejb.DeviceInfoPK)
	 */
	@Override
	public void remove(DeviceInfoPK pk) throws RemoveException, EJBException {
		m_log.debug("remove()");
		
		super.remove(pk);

		// キャッシュをクリアする
		clearCacheAll();
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.DeviceInfoDAO#create(com.clustercontrol.repository.ejb.DeviceInfoBean)
	 */
	@Override
	public DeviceInfoPK create(DeviceInfoBean ejb) throws CreateException, EJBException {
		m_log.debug("create()");
		
		DeviceInfoPK pk = super.create(ejb);
		
		// キャッシュをクリアする
		clearCacheAll();
		
		return pk;
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.DeviceInfoDAO#findByFacilityId(java.lang.String)
	 */
	@Override
	public Collection findByFacilityId(String facilityId) throws FinderException {
		// キャッシュを利用しない設定の場合
		if(m_facilityIdCache == null){
			return super.findByFacilityId(facilityId);
		}

		synchronized (m_facilityIdCache) {
			// キャッシュに存在する場合はそれを返す
			if(m_facilityIdCache.containsKey(facilityId)){
				return m_facilityIdCache.get(facilityId);
			}
		}
		
		// キャッシュに存在しない場合
		Collection ret = super.findByFacilityId(facilityId);
		synchronized (m_facilityIdCache) {
			m_log.debug("findByFacilityId(String facilityId) put : " + facilityId
					+ "  entry size : " + ret.size()
					+ "  total entry : " + m_facilityIdCache.size());
			m_facilityIdCache.put(facilityId, ret);
		}
		return ret;
	}
}
