package com.clustercontrol.repository.dao.cache;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.repository.dao.FacilityDAOImpl;
import com.clustercontrol.repository.ejb.entity.FacilityBean;
import com.clustercontrol.repository.ejb.entity.FacilityPK;
import com.clustercontrol.repository.util.RepositoryCacheProperties;

public class CachedFacilityDAOImpl extends FacilityDAOImpl {
	private static Log m_log = LogFactory.getLog( CachedFacilityDAOImpl.class );
	
	// 初期化されているか否かを示すフラグ
	private Boolean m_isInit = false;
	
	// IPアドレスキャッシュ
	private static HashMap<String, Collection> m_ipAddressCache = null;

	// ノード名キャッシュ
	private static HashMap<String, Collection> m_nodeNameCache = null;
	
	// 保持しているキャッシュ用HashMapのリスト
	private static List<HashMap> m_cacheMapList  = new ArrayList<HashMap>();
	
	// 初期化
	@Override
	public void init() {
		synchronized (m_isInit) {
			if (m_isInit) {
				// 既に初期化されているため何もしない
				return;
			} else {
				// キャッシュ対象となるもののHashMapを生成
				if (RepositoryCacheProperties
						.getProperties(RepositoryCacheProperties.FACILITY_IP_ADDRESS_KEY) > 0) {
					m_ipAddressCache = new HashMap<String, Collection>();
					m_cacheMapList.add(m_ipAddressCache);
					m_log.info("FacilityDAO ipAddressCache : enable");
				}

				if (RepositoryCacheProperties
						.getProperties(RepositoryCacheProperties.FACILITY_NODE_NAME_KEY) > 0) {
					m_nodeNameCache = new HashMap<String, Collection>();
					m_cacheMapList.add(m_nodeNameCache);
					m_log.info("FacilityDAO nodeNameCache : enable");
				}

				m_isInit = true;
			}
		}
	}
	
	/**
	 * 全てのキャッシュをクリアする。
	 */
	private static void clearCacheAll(){
		synchronized (m_cacheMapList) {
			Iterator<HashMap> itr = m_cacheMapList.iterator();
			while(itr.hasNext()){
				HashMap map = itr.next();
				
				synchronized (map) {
					m_log.debug("clearCache() clear : " + map.size());
					map.clear();
				}
			}
		}
	}

//	LDAPアクセス必須
//	public void load(FacilityPK pk, FacilityBean ejb) throws EJBException {
//	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityDAO#store(com.clustercontrol.repository.ejb.FacilityBean)
	 */
	@Override
	public void store(FacilityBean ejb) throws EJBException {
		m_log.debug("store()");
		
		super.store(ejb);
		
		// キャッシュをクリアする
		clearCacheAll();
	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityDAO#remove(com.clustercontrol.repository.ejb.FacilityPK)
	 */
	@Override
	public void remove(FacilityPK pk) throws RemoveException, EJBException {
		m_log.debug("remove()");
		
		super.remove(pk);
		
		// キャッシュをクリアする
		clearCacheAll();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.clustercontrol.repository.dao.FacilityDAO#create(com.clustercontrol.repository.ejb.FacilityBean)
	 */
	@Override
	public FacilityPK create(FacilityBean ejb) throws CreateException, EJBException {
		m_log.debug("create()");
		
		FacilityPK pk = super.create(ejb);
		
		// キャッシュをクリアする
		clearCacheAll();
		
		return pk;		
	}

//	呼び出しを確認できなかったため、キャッシュ対象からはずす
//	public Collection findAll() throws FinderException {
//	}
	
//	呼び出しを確認できなかったため、キャッシュ対象からはずす
//	public Collection findAllNode() throws FinderException {
//	}
	
//	EntityBeanの機能でキャッシュされるためキャッシュ対象からはずす
//	public FacilityPK findByPrimaryKey(FacilityPK pk) throws FinderException {
//	}
	
//	LDAPへのアクセスが発生しないメソッドであるため対象からはずす
//	public FacilityPK findByFacilityId(String facilityId) throws FinderException {
//	}
	
//	呼び出しを確認できなかったため、キャッシュ対象からはずす
//	public Collection findNode(
//			String facilityId, 
//			String facilityName, 
//			String description, 
//			String ipAddress, 
//			String osName, 
//			String osRelease, 
//			String managerName, 
//			String managerContact) throws FinderException {
//	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityDAO#findByIpAddress(java.lang.String)
	 */
	@Override
	public Collection findByIpAddress(String ipAddress) throws FinderException {
		// キャッシュを利用しない設定の場合
		if(m_ipAddressCache == null){
			return super.findByIpAddress(ipAddress);
		}
		
		synchronized (m_ipAddressCache) {
			// キャッシュに存在する場合はそれを返す
			if(m_ipAddressCache.containsKey(ipAddress)){
				return m_ipAddressCache.get(ipAddress);
			}
		}
		
		// キャッシュに存在しない場合
		Collection ret = super.findByIpAddress(ipAddress);
		synchronized (m_ipAddressCache) {
			if(m_log.isDebugEnabled()){
				m_log.debug("findByIpAddress(String ipAddress) put : "
						+ ipAddress + "  size : " + m_ipAddressCache.size());
			}
			m_ipAddressCache.put(ipAddress, ret);
		}
		return ret;
	}
	
    /* (non-Javadoc)
     * @see com.clustercontrol.repository.dao.FacilityDAO#findByNodeName(java.lang.String)
     */
	@Override
	public Collection findByNodeName(String nodeName) throws FinderException {
		// キャッシュを利用しない設定の場合
		if(m_nodeNameCache == null){
			return super.findByNodeName(nodeName);
		}
		
		synchronized (m_nodeNameCache) {
			// キャッシュに存在する場合はそれを返す
			if(m_nodeNameCache.containsKey(nodeName)){
				return m_nodeNameCache.get(nodeName);
			}
		}
		
		// キャッシュに存在しない場合
		Collection ret = super.findByNodeName(nodeName);
		synchronized (m_nodeNameCache) {
			if(m_log.isDebugEnabled()){
				m_log.debug("findByNodeName(String nodeName) put : "
						+ nodeName + "  size : " + m_nodeNameCache.size());
			}
			m_nodeNameCache.put(nodeName, ret);
		}
		return ret;
	}
}
