/*

 Copyright (C) 2008 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 */

package com.clustercontrol.snmppoller;

import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.opennms.protocols.snmp.SnmpEndOfMibView;
import org.opennms.protocols.snmp.SnmpHandler;
import org.opennms.protocols.snmp.SnmpObjectId;
import org.opennms.protocols.snmp.SnmpParameters;
import org.opennms.protocols.snmp.SnmpPduPacket;
import org.opennms.protocols.snmp.SnmpPduRequest;
import org.opennms.protocols.snmp.SnmpPeer;
import org.opennms.protocols.snmp.SnmpSession;
import org.opennms.protocols.snmp.SnmpSyntax;
import org.opennms.protocols.snmp.SnmpTimeTicks;
import org.opennms.protocols.snmp.SnmpUInt32;
import org.opennms.protocols.snmp.SnmpVarBind;

import com.clustercontrol.sharedtable.bean.DataTable;
import com.clustercontrol.sharedtable.bean.ValueObject;

/**
 * SNMPのポーリングを実行するクラス
 * 
 * @version 2.4.2
 * @since 2.0.0
 */
public class SnmpPoller implements SnmpHandler {
	protected static Log m_log = LogFactory.getLog(SnmpPoller.class);

	// IPアドレス（ポーリングの度に更新）
	private String m_ipAddress;

	// 収集のターゲットとなるOID（全てのポーリングが終了するまで値を保持）
	private String[] m_oidText;

	// 収集を開始するOID（ポーリングの度に更新）
	private ArrayList<SnmpObjectId> m_startOids;

	// 収集を終了するOID（ポーリングの度に更新）
	private ArrayList<SnmpObjectId> m_stopOids;

	// 取得したデータを格納するテーブル（ポーリングの度に更新）
	private DataTable m_dataTable;

	// メッセージ（ポーリングの度に更新）
	private String m_message = null;

	// 待機フラグ
	private boolean m_waitFlg;

	// 収集したOIDの最後のインデックスをチェックするか否かのフラグ（ポーリングの度に更新）
	private boolean m_indexCheckFlg = false;

	// m_indexCheckFlg = true の場合に、取得したOIDのインデックスが異なることで
	// リトライした回数を保持（ポーリングの度に更新）
	private int m_indexCheckDiffIndexRetryCount = 0;

	// m_indexCheckFlg = true の場合に、取得できたVarBindの数が異なることで
	// リトライした回数を保持（ポーリングの度に更新）
	private int m_indexCheckVarBindSizeRetryCount = 0;
	
	// m_indexCheckFlg = true の場合にリトライした回数の合計を保持（ポーリングの度に更新）
	private int m_indexCheckRetryCountTotal = 0;
	
	// m_indexCheckFlg = true の場合にリトライする回数の最大値
	private final int MAX_RETRY_COUNT = 10;
	
	// ポーリング対象のスコープを抜ける際のOIDが本当にそのスコープの最後のOIDなのかをチェックする
	// （Net-SNMPのHOST-RESOURCES-MIBでプロセス情報を取得した際に、
	//  プロセスのリストを最後まで取得できない場合があることへの対処）
	// そのために最後のOIDでリトライ処理をするその回数を保持
	// （ポーリングの度に更新）
	private int m_indexCheckLastIndexRetryCount = 0;

	// 最期にポーリングを行なった際のOIDのリスト m_indexCheckFlg = true の場合に使用する
	private ArrayList<String> m_lastPollingOids;

	// 全体的なポーリングをリトライするか否かのフラグ
	private boolean m_retryFlg = false;

	// 全体的なポーリングのリトライ回数
	private final int m_retryCount = 1;

	// エラーコード（ポーリングの度に更新）
	private int m_errorCode = 0;

	// エラーコード
	private final int Normal = 0; // 正常に動作いている場合
	private final int TimeOutError = 1; // 問い合わせがタイムアウトした場合
	private final int InternalError = 2; // タイムアウト以外のエラー

	// クラス変数の初期化
	private void init(){
		m_indexCheckDiffIndexRetryCount = 0;
		m_indexCheckVarBindSizeRetryCount = 0;
		m_indexCheckRetryCountTotal = 0;
		m_indexCheckLastIndexRetryCount = 0;
		m_errorCode = Normal;
		m_message = null;
	}

	/**
	 * メインルーチン　
	 * IPアドレスと　DataTableを受け取り、
	 * ポーリングした結果をDataTableに代入する
	 *  
	 * @param ipAddress IPアドレス
	 * @param port ポート番号
	 * @param version バージョン（0:SNMP V1 protocol, 1:SNMP V2 protocol）
	 * @param community コミュニティ
	 * @param retries １回のポーリングでのリトライ回数
	 * @param timeout ポーリングのタイムアウト
	 * @param oidList 対象OIDのリスト
	 * @param indexCheckFlg ポーリング結果のインデックスが揃っているかのチェック
	 * @param table 取得したMIB値を格納するテーブル
	 */
	public void polling(
			InetAddress ipAddress, 
			int port, 
			int version,
			String community, 
			int retries, 
			int timeout, 
			List<String> oidList,
			boolean indexCheckFlg,
			DataTable table) {
		// クラス変数を初期化
		init();

		m_dataTable = table;
		m_ipAddress = ipAddress.toString();	
		m_indexCheckFlg = indexCheckFlg;

		SnmpPeer peer = new SnmpPeer(ipAddress);
		if (port != -1)
			peer.setPort(port);

		if (timeout != -1)
			peer.setTimeout(timeout);

		if (retries != -1)
			peer.setRetries(retries);

		// peer　を初期化
		SnmpParameters parms = peer.getParameters();
		parms.setVersion(version);
		if (community != null)
			parms.setReadCommunity(community);

		// デバッグ出力
		if (m_log.isDebugEnabled()) {
			m_log.debug("polling() start :" + ipAddress.toString());
			m_log.debug("Port            : " + peer.getPort());
			m_log.debug("Version         : " + peer.getParameters().getVersion());
			m_log.debug("Community       : " + peer.getParameters().getReadCommunity());
			m_log.debug("Retries         : " + peer.getRetries());
			m_log.debug("Timeout         : " + peer.getTimeout());
			m_log.debug("IndexCheckFlg   : " + indexCheckFlg);
		}
		
		//　sessionを生成
		SnmpSession session = null;
		try {
			session = new SnmpSession(peer);
		} catch (SocketException e) {
			m_log.error("polling() warning  :" + ipAddress.toString()
					+ " SocketException creating the SNMP session");
			m_errorCode = InternalError;
			return;
		}
		session.setDefaultHandler(this);

		try {
			m_oidText = new String[oidList.size()];
			m_startOids = new ArrayList<SnmpObjectId>(oidList.size());
			m_stopOids = new ArrayList<SnmpObjectId>(oidList.size());
			// 最期にポーリングを行なった際のOIDのリスト(m_indexCheckFlg = true の場合に使用)
			m_lastPollingOids = new ArrayList<String>(oidList.size());
			int i = 0;

			SnmpPduRequest pdu = new SnmpPduRequest(SnmpPduPacket.GETNEXT);
			pdu.setRequestId(SnmpPduPacket.nextSequence());

			Iterator<String> itr = oidList.iterator();
			while (itr.hasNext()) {
				m_oidText[i] = (String) itr.next();

				// OIDの最後が0で終わる場合は、.0を削除する
				// GETNEXTで取得するため
				if(m_oidText[i].endsWith(".0")){
					m_oidText[i] = m_oidText[i].substring(0, m_oidText[i].lastIndexOf(".0"));
				}

				// 開始OIDを設定
				SnmpObjectId startOid = new SnmpObjectId(m_oidText[i]);

				// 終了するOIDを設定
				SnmpObjectId stopOid = new SnmpObjectId(m_oidText[i]);
				int[] ids = stopOid.getIdentifiers();
				++ids[ids.length - 1];
				stopOid.setIdentifiers(ids);

				SnmpObjectId oId = new SnmpObjectId(m_oidText[i]);
				pdu.addVarBind(new SnmpVarBind(oId));
				m_startOids.add(startOid);
				m_stopOids.add(stopOid);
				m_lastPollingOids.add(m_oidText[i]);

				i++;
			}

			for (int count = 0; count < m_retryCount+1; count++) {
				synchronized (session) {
					m_waitFlg = true;
					session.send(pdu);
					m_log.debug("polling() : " + m_ipAddress + " send SnmpPduRequest");

					if (m_waitFlg) {
						session.wait();
					}
				}

				if(m_retryFlg == false){
					//　リトライしない
					break;
				} else {
					// 一旦取得した全てのデータを削除してリトライ
					m_dataTable.clear();
				}
			}
		} catch (InterruptedException e) {
			m_log.error("polling() warning :" + ipAddress.toString()
					+ " polling failed at InterruptedException");
			m_errorCode = InternalError;
			return;
		} finally {
			try {

				session.close();

			} catch (Exception e) {
				/**
				 * FIXME
				 * Joe-SNMPの不具合の回避コード
				 * Joe-SNMPで不具合が修正されたら削除すること
				 */
				m_log.warn("polling():" + m_ipAddress
						+ " Session close failed");
			}
		}

		// 例外処理(未実装)
		if (this.m_errorCode == TimeOutError) {

		} else if (this.m_errorCode == InternalError) {

		}

		// リトライ回数が多い場合は出力する
		if(m_indexCheckRetryCountTotal >= 100){
			m_log.warn("polling():" + m_ipAddress + 
					" too many retries. count : " + m_indexCheckRetryCountTotal);
		}
		
		// デバッグ出力
		if (m_log.isDebugEnabled()) {
			m_log.debug("polling() " + m_ipAddress + " retry total count : " + m_indexCheckRetryCountTotal);
			m_log.debug("polling() end :" + ipAddress.toString());
		}
	}

	/**
	 * SNMPポーリングの結果を受け取った際にコールバックされるメソッド
	 */
	public void snmpReceivedPdu(SnmpSession session, int cmd, SnmpPduPacket pdu) {
		//		m_log.debug("snmpReceivedPdu() start");

		try {
			long time = System.currentTimeMillis(); // 取得時刻

			SnmpPduRequest req = null;
			if (pdu instanceof SnmpPduRequest) {
				req = (SnmpPduRequest) pdu;
			} else {
				m_log.error("polling() error :" + session.toString()
						+ " Received non-request pdu");
				synchronized (session) {
					m_errorCode = InternalError;
					session.notify();
				}
				return;
			}

			if (pdu.getCommand() != SnmpPduPacket.RESPONSE) {
				m_log.error("polling() error :" + session.toString()
						+ "  Received non-response command "
						+ pdu.getCommand());
				synchronized (session) {
					m_errorCode = InternalError;
					session.notify();
				}
				return;
			}

			if (req.getErrorStatus() != 0) {
				m_log.error("polling() error :" + session.toString()
						+ "  Error Status " + req.getErrorStatus());
				synchronized (session) {
					m_errorCode = InternalError;
					session.notify();
				}
				return;
			}

			// 次の pduを作成
			SnmpPduRequest nxt = new SnmpPduRequest(SnmpPduPacket.GETNEXT);
			nxt.setRequestId(SnmpPduPacket.nextSequence());

			// インデックスチェックフラグが設定されている場合にOIDの最後のインデックスが
			// 一致しているか否かをチェックするのに使用
			String lastIndexStr = null;

			// インデックスの最大値を保持する(m_indexCheckFlgがtrueの場合のみ利用)
			// 取得できたOIDのインデックスに差異がない場合は-lのまま
			int maxIndex = -1;

			// 次回のポーリングでpduにセットするOIDの開始と終了のOIDを保持する
			ArrayList<SnmpObjectId> nextStartOids = new ArrayList<SnmpObjectId>(m_startOids.size());
			ArrayList<SnmpObjectId> nextStopOids = new ArrayList<SnmpObjectId>(m_stopOids.size());
			// 最期にポーリングを行なった際に取得したOIDのリスト(m_indexCheckFlg = true の場合に使用)
			ArrayList<String> lastPollingOids = new ArrayList<String>(m_startOids.size());
			// インデックスチェックが必要な場合は、この時点ではテーブルに反映することはできないため、
			// 一時的にArrayListに保持する(m_indexCheckFlg = true の場合に使用)
			ArrayList<ValueObject> entryBuffer = new ArrayList<ValueObject>(m_startOids.size());
			
			// pduの戻りとして帰ってきた SnmpVarBindを順次処理する
			int length = pdu.getLength();
			for (int i = 0; i < length; i++) {
				SnmpVarBind var = pdu.getVarBindAt(i);

				// ポーリングの終了判定
				// 条件：
				// 戻りの型が EndOfMibView の場合
				// 停止条件のOIDにマッチした場合は終了
				if (var.getValue().typeId() == SnmpEndOfMibView.ASNTYPE
						|| ((m_stopOids.get(i) != null && m_stopOids.get(i).compare(var.getName()) < 0))) {

					// 該当OIDの収集を終了します
					if (m_log.isDebugEnabled()) {
						m_log.debug(m_ipAddress + " stop polling. " + i + ", "
								+ m_startOids.get(i) + ", " + m_stopOids.get(i)
								+ " - " + var.getName());
					}
					continue;
				}

				// OIDを設定
				String oidString = var.getName().toString();

				// デバッグ出力
				if (m_log.isDebugEnabled()) {
					m_log.debug("get " + m_ipAddress + " " + oidString + " "
							+ time + " " + var.getValue().getClass().getName()
							+ " " + var.getValue().toString());
				}

				if (!m_startOids.get(i).isRootOf(var.getName())) {
					// 帰ってきたMIB値のOIDが開始OIDのツリーの配下でない場合は、
					// 該当OIDの収集を終了します
					if (m_log.isDebugEnabled()) {
						m_log.debug(m_ipAddress + " stop polling.. " + i + ", "
								+ m_startOids.get(i) + ", " + m_stopOids.get(i)
								+ " - " + var.getName());
					}
					continue;
				} else {
					// インデックスチェックフラグが設定されている場合は、OIDの最後のインデックスが
					// 一致しているか否かをチェックする
					if (m_indexCheckFlg == true) {
						if (lastIndexStr == null) { // ループの1回目（i=0の場合）
							lastIndexStr = oidString.substring(oidString
									.lastIndexOf(".")+1);
						} else { // ループの2回目以降は値が設定されている
							String indexStr = oidString.substring(oidString
									.lastIndexOf(".")+1);

							// OIDの最後のインデックスが一致しない場合
							if (!lastIndexStr.equals(indexStr)) {
								m_log.debug("Index not match. " + m_ipAddress
										+ ", " + lastIndexStr + ", "
										+ oidString);

								// 最大のインデックスを設定
								int lastIndex = Integer.parseInt(lastIndexStr);
								if(lastIndex > maxIndex){
									maxIndex = lastIndex;
								}
							}
							lastIndexStr = indexStr;
						}
					}

					// デバッグ
					if (m_log.isDebugEnabled()) {
						m_log.debug("set " + m_ipAddress + ","
								+ oidString + "," + time + ","
								+ var.getValue().toString() + ", "
								+ var.getValue());
					}
					
					if (var.getValue() instanceof SnmpTimeTicks) {
						long value = ((SnmpUInt32) var.getValue()).getValue();
						
						if(m_indexCheckFlg == true){
							entryBuffer.add(new ValueObject(oidString, time, String.valueOf(value)));
							
							m_log.debug("snmpReceivedPdu() dataTable put : " +
									"entryKey : " + oidString + 
									", time : " + time +
									", longValue : " + String.valueOf(value));
						} else {
							m_dataTable.putValue(oidString, time, String.valueOf(value));
							
							m_log.debug("snmpReceivedPdu() dataTable put : " +
									"entryKey : " + oidString + 
									", time : " + time +
									", longValue : " + String.valueOf(value));
						}
					} else {
						if(m_indexCheckFlg == true){
							entryBuffer.add(new ValueObject(oidString, time, var.getValue().toString()));
							
							m_log.debug("snmpReceivedPdu() dataTable put : " +
									"entryKey : " + oidString + 
									", time : " + time +
									", longValue : " + var.getValue().toString());
						} else {
							m_dataTable.putValue(oidString, time, var.getValue().toString());
							
							m_log.debug("snmpReceivedPdu() dataTable put : " +
									"entryKey : " + oidString + 
									", time : " + time +
									", longValue : " + var.getValue().toString());
						}
					}

					// 次のポーリングで収集するものだけを追加
					nextStartOids.add(m_startOids.get(i));
					nextStopOids.add(m_stopOids.get(i));
					nxt.addVarBind(new SnmpVarBind(var.getName()));
					// 最期に実行したポーリングで取得したOIDのリストを保持する
					// m_indexCheckFlg が true の場合のリトライ処理のため
					lastPollingOids.add(var.getName().toString());
				}
			}

			// インデックスチェックを行なっている場合
			if(m_indexCheckFlg == true){
				// 正常に値取得できている場合は、値をテーブルに設定する
				if (maxIndex == -1
						&& ((lastPollingOids.size() == m_lastPollingOids.size())
								|| (lastPollingOids.size() == 0))) {
					// 一時的にArrayListで保持されていた取得値をセットする
					Iterator<ValueObject> itr = entryBuffer.iterator();
					while(itr.hasNext()){
						ValueObject entry = itr.next();
						
						// デバッグ
						if (m_log.isDebugEnabled()) {
							m_log.debug("set (buffer to table) " + m_ipAddress + ","
									+ entry.getKey() + "," + time + ","
									+ entry.getValue().toString() + ", "
									+ entry.getValue());
						}
						
						m_dataTable.putValue(entry);
					}
					
					// リトライ回数をリセット
					m_indexCheckDiffIndexRetryCount = 0;
					m_indexCheckVarBindSizeRetryCount = 0;
				} else {
					m_log.debug("detect error. " + m_ipAddress);
				}

				// 対応しているエラーパターン（VarBindの数をn個とする）
				// 1) n個全て取得できているが、対象のMIBツリーの最後ではない
				// 2) n個全て取得できているが、indexが揃っていない（少なくともどれかのindexが異なる）
				// 3) n個全ては取得できていない。かつ、indexが揃っていない（少なくともどれかのindexが異なる）
				// 4) n個全ては取得できていないが、indexは全て同じ
				if (lastPollingOids.size() == 0) {  // エラーパターン1への対応
					// 次に実行するOIDが１つもない場合
					// ポーリング対象のスコープを抜ける際のOIDが本当にそのスコープの最後のOIDなのかをチェックする
					// （Net-SNMPのHOST-RESOURCES-MIBでプロセス情報を取得した際に、
					//  プロセスのリストを最後まで取得できない場合があることへの対処）
					
					// リトライ回数をリセット
					m_indexCheckDiffIndexRetryCount = 0;
					m_indexCheckVarBindSizeRetryCount = 0;

					if (m_indexCheckLastIndexRetryCount >= 1) {
						// 2回連続で同じOID（同じVarBindのセット）でポーリングの結果、全てのポーリングの戻りの
						// PDUに入っている「次の問い合わせ対象OID」が取得対象のMIBツリーからはずれているため
						// 正常に取得できたと判断
					} else {
						// ポーリング対処のOIDのリストの全てが同時に終了した場合、
						// 対象スコープを抜ける際のOIDが本当にそのスコープの最後のOIDなのかをチェックする。
						// （Net-SNMPのHOST-RESOURCES-MIBでプロセス情報を取得した際に、
						// プロセスのリストを最後まで取得できない場合があることへの対処）
						if (m_log.isDebugEnabled()) {
							m_log.debug("check retry. count="
									+ m_indexCheckLastIndexRetryCount + " "
									+ m_ipAddress);
						}
						
						nextStartOids.clear();
						nextStopOids.clear();
						lastPollingOids.clear();
						nxt = new SnmpPduRequest(SnmpPduPacket.GETNEXT);
						nxt.setRequestId(SnmpPduPacket.nextSequence());

						for (int i = 0; i < m_lastPollingOids.size(); i++) {
							if (m_log.isDebugEnabled()) {
								m_log.debug("retry last polling set VarBind "
										+ m_ipAddress + ","
										+ m_lastPollingOids.get(i));
							}
							nextStartOids.add(m_startOids.get(i));
							nextStopOids.add(m_stopOids.get(i));
							nxt.addVarBind(new SnmpVarBind(m_lastPollingOids
									.get(i)));
							lastPollingOids = m_lastPollingOids;
						}

						// 最後のOIDでリトライ処理を実行した回数をカウントアップ
						m_indexCheckLastIndexRetryCount++;
						m_indexCheckRetryCountTotal++;
					}
				} else {  // エラーパターン1以外の対応（正常動作の場合も含む）
					// 次に実行するOIDが１つ以上ある場合

					// ログ出力
					if (m_log.isDebugEnabled()) {
						if (m_indexCheckLastIndexRetryCount != 0) {
							m_log.debug("detect error." + m_ipAddress);
						}
					}
					
					// 最後のOIDでリトライ処理を実行した回数をリセット
					m_indexCheckLastIndexRetryCount = 0;

					// インデックスの異なるものが戻ってきていた場合
					// 再度ポーリングを実行する
					if (maxIndex >= 0){  // エラーパターン2or3の対応
						// ログ出力
						if (m_log.isDebugEnabled()){
							m_log.debug("recovery(index not match) " + m_ipAddress
									+ "  max index:" + maxIndex
									+ "  next VarBind size:" + lastPollingOids.size());
						}

//						// 格納されたMIB値を削除します。
//						for(int j=0; j < pdu.getLength(); j++){
//							String removeOid = pdu.getVarBindAt(j).getName().toString();
//							m_log.debug("remove " + m_ipAddress + ", " + removeOid);
//							m_dataTable.removeValue(removeOid);
//						}
						
						nextStartOids.clear();
						nextStopOids.clear();
						lastPollingOids.clear();
						nxt = new SnmpPduRequest(SnmpPduPacket.GETNEXT);
						nxt.setRequestId(SnmpPduPacket.nextSequence());

						if(m_indexCheckDiffIndexRetryCount < MAX_RETRY_COUNT){
							for (int i = 0; i < m_lastPollingOids.size(); i++) {
								String lastOid = m_lastPollingOids.get(i);
								// 最大のインデックスのものがリトライによって収集されるように
								// maxIndex-lをインデックスに設定する
								String nextOid = lastOid.substring(0, lastOid.lastIndexOf(".") + 1)
								+ (maxIndex - 1);

								nextStartOids.add(m_startOids.get(i));
								nextStopOids.add(m_stopOids.get(i));
								nxt.addVarBind(new SnmpVarBind(nextOid));
								lastPollingOids.add(nextOid);
							
								// ログ出力
								if (m_log.isDebugEnabled()) {
									m_log.debug("set VarBind " + m_ipAddress + "," + nextOid);
								}
							}
						} else { // リトライ回数の上限を超えた場合
							m_log.error("too many retries(index not match). " + m_ipAddress 
									+ " count:" + m_indexCheckDiffIndexRetryCount);
							// 収集したデータを全てクリアする
							m_dataTable.clear();
							
							// nxtが空となるためポーリングは継続されない
						}
						
						// リトライ回数をカウントアップ
						m_indexCheckDiffIndexRetryCount++;
						m_indexCheckVarBindSizeRetryCount = 0;
						m_indexCheckRetryCountTotal++;
					} else if(lastPollingOids.size() != m_lastPollingOids.size()) {  // エラーパターン4の対応
						// ログ出力
						if (m_log.isDebugEnabled()){
							m_log.debug("recovery(next count:" + lastPollingOids.size() + ") "
									+ m_ipAddress);
						}
						
//						// 格納されたMIB値を削除します。
//						for(int j=0; j < pdu.getLength(); j++){
//							String removeOid = pdu.getVarBindAt(j).getName().toString();
//							m_log.debug("remove " + m_ipAddress + ", " + removeOid);
//							m_dataTable.removeValue(removeOid);
//						}

						nextStartOids.clear();
						nextStopOids.clear();
						lastPollingOids.clear();
						nxt = new SnmpPduRequest(SnmpPduPacket.GETNEXT);
						nxt.setRequestId(SnmpPduPacket.nextSequence());
						
						// 最大リトライ回数（デフォルト3回）までは、同じOIDでリトライ処理
						// 永遠に同じOIDでリトライ処理が実行されないように、最大リトライ回数を超えたものは、
						// IF文分岐の下のロジックを通る
						if(m_indexCheckVarBindSizeRetryCount < MAX_RETRY_COUNT){
							for (int i = 0; i < m_lastPollingOids.size(); i++) {
								String nextOid = m_lastPollingOids.get(i);

								nextStartOids.add(m_startOids.get(i));
								nextStopOids.add(m_stopOids.get(i));
								nxt.addVarBind(new SnmpVarBind(nextOid));
								lastPollingOids.add(nextOid);
							
								// ログ出力
								if (m_log.isDebugEnabled()) {
									m_log.debug("set VarBind " + m_ipAddress + "," + nextOid);
								}
							}
						} else { // リトライ回数の上限を超えた場合
							m_log.error("too many retries(VarBind size not match). " + m_ipAddress 
									+ " count:" + m_indexCheckVarBindSizeRetryCount);
							// 収集したデータを全てクリアする
							m_dataTable.clear();
							
							// nxtが空となるためポーリングは継続されない
						}
							
						// リトライ回数をカウントアップ
						m_indexCheckDiffIndexRetryCount = 0;
						m_indexCheckVarBindSizeRetryCount++;
						m_indexCheckRetryCountTotal++;
					}
				}
			}

			if (nxt.getLength() != 0) {
				m_startOids = nextStartOids;
				m_stopOids = nextStopOids;
				m_lastPollingOids = lastPollingOids;
				session.send(nxt, this);
			} else {
				// 収集を終了します
				synchronized (session) {
					session.notify();
				}
				
				return;
			}
		} catch (Exception e) { // 何か例外が生じた場合は収集を停止する
			m_log.error("InternalError ", e);

			// 収集したデータを全てクリアする
			m_dataTable.clear();
			
			// 収集を終了します
			synchronized (session) {
				m_errorCode = InternalError;
				m_message = e.getMessage();
				session.notify();
			}
		}
	}

	/** 
	 * 内部エラー発生時にコールバックされるメソッド
	 * 
	 * @see org.opennms.protocols.snmp.SnmpHandler#snmpInternalError(org.opennms.protocols.snmp.SnmpSession, int, org.opennms.protocols.snmp.SnmpSyntax)
	 */
	public void snmpInternalError(SnmpSession session, int err, SnmpSyntax pdu) {
		String message = "InternalError. The error code is " + err + ". IP:"
		+ m_ipAddress + " OID:" + m_oidText[0];
		m_log.error("snmpInternalError():" + session.toString() + " " + message);

		synchronized (session) {
			m_errorCode = InternalError;
			m_message = message;

			// 既に収集されたものも含めて全て削除する
			for (int j = 0; j < m_oidText.length; j++) {
				// 収集中のOID配下のOIDの値をテーブルから削除
				removeTableValue(m_oidText[j]);
			}

			m_waitFlg = false;
			session.notify();
		}
	}

	/**
	 * ポーリングでタイムアウト発生時にコールバックされるメソッド
	 * @see org.opennms.protocols.snmp.SnmpHandler#snmpTimeoutError(org.opennms.protocols.snmp.SnmpSession, org.opennms.protocols.snmp.SnmpSyntax)
	 */
	public void snmpTimeoutError(SnmpSession session, SnmpSyntax pdu) {
//		m_log.warn("snmpTimeoutError():"
//		+ session.getPeer().getPeer().toString() + " "
//		+ ((SnmpPduRequest) pdu).toVarBindArray()[0].getName()
//		+ " polling failed at TimeoutError");

		SnmpVarBind[] bind = ((SnmpPduRequest) pdu).toVarBindArray();
		String ipAddress = session.getPeer().getPeer().toString();
		String message = "Polling failed at TimeoutError." + " IP:" + ipAddress
		+ " OID:" + bind[0].getName();

		// より詳細なメッセージを作成
		String addMessage = "";
		if(m_log.isDebugEnabled()){
			for (int i = 1; i < bind.length; i++) {
				addMessage = addMessage + "\n\t" + ipAddress + " " + bind[i].getName();
			}
			message = message + addMessage;
		}

		m_log.warn(message);

		synchronized (session) {
			m_errorCode = TimeOutError;
			m_message = message;

			// 既に収集されたものも含めて全て削除する
			for (int j = 0; j < m_oidText.length; j++) {
				// 収集中のOID配下のOIDの値をテーブルから削除
				removeTableValue(m_oidText[j]);
			}

			session.notify();
		}
	}

	/**
	 * 収集指定のOID配下のMIB値を全て削除する
	 * 
	 * 例)
	 * .1.3.6.1.2.1.2.2.1   を対象に収集している場合で
	 * .1.3.6.1.2.1.2.2.1.3 を収集の際にこのメソッドを呼び出した場合
	 * .1.3.6.1.2.1.2.2.1   配下のMIB値は全て削除されるため
	 * 既にテーブルに格納済みの
	 * .1.3.6.1.2.1.2.2.1.1 のMIB値および .1.3.6.1.2.1.2.2.1.2
	 * のMIB値を削除する
	 * 
	 * @param oidText OID
	 */
	private void removeTableValue(String oidText) {
		// テーブルに格納されているOIDのセットを取得
		Set<String> oidSet = m_dataTable.keySet();

		String[] fullOids = (String[]) oidSet
		.toArray(new String[oidSet.size()]);

		for (int i = 0; i < fullOids.length; i++) {

			String fullOid = fullOids[i];

			// もし先頭の部分が一致するならそのツリーの子孫と判定
			if (fullOid.startsWith(oidText)) {
				// テーブルに格納されている値を削除
				m_dataTable.removeValue(fullOid);
			}
		}
	}
	
	/**
	 * 単体試験用
	 * 
	 * @param args
	 */
	public static void main(String[] args) {
		// パラメータを設定
		InetAddress ipAddress = null;
		try {
			ipAddress = InetAddress.getByName(args[0]);
		} catch (UnknownHostException e) {
			m_log.error(e, e);
			return;
		}
		SnmpPoller poller = new SnmpPoller();
		int port = Integer.parseInt(args[1]);
		int version = Integer.parseInt(args[2]);
		String comunity = args[3];
		int retries = Integer.parseInt(args[4]);
		int timeout = Integer.parseInt(args[5]);
		boolean indexCheckFlg = Boolean.parseBoolean(args[6]);
		ArrayList<String> oids = new ArrayList<String>();

		for(int i=7; i<args.length; i++){
			oids.add(args[i]);
		}

		// ポーリングを行い値を収集
		DataTable dataTable = new DataTable();

		poller.polling(
				ipAddress, 
				port,
				version,
				comunity,
				retries,
				timeout,
				oids, 
				indexCheckFlg,
				dataTable);

		Set<String> keyset = dataTable.keySet();
		// OID順に並び替える
		TreeSet<String> keys = new TreeSet<String>(keyset);
		Iterator<String> keyItr = keys.iterator();

		// テーブルの中の値を取得時刻順に並び替える
		TreeSet<ValueObject> ts = new TreeSet<ValueObject>();
		while(keyItr.hasNext()){
			ts.add(dataTable.getValue(keyItr.next()));
		}

		// 取得値の一覧を表示
		Iterator<ValueObject> itr = ts.iterator();
		while(itr.hasNext()){
			m_log.info(itr.next());
		}
	}
}
