/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performance.bean;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import com.clustercontrol.performance.bean.CollectedDataSet;

/**
 * 複数の収集済み性能値のリスト（CollectedDataInfo型のリスト）をまとめて保持するクラス
 * ファシリティIDと収集項目の２つをキーとして、収集済み性能値のリストを取得することが出来ます。
 * @version 1.0
 * @since 1.0
 */
public class CollectedDataSet implements Serializable {
	private static final long serialVersionUID = -2016025252868760081L;
	private HashMap<String, HashMap<CollectorItemInfo, List<CollectedDataInfo>>> dataMap;
	
	/**
	 * コンストラクター
	 *
	 */
	public CollectedDataSet(){
		dataMap = new HashMap<String, HashMap<CollectorItemInfo, List<CollectedDataInfo>>>();
	}
	
	/**
	 * 指定のファシリティIDと収集項目IDをキーに収集済み性能値データのリストを登録します。
	 * 
 	 * @param facilityID ファシリティID　
	 * @param itemCode 収集項目コード
	 * @param data 性能値データのリスト
	 */
	public void setCollectedDataList(String facilityID, CollectorItemInfo key, List<CollectedDataInfo> data){
		HashMap<CollectorItemInfo, List<CollectedDataInfo>> subDataMap = dataMap.get(facilityID);
		
		if(subDataMap == null){
			// このファシリティIDのデータは初めての登録であるため新規作成
			subDataMap = new HashMap<CollectorItemInfo, List<CollectedDataInfo>>();
		}
		subDataMap.put(key, data);
		
		dataMap.put(facilityID, subDataMap);
	}
	
	/**
	 * 現在の性能値データのセットに、
	 * 指定のCollectedDataSetの収集済み性能値データを追加します。
	 * 同じ facilityID, itemCode のデータがある場合は上書きされます。
	 * 
 	 * @param dataSet 追加する性能値データ
	 */
	public void addCollectedDataList(CollectedDataSet dataSet){
		Set<String> facilityIDSet = dataSet.dataMap.keySet();
		Iterator<String> fItr = facilityIDSet.iterator();
		
		while(fItr.hasNext()){
			String facilityID = (String)fItr.next();

			HashMap<CollectorItemInfo, List<CollectedDataInfo>> subDataMap = dataSet.dataMap.get(facilityID);
			Set<CollectorItemInfo> pks = subDataMap.keySet();		
			Iterator<CollectorItemInfo> iItr = pks.iterator();			
			
			while(iItr.hasNext()){
				CollectorItemInfo key = iItr.next();
				
				// 移し替えもとのデータを取得する
				List<CollectedDataInfo> dataList = dataSet.getCollectedDataList(facilityID, key);
				
				// データをセットする
				setCollectedDataList(facilityID, key, dataList);
			}
		}
	}
	
	/**
	 * 指定のファシリティIDと収集項目IDをキーに収集済み性能値情報のリストを取得します。
	 * 
	 * @param facilityID ファシリティID
	 * @param itemCode 収集項目コード
	 * @return 性能値データのリスト
	 */
	public List<CollectedDataInfo> getCollectedDataList(String facilityID, CollectorItemInfo key){
		HashMap<CollectorItemInfo, List<CollectedDataInfo>> subDataMap = dataMap.get(facilityID);
		if(subDataMap != null){
			return subDataMap.get(key);
		} else {
			// 空のリストを生成して返す
			return new ArrayList<CollectedDataInfo>();
		}
	}
	
	/**
	 * 指定のファシリティIDで登録されている収集済みデータリストの数を取得します。
	 * 
	 * @param facilityID ファシリティID
	 * @return 指定のファシリティIDで登録されている収集済みデータリストの数
	 */
	public int getDataListNum(String facilityID){
		HashMap<CollectorItemInfo, List<CollectedDataInfo>> subDataMap = dataMap.get(facilityID);
		if(subDataMap != null){
			return subDataMap.size();
		} else {
			return 0;
		}
	}
}
