/*
 
Copyright (C) 2009 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.repository.dao.cache;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.repository.dao.FacilityTreeDAOImpl;
import com.clustercontrol.repository.ejb.entity.FacilityTreeBean;
import com.clustercontrol.repository.ejb.entity.FacilityTreePK;
import com.clustercontrol.repository.util.RepositoryCacheProperties;

public class CachedFacilityTreeDAOImpl extends FacilityTreeDAOImpl {
	private static Log m_log = LogFactory.getLog( CachedFacilityTreeDAOImpl.class );
	
	// 初期化されているか否かを示すフラグ
	private Boolean m_isInit = false;
	
	// 全てのファシリティのキャッシュ
	private static HashMap<ArrayList, Collection> m_findAllCache = null;
	private static long m_findAllCacheHitCount = 0;

	// ファシリティIDからの検索キャッシュ
	private static HashMap<String, Collection> m_facilityIdCache = null;
	private static long m_facilityIdCacheHitCount = 0;
	
	// 保持しているキャッシュ用HashMapのリスト
	private static List<HashMap> m_cacheMapList = new ArrayList<HashMap>();
	
	// 初期化
	@ Override
	public void init() {
		synchronized (m_isInit) {
			if (m_isInit) {
				// 既に初期化されているため何もしない
				return;
			} else {
				// キャッシュ対象となるもののHashMapを生成
				if (RepositoryCacheProperties
						.getProperties(RepositoryCacheProperties.FACILITY_TREE_FIND_ALL_KEY) > 0) {
					m_findAllCache = new HashMap<ArrayList, Collection>();
					m_cacheMapList.add(m_findAllCache);
					m_log.info("FacilityTreeDAO findAllCache : enable");
				}

				if (RepositoryCacheProperties
						.getProperties(RepositoryCacheProperties.FACILITY_TREE_FACILITY_ID_KEY) > 0) {
					m_facilityIdCache = new HashMap<String, Collection>();
					m_cacheMapList.add(m_facilityIdCache);
					m_log.info("FacilityTreeDAO facilityIdCache : enable");
				}

				m_isInit = true;
			}
		}
	}
	
	/**
	 * 全てのキャッシュをクリアする。
	 */
	private static void clearCacheAll(){
		synchronized (m_cacheMapList) {
			Iterator<HashMap> itr = m_cacheMapList.iterator();
			while(itr.hasNext()){
				HashMap map = itr.next();
				
				synchronized (map) {
					m_log.debug("clearCache() clear : " + map.size());
					map.clear();
				}
			}
		}
	}
	
//	LDAPアクセス必須
//	public void load(FacilityTreePK pk, FacilityTreeBean ejb) throws EJBException {
//	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#store(com.clustercontrol.repository.ejb.FacilityTreeBean)
	 */
	@Override
	public void store(FacilityTreeBean ejb) throws EJBException {
		m_log.debug("store()");
		
		super.store(ejb);
		
		// キャッシュをクリアする
		// （更新後にクリアしないと他のスレッドによりキャッシュの更新が入る可能性がある）
		clearCacheAll();
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#remove(com.clustercontrol.repository.ejb.FacilityTreePK)
	 */
	@Override
	public void remove(FacilityTreePK pk) throws RemoveException, EJBException {
		m_log.debug("remove()");
		
		super.remove(pk);
		
		// キャッシュをクリアする
		// （更新後にクリアしないと他のスレッドによりキャッシュの更新が入る可能性がある）
		clearCacheAll();
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#create(com.clustercontrol.repository.ejb.FacilityTreeBean)
	 */
	@Override
	public FacilityTreePK create(FacilityTreeBean ejb) throws CreateException, EJBException {
		m_log.debug("create()");
		
		FacilityTreePK pk = super.create(ejb);
		
		// キャッシュをクリアする
		// （更新後にクリアしないと他のスレッドによりキャッシュの更新が入る可能性がある）
		clearCacheAll();
		
		return pk;
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findAll()
	 */
	@Override
	public Collection findAll() throws FinderException {
		return findAll(null, null);
	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findAll(java.lang.String)
	 */
	@Override
	public Collection findAll(String facilityId) throws FinderException {
		return findAll(facilityId, null);
	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findAllNode(java.lang.String)
	 */
	@Override
	public Collection findAllNode(String facilityId) throws FinderException {
		return findAll(facilityId, new Integer(FacilityConstant.TYPE_NODE));
	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findAllScope(java.lang.String)
	 */
	@Override
	public Collection findAllScope(String facilityId) throws FinderException {
		return findAll(facilityId, new Integer(FacilityConstant.TYPE_SCOPE));
	}

	/**
	 * ファシリティツリー配下検索
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @param type 検索対象（ノードまたはスコープ）
	 * @return
	 * @throws FinderException
	 */
	@Override	
	public Collection findAll(String facilityId, Integer type) throws FinderException {
		// キャッシュを利用しない設定の場合
		if(m_findAllCache == null){
			return super.findAll(facilityId, type);
		}
		
		ArrayList<Object> key = new ArrayList<Object>();
		key.add(facilityId);
		key.add(type);

		synchronized (m_findAllCache) {
			// キャッシュに存在する場合はそれを返す
			if(m_findAllCache.containsKey(key)){
				m_findAllCacheHitCount++;
				if (m_log.isDebugEnabled()) {
					if(m_findAllCacheHitCount % 1000 == 0){ // 1000単位で出力
						m_log.debug("findAll(String facilityId, Integer type) hit : " + m_findAllCacheHitCount);
					}
				}
				return m_findAllCache.get(key);
			}
		}
		
		// キャッシュに存在しない場合
		Collection ret = super.findAll(facilityId, type);
		synchronized (m_findAllCache) {
			if (m_log.isDebugEnabled()) {
				m_log.debug("findAll(String facilityId, Integer type) put : "
						+ facilityId + ", " + type + "  entry size : "
						+ ret.size() + "  total entry : "
						+ m_findAllCache.size());
			}
			m_findAllCache.put(key, ret);
		}
		return ret;
	}
	
//	EntityBeanの機能でキャッシュされるためキャッシュ対象からはずす
//	public FacilityTreePK findByPrimaryKey(FacilityTreePK pk) throws FinderException {
//	}

//  対象外とする
//	public Collection findChildren(FacilityTreePK pk) throws FinderException {
//	}
	
//  リポジトリ機能内からの呼び出しのみであるため対象外とする
//	public FacilityTreePK findScopeByFacilityId(String facilityId) throws FinderException {
//	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findNodeByFacilityId(java.lang.String)
	 */
	@Override
	public Collection findByFacilityId(String facilityId) throws FinderException {
		// キャッシュを利用しない設定の場合
		if(m_facilityIdCache == null){
			return super.findByFacilityId(facilityId);
		}

		synchronized (m_facilityIdCache) {
			// キャッシュに存在する場合はそれを返す
			if(m_facilityIdCache.containsKey(facilityId)){
				m_facilityIdCacheHitCount++;
				if (m_log.isDebugEnabled()) {
					if(m_facilityIdCacheHitCount % 1000 == 0){ // 1000単位で出力
						m_log.debug("findByFacilityId(String facilityId) hit : " + m_facilityIdCacheHitCount);
					}
				}
				return m_facilityIdCache.get(facilityId);
			}
		}
		
		// キャッシュに存在しない場合
		Collection ret = super.findByFacilityId(facilityId);
		synchronized (m_facilityIdCache) {
			if (m_log.isDebugEnabled()) {
				m_log.debug("findByFacilityId(String facilityId) put : "
						+ facilityId + "  entry size : " + ret.size()
						+ "  total entry : " + m_facilityIdCache.size());
			}
			m_facilityIdCache.put(facilityId, ret);
		}
		return ret;
	}
	
//	/* (non-Javadoc)
//	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findOneLevel()
//	 */
//	@Override
//	public Collection findOneLevel() throws FinderException {
//		return findOneLevel(null, null);
//	}
//	
//	/* (non-Javadoc)
//	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findOneLevel()
//	 */
//	@Override
//	public Collection findOneLevel(String facilityId) throws FinderException {
//		return findOneLevel(facilityId, null);
//	}
//
//	/* (non-Javadoc)
//	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findOneLevelNode(java.lang.String)
//	 */
//	@Override
//	public Collection findOneLevelNode(String facilityId) throws FinderException {
//		return findOneLevel(facilityId, new Integer(FacilityConstant.TYPE_NODE));
//	}
//
//	/* (non-Javadoc)
//	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findOneLevelScope(java.lang.String)
//	 */
//	@Override
//	public Collection findOneLevelScope(String facilityId) throws FinderException {
//		return findOneLevel(facilityId, new Integer(FacilityConstant.TYPE_SCOPE));
//	}
//	
//	/**
//	 * ファシリティツリー直下検索
//	 * 
//	 * @version 1.0.0
//	 * @since 1.0.0
//	 * 
//	 * @param facilityId ファシリティID
//	 * @param type 検索対象（ノードまたはスコープ）
//	 * @return
//	 * @throws FinderException
//	 */
//	@Override
//	public Collection findOneLevel(String facilityId, Integer type) throws FinderException {
//	}
}
