package org.postgresforest.mng;

import java.io.IOException;
import java.lang.ref.WeakReference;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.logging.ConsoleHandler;
import java.util.logging.FileHandler;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;

import net.jcip.annotations.GuardedBy;

import org.postgresforest.constant.ConstInt;
import org.postgresforest.constant.ConstStr;
import org.postgresforest.constant.ErrorStr;
import org.postgresforest.constant.LogStr;
import org.postgresforest.constant.UdbValidity;
import org.postgresforest.exception.ForestException;
import org.postgresforest.exception.ForestInitFailedException;
import org.postgresforest.exception.ForestInvalidMngInfoException;
import org.postgresforest.util.ForestLogFormatter;
import org.postgresforest.util.PgUrl;
import org.postgresforest.util.ForestThreadFactory;
import org.postgresforest.util.ForestUrl;
import org.postgresforest.util.RecoveryCompletedListener;

/**
 * 管理情報を管理するためのクラス
 */
public final class MngInfoManager {
    
    /** 管理情報の読み込みをやコネクション作成を行うスレッドプール */
    private final ExecutorService executor;
    
    /** 管理情報の定期読み込みのマスタースレッド */
    private final ScheduledExecutorService refreshExecutor;
    
    /** 管理情報マネージャが接続する先のURLを示したForestUrlオブジェクト */
    private final ForestUrl forestUrl;
    
    /** 現在の管理情報のスナップショット */
    private final AtomicReference<MngInfo> mngInfo = new AtomicReference<MngInfo>();
    
    /** 現在の管理情報のスナップショットを返す */
    public MngInfo getMngInfo() {
        return mngInfo.get();
    }
    
    /**
     * java.util.logging.Loggerのインスタンス。MngInfoManagerのinit()関数内で
     * ロガーを初期化し、この変数にインスタンスを割り当てる。<br>
     * この変数を直接使わず、必ず logWithException か logWithoutException を使ってロギングを行う
     */
    private volatile Logger logger;
    
    /**
     * 通常運転時のログを出力するためのログ関数
     * @param level
     * @param message
     */
    private void logWithoutException(Level level, String message) {
        if (logger != null) {
            logger.log(level, message);
        }
    }
    
    /**
     * 問題が発生した際のログを出力するためのログ関数<br>
     * この関数は、ロギングレベルがCONFIG以下の場合のみ、
     * スタックトレース情報を出力する。
     * @param level
     * @param message
     * @param throwable
     */
    private void logWithException(Level level, String message, Throwable throwable) {
        if (logger != null) {
            // 現在ロガーにセットされているログレベルがCONFIG未満の場合には、スタックトレースまで出力する
            if (logger.getLevel().intValue() < Level.CONFIG.intValue()) {
                logger.log(level, message, throwable);
            } else {
                logger.log(level, message);
            }
        }
    }
    
    /**
     * リカバリ時のAPI実行抑制のためのラッチ。このAtomicReferenceにラッチをセットした場合、
     * セットしたラッチは必ずいつかのタイミングでCountDownされなくてはならない。
     */
    private final AtomicReference<CountDownLatch> waitRecoveryLatch = new AtomicReference<CountDownLatch>(null);
    
    private final Properties prop;
    
    /**
     * コンストラクタ<br>
     * Executorの初期化
     */
    public MngInfoManager(final ForestUrl forestUrl, final Properties prop) {
        this.forestUrl = forestUrl;
        this.prop = (Properties) prop.clone();
        executor = Executors.newCachedThreadPool(new ForestThreadFactory(forestUrl, "MngInfoQueryExecThreadPool"));
        refreshExecutor = Executors.newScheduledThreadPool(1, new ForestThreadFactory(forestUrl, "MngInfoRefreshThreadPool"));
    }
    
    /** この管理情報管理クラスの各種リソースを解放する（リソースマネージャが呼ぶべき） */
    public void destroy() {
        executor.shutdownNow();
        refreshExecutor.shutdownNow();
        for (int i = 0; i < 2; i++) {
            if (mngInfoReadCons.get(i) != null) {
                try { mngInfoReadCons.get(i).close(); } catch (SQLException ignore) { }
            }
        }
    }
    
    /**
     * 管理DBにアクセスし、初回アクセス時の管理情報を作成する<br>
     * <br>
     * この関数は以下の処理を行う<br>
     * 1. この管理情報マネージャがアクセスすべきForestUrlを基に、コネクションを生成<br>
     * 2. （１個以上コネクションができたなら）各コネクションからMngInfoを生成<br>
     * 3. （MngInfoが２個生成できたなら）両方のMngInfoをドッキング<br>
     * 4. コネクションを作成したForestUrlに指定されたIP:PORTが、管理情報DBから<br>
     *    取得したIP:PORTと等しいかを検証<br>
     * 5. 全て通ったなら3（ないしは2）で生成したMngInfoを公開する<br>
     * 6. ForestUrl.toString()の名前でjava.util.logger.Loggerを生成、初期化する<br>
     * 
     * @throws ForestException （インタラプト以外の理由で）初期化に失敗した場合
     * @throws InterruptedException このスレッドがinterruptされた場合
     */
    public void init() throws ForestException, InterruptedException {
        
        // コネクションプールを補充する
        supplyConnection();
        // プールのコネクションを使ってデータベースからMngInfoを読み込む
        final List<MngInfo> mngInfoList = readMngInfoFromDb();
        // データベースから読み込んだMngInfoのリストを1つに集約する
        final MngInfo newMngInfo;
        switch (mngInfoList.size()) {
            case 1: {
                // 片系から取得できたなら、それをデータベースから読んだMngInfoとして扱う
                newMngInfo = mngInfoList.get(0);
                break;
            }
            case 2: {
                // 両系から取得できたなら、MngInfoの合成を行う
                newMngInfo  = MngInfo.synthMngInfoFromDatabase(mngInfoList.get(0), mngInfoList.get(1));
                if (newMngInfo == null) {
                    // 合成に失敗した場合はinitに失敗
                    throw new ForestInitFailedException(ErrorStr.MNGINIT_SYNTH_INFO_FAILED.toString());
                }
                break;
            }
            default: {
                // 1個もとれなかった場合はinitに失敗
                throw new ForestInitFailedException(ErrorStr.MNGINIT_LOAD_INFO_FAILED.toString());
            }
        }
        // ここまで例外なしで通ったなら、作ったMngInfoを公開する
        mngInfo.set(newMngInfo);
        
        initLogger();
        
        // 管理情報更新の間隔が0以上なら、管理情報更新のスレッドを開始する
        if (newMngInfo.getGlobalConfig().getMngdbReadDuration() > 0) {
            // 初期実行は10秒後、それ以降はタスクが終わり次第1秒後に再度タスクを再開する。
            // （タスク自身が内部的にsleepを発行する。このsleepがMngInfo中のリフレッシュ間隔に従って
            // 動作しているため、ここで間隔の調整をする必要はない。また、1秒の間隔を設定しているのは
            // 万一Sleepが短時間で戻るようになってしまった場合に高負荷状態となるのを防ぐため）
            refreshExecutor.scheduleWithFixedDelay(new MngInfoRefreshTask(), 10, 1, TimeUnit.SECONDS);
            logWithoutException(Level.INFO, LogStr.INFO_MNGINFO_REFRESH_TASK_START.toString());
        }
    }
    
    /**
     * 現在の管理情報のglobal_configの値に従って、ロガーの設定を適切なものに
     * 変更する関数。この関数を呼んでいる間に他のスレッドがロガーを使用した場合、
     * ログロストする可能性がある<br>
     * この関数は複数スレッドから同時に呼ばれることを想定していない
     */
    private void initLogger() {
        final MngInfo.LogConfig logConfig = mngInfo.get().getGlobalConfig().getLogConfig();
        // ForestUrl.toStringの名前でロガーを作成
        // JVM起動後初めて呼ばれた場合であればインスタンスが新規に生成され、
        // そうでなければ既にあるロガーが呼び出される
        final Logger logger = Logger.getLogger(forestUrl.toString());
        // 親ロガー（rootロガー）のハンドラにメッセージが送られるのを抑止する
        logger.setUseParentHandlers(false);
        // 既存で登録されているハンドラを全消去する（以前に同一URLでForestが動作していた場合を考慮）
        for (final Handler handler : logger.getHandlers()) {
            handler.close();
            logger.removeHandler(handler);
        }
        // Logger側でログレベルをセットする（レベルを参照する際はロガーのレベルを参照する）
        logger.setLevel(logConfig.getLevel());
        
        // Formatterのインスタンスを生成する
        // ユーザに指定されたフォーマッタの生成に成功した場合はそれを使う。
        // 失敗した場合は全てForestでデフォルトで用意しているフォーマッタを使う
        Formatter logFormatter = null;
        try {
            final Object newFormatterObj = Class.forName(logConfig.getFormatterName()).newInstance();
            if (newFormatterObj != null && newFormatterObj instanceof java.util.logging.Formatter) {
                logFormatter = (Formatter) newFormatterObj;
            }
        } catch (ClassNotFoundException ignore) {
        } catch (IllegalAccessException ignore) {
        } catch (InstantiationException ignore) {
        } catch (SecurityException ignore) {
        }
        if (logFormatter == null) {
            logFormatter = new ForestLogFormatter();
        }
        
        // Handlerを生成
        final List<Handler> handlerList = new ArrayList<Handler>(2);
        // 標準エラー出力のハンドラ
        if (logConfig.getConsoleEnable() == true) {
            handlerList.add(new ConsoleHandler());
        }
        // ファイル出力のハンドラ
        if (logConfig.getFileEnable() == true) {
            // 指定されたファイル名、既存ファイルへの追記モードでファイルハンドラを開く
            try {
                handlerList.add(new FileHandler(logConfig.getFileName(), logConfig.getFileSizeMb() * 1024 * 1024, logConfig.getFileRotateCount(), true));
            } catch (SecurityException ignore) {
            } catch (IOException ignore) {
            } catch (IllegalArgumentException ignore) {
                // 設定のサイズ上限値やカウント値が無効な場合
            }
        }
        
        // Logger・Handler・Formatterを結合させる。
        // また、HandlerのログレベルはALLとする（Loggerのほうでフィルタしているため）
        for (final Handler handler : handlerList) {
            logger.addHandler(handler);
            handler.setFormatter(logFormatter);
            handler.setLevel(Level.ALL);
        }
        
        this.logger = logger;
    }
    
    /**
     * 指定されたidのユーザデータベースの整合性情報をINVALIDとして、
     * 新規MngInfoのスナップショットを作成・公開する。<br>
     * 縮退と判断した時点の管理情報を引数として取るが、この関数が管理情報を
     * 書き換えようとしている最中に他スレッドによって管理情報を変更されている
     * 場合、この関数はfalseを返し、何も実施しない。
     * @param oldMngInfo 縮退と判断した時点でのMngInfoのスナップショット
     * @param serverId 縮退状態へステータスを変更する対象のサーバID（0 or 1）
     * @param exception 縮退と判断した例外
     * @param taskClassName 縮退が発生した時に実行していたタスクのクラス名
     * @return 置き換えが成功した場合はtrue、置き換えまでに他のスレッドにより置き換えが発生していた場合はfalse
     */
    public boolean setUdbInvalid(final MngInfo oldMngInfo, final int serverId, final Exception exception, final String taskClassName) {
        // 現在のMngInfoから取得したServerInfoリストを基に、縮退させた後のServerInfoを作成する
        final List<MngInfo.ServerInfo> oldServerInfoList = oldMngInfo.getServerInfoList();
        final MngInfo.ServerInfo targetServerInfo = oldServerInfoList.get(serverId);
        final MngInfo.ServerInfo newServerInfo = 
            targetServerInfo.getNewValidityServerInfo(UdbValidity.INVALID);
        
        // 縮退させたServerInfoを含む、新しいServerInfoのリストを作成する
        // （oldServerInfoListは変更不可のため、新規にリストを生成する）
        final List<MngInfo.ServerInfo> newServerInfoList = new ArrayList<MngInfo.ServerInfo>();
        newServerInfoList.addAll(oldServerInfoList);
        newServerInfoList.set(serverId, newServerInfo);
        
        // 新規にMngInfoを作って登録する。仮にMngInfoを作ってから登録するまでの間に
        // 別のものに置き換えられていたとしたら、置き換えは失敗したことを呼び出し側に通知する
        final MngInfo newMngInfo = new MngInfo(newServerInfoList, oldMngInfo.getGlobalConfig(), oldMngInfo.getLocalConfigMap());
        if (mngInfo.compareAndSet(oldMngInfo, newMngInfo) == false) {
            return false;
        }
        // 置き換えに成功した場合、縮退の情報を必要な箇所に通知する
        // まずは実際にDBに書き込む（厳密には上のCompareAndSet操作をする段階から、DBへの
        // 書き込みが完了するまでの間を、他の処理が同時に行わないようにロックをとる必要が
        // あるが、仮にロックを握っても別アプリ（JVM）からアクセスがあればここでの排他制御の
        // 意味はなくなってしまうことと、DBアクセス中にロックを握りっぱなしになり危険なことと、
        // そもそも管理情報DBへの処理は非同期に行いたいため、ロックを握ることができない。
        // よってここで排他制御は行わない）
        final String appThreadName = Thread.currentThread().getName();
        final List<StackTraceElement> appThreadStack =
            Arrays.<StackTraceElement>asList(Thread.currentThread().getStackTrace());
        executor.execute(new WriteUdbInvalidTask(forestUrl.getMngDbUrls().get(0), serverId, exception, appThreadName, appThreadStack, taskClassName, executor, prop));
        executor.execute(new WriteUdbInvalidTask(forestUrl.getMngDbUrls().get(1), serverId, exception, appThreadName, appThreadStack, taskClassName, executor, prop));
        // 縮退を自発的に行ったことをロガーに通知
        logWithException(Level.SEVERE, LogStr.SEVERE_BROKEN_SERVER.toString() + serverId, exception);
        return true;
    }
    
    /**
     * 現在リカバリのために更新抑制フレーズであるならば動作が停止し、
     * 特に抑制がかかっていない状態であれば何もせずに即座に返る関数。
     * 
     * @throws InterruptedException ラッチを待つ間にinterruptされた場合
     */
    public void waitRecovery() throws InterruptedException {
        final CountDownLatch latch = waitRecoveryLatch.get();
        if (latch == null) {
            return;
        } else {
            latch.await();
        }
    }
    
    /**
     * 管理情報読み込みに使用する両系へのコネクションを格納する配列（コネクションプール）<br>
     * この配列を操作するのは複数スレッドの可能性があるため、AtomicReferenceを使用する
     */
    private final AtomicReferenceArray<Connection> mngInfoReadCons = new AtomicReferenceArray<Connection>(2);
    
    /**
     * 管理情報読み込みのための両系へのコネクションプール（mngConnections）を補給する関数<br>
     * この関数を呼ぶと、コネクションがプールに無い場合には補給し、コネクションがプールに
     * ある場合には何もしない
     * @throws InterruptedException コネクション生成最中に割り込みが発生した場合
     * @throws ForestException 両方の系に対してコネクションを生成できなかった場合
     */
    private void supplyConnection() throws InterruptedException, ForestException {
        // コネクションが無い系について、コネクション補充をするタスクを作る
        // コネクションが既にある系については、何も実行しないタスクを作る
        final List<Callable<Connection>> conCreateTasks = new ArrayList<Callable<Connection>>(2);
        for (int i = 0; i < 2; i++) {
            if (mngInfoReadCons.get(i) == null) {
                conCreateTasks.add(new CreateConnectionTask(forestUrl.getMngDbUrls().get(i), prop));
            } else {
                conCreateTasks.add(new CreateConnectionTask(null, null));
            }
        }
        // コネクション生成のタスクを実行する
        final List<Future<Connection>> conCreateFutureList;
        final int conCreateTimeout;
        if (getMngInfo() == null) {
            conCreateTimeout = ConstInt.MNGINIT_CONNECT_TIMEOUT.getInt();
        } else {
            conCreateTimeout = getMngInfo().getGlobalConfig().getMngConCreateTimeout();
        }
        try {
            conCreateFutureList = executor.invokeAll(conCreateTasks, conCreateTimeout, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            // 割り込みが発生した場合、スレッドに割り込みフラグを再度立てた上で即座に呼び出し元に戻る
            Thread.currentThread().interrupt();
            throw e;
        }
        // コネクション生成が両系共にできなかった場合に投げる例外リストを用意
        final List<Exception> exceptionList = new ArrayList<Exception>(2);
        
        // コネクション生成タスクの実行結果を取り出す
        for (int i = 0; i < 2; i++) {
            try {
                // コネクションが新規に生成できたならコネクションプールにセットする。
                // 出来ていない場合は何もしない。できていない＝nullタスクをセットした場合
                // なので、そもそも作る必要がないため例外を上げる必要がない。
                final Connection newCon = conCreateFutureList.get(i).get();
                if (newCon != null) {
                    mngInfoReadCons.set(i, newCon);
                }
                exceptionList.add(null);
            } catch (CancellationException e) {
                exceptionList.add(e);
                logWithoutException(Level.WARNING, LogStr.WARN_MNGINFO_SUPPLYCON_CANCEL.toString() + i);
            } catch (ExecutionException e) {
                if (e.getCause() instanceof Exception) {
                    exceptionList.add((Exception) e);
                    logWithException(Level.WARNING, LogStr.WARN_MNGINFO_SUPPLYCON_EXCEPTION.toString() + i, e.getCause());
                } else {
                    logWithException(Level.SEVERE, LogStr.SEVERE_MNGINFO_SUPPLYCON_ERROR.toString() + i, e.getCause());
                    throw (Error) e.getCause();
                }
            } catch (InterruptedException e) {
                // 割り込みが発生した場合、スレッドに割り込みフラグを再度立てた上で即座に呼び出し元に戻る
                Thread.currentThread().interrupt();
                throw e;
            }
        }
        // 両方実行して両方ともエラーだったなら例外を返す
        if (exceptionList.get(0) != null && exceptionList.get(1) != null) {
            final ForestException retExp = new ForestException(ErrorStr.MNGDB_CONNECT_FAILED.toString());
            retExp.setMultipleCause(exceptionList);
            logWithException(Level.SEVERE, LogStr.SEVERE_MNGINFO_SUPPLYCON_BOTHFAIL.toString(), retExp);
            throw retExp;
        }
    }
    
    /**
     * コネクションプール（mngConnections）に存在するコネクションを使い、MngInfoを読み出し、
     * 読みだせた分のMngInfoをリストとして返却する。MngInfoの読み込みに失敗した場合、
     * その系のコネクションは解放され、コネクションプールから取り除かれる。また、実行時に
     * 割り込みが起きた場合、nullが返却される。返却されるMngInfoは、MngInfoManagerが保持
     * しているForestUrlとサーバ接続先情報が等しいことが保障される。
     * また、両方の系からMngInfoを取得できなかった場合には、その旨エラーが返される
     * @throws InterruptedException 
     * @throws ForestException １つも管理情報を取れなかった場合
     */
    private List<MngInfo> readMngInfoFromDb() throws InterruptedException, ForestException {
        // MngInfoを各系から読み込むためのタスクを作る
        final List<Callable<MngInfo>> mnginfoReadTasks = new ArrayList<Callable<MngInfo>>(2);
        for (int i = 0; i < 2; i++) {
            mnginfoReadTasks.add( new ReadMngInfoTask(mngInfoReadCons.get(i)) );
        }
        // MngInfoを各系から読み込むタスクを実行する
        final int queryExecTimeout;
        if (getMngInfo() == null) {
            queryExecTimeout = ConstInt.MNGINIT_QUERYEXEC_TIMEOUT.getInt();
        } else {
            queryExecTimeout = getMngInfo().getGlobalConfig().getMngQueryExecTimeout();
        }
        final List<Future<MngInfo>> readMngInfoFutureList;
        try {
            readMngInfoFutureList = executor.invokeAll(mnginfoReadTasks, queryExecTimeout, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            // 割り込みが発生した場合、スレッドに割り込みフラグを再度立てそのまま例外を投げる
            Thread.currentThread().interrupt();
            throw e;
        }
        // MngInfo読み込みタスクの実行結果を取り出す
        final List<MngInfo> mngInfoList = new ArrayList<MngInfo>(2);
        final List<Exception> exceptionList = new ArrayList<Exception>(2);
        for (int i = 0; i < 2; i++) {
            final MngInfo tmpMngInfo;
            try {
                // 正常にMngInfoが取れたならリストに追加する
                tmpMngInfo = readMngInfoFutureList.get(i).get();
                if (tmpMngInfo != null && tmpMngInfo.equalServerInfo(forestUrl)) {
                    mngInfoList.add(tmpMngInfo);
                }
                exceptionList.add(null);
            } catch (InterruptedException e) {
                // 割り込みが発生した場合、スレッドに割り込みフラグを再度立てた上でそのまま例外を投げる
                Thread.currentThread().interrupt();
                throw e;
            } catch (ExecutionException e) {
                // 実行中にエラーが起きた場合、そのコネクションは解放する
                final Connection closeTarget = mngInfoReadCons.getAndSet(i, null);
                if (closeTarget != null) {
                    try {
                        // クエリが実行中の場合、バックエンドが止まってくれないためキャンセルする
                        ((org.postgresql.jdbc2.AbstractJdbc2Connection) closeTarget).cancelQuery();
                    } catch (SQLException ignore) {
                    } finally {
                        try { closeTarget.close(); } catch (SQLException ignore) { }
                    }
                }
                if (e.getCause() instanceof Error) {
                    logWithException(Level.SEVERE, LogStr.SEVERE_MNGINFO_READDB_ERROR.toString() + i, e.getCause());
                    throw (Error) e.getCause();
                } else {
                    exceptionList.add((Exception) e.getCause());
                    logWithException(Level.WARNING, LogStr.WARN_MNGINFO_READDB_EXCEPTION.toString() + i, e.getCause());
                }
            } catch (CancellationException e) {
                // 実行が指定時間内に終わらなかった場合、そのコネクションは解放する
                final Connection closeTarget = mngInfoReadCons.getAndSet(i, null);
                exceptionList.add(e);
                logWithoutException(Level.WARNING, LogStr.WARN_MNGINFO_READDB_CANCEL.toString() + i);
                if (closeTarget != null) {
                    try {
                        // クエリが実行中の場合、バックエンドが止まってくれないためキャンセルする
                        ((org.postgresql.jdbc2.AbstractJdbc2Connection) closeTarget).cancelQuery();
                    } catch (SQLException ignore) {
                    } finally {
                        try { closeTarget.close(); } catch (SQLException ignore) { }
                    }
                }
            }
        }
        // 1個も結果を取得できなかったなら例外を投げる
        if (mngInfoList.size() == 0) {
            final ForestException throwExp = new ForestException();
            throwExp.setMultipleCause(exceptionList);
            logWithException(Level.SEVERE, LogStr.SEVERE_MNGINFO_READDB_BOTHFAIL.toString(), throwExp);
            throw throwExp;
        }
        return mngInfoList;
    }
    
    private final class MngInfoRefreshTask implements Runnable {
        public MngInfoRefreshTask() {}
        public void run() {
            logWithoutException(Level.INFO, LogStr.INFO_MNGINFO_REFRESH_TASK_START.toString());
            // TODO （同期）管理情報のサーバ情報に差異があった場合に同期させる処理が必要
            final List<MngInfo> mngInfoList;
            try {
                // まずコネクションプールのコネクションを補充し
                supplyConnection();
                // プールのコネクションを使ってデータベースからMngInfoを読み込む
                mngInfoList = readMngInfoFromDb();
            } catch (InterruptedException e) {
                // MngInfo読み込み中に割り込まれた場合、割りこみを呼び出し元にそのまま広める
                // （終了処理に応答するため）
                Thread.currentThread().interrupt();
                return;
            } catch (ForestException e) {
                logWithoutException(Level.SEVERE, LogStr.SEVERE_MNGINFO_REFRESH_FAIL.toString());
                // 何らかの原因で両系の管理情報から情報を取れなかった場合、現在のリフレッシュ
                // 間隔に従い、タスクを一時停止し、次のタスクを実行する
                try {
                    final int duration = getMngInfo().getGlobalConfig().getMngdbReadDuration();
                    Thread.sleep(duration * 1000);
                } catch (InterruptedException ee) {
                    // 一時停止中に割り込まれた場合、interruptをセットした上で
                    // すみやかに終了する
                    Thread.currentThread().interrupt();
                }
                return;
            }
            
            // このタスク中で比較対象とする現在のMngInfoのスナップショットを取得しておく
            final MngInfo currentMngInfo = getMngInfo();
            
            // データベースから読み込んだMngInfoのリストを1つに集約する
            final MngInfo tmpMngInfo;
            switch (mngInfoList.size()) {
                case 1: {
                    // 片系から取得できたなら、それをデータベースから読んだMngInfoとして扱う
                    tmpMngInfo = mngInfoList.get(0);
                    break;
                }
                case 2: {
                    // 両系から取得できたなら、MngInfoの合成を行う
                    tmpMngInfo  = MngInfo.synthMngInfoFromDatabase(mngInfoList.get(0), mngInfoList.get(1));
                    break;
                }
                default: {
                    // 1個もとれなかった場合、現在JVM中に存在するMngInfoをそのまま使う
                    tmpMngInfo = currentMngInfo;
                    break;
                }
            }
            
            // JVM中のMngInfoのうち、ServerInfoの情報に関して状態遷移を行う。
            // 許可された状態遷移であれば新規のServerInfoを用い、
            // 許可されない状態遷移であれば以前からのServerInfoを用いる
            // FIXME （情報出力）許可されている状態遷移をWARN、許可されていない状態遷移をERRORとしてロギングする
            List<MngInfo.ServerInfo> newServerInfoList = currentMngInfo.getServerInfoList();
            switch (currentMngInfo.getEnumValidity()) {
                case VALID_VALID: {
                    switch (tmpMngInfo.getEnumValidity()) {
                        case VALID_INVALID:
                        case INVALID_VALID:
                        case INVALID_INVALID:
                            newServerInfoList = tmpMngInfo.getServerInfoList();
                            break;
                    }
                    break;
                }
                case VALID_INVALID: {
                    switch (tmpMngInfo.getEnumValidity()) {
                        case RECOVER_INVALID:
                            // リカバリのための抑制状態とする
                            waitRecoveryLatch.set(new CountDownLatch(1));
                        case INVALID_INVALID:
                            newServerInfoList = tmpMngInfo.getServerInfoList();
                    }
                    break;
                }
                case INVALID_VALID: {
                    switch (tmpMngInfo.getEnumValidity()) {
                        case INVALID_RECOVER:
                            // リカバリのための抑制状態とする
                            waitRecoveryLatch.set(new CountDownLatch(1));
                        case INVALID_INVALID:
                            newServerInfoList = tmpMngInfo.getServerInfoList();
                    }
                    break;
                }
                case RECOVER_INVALID: {
                    switch (tmpMngInfo.getEnumValidity()) {
                        case VALID_VALID:
                            // 2系がリカバリが完了したため、通知する
                            notifyRecoveryCompleted(1);
                        case VALID_INVALID:
                        case INVALID_INVALID:
                            // リカバリ状態解除のため、ラッチを解放・削除する
                            final CountDownLatch latch = waitRecoveryLatch.get();
                            if (latch != null) {
                                latch.countDown();
                            }
                            waitRecoveryLatch.set(null);
                            newServerInfoList = tmpMngInfo.getServerInfoList();
                    }
                    break;
                }
                case INVALID_RECOVER: {
                    switch (tmpMngInfo.getEnumValidity()) {
                        case VALID_VALID:
                            // １系がリカバリが完了したため、通知する
                            notifyRecoveryCompleted(0);
                        case INVALID_VALID:
                        case INVALID_INVALID:
                            // リカバリが完了したためラッチを解放・削除する
                            final CountDownLatch latch = waitRecoveryLatch.get();
                            if (latch != null) {
                                latch.countDown();
                            }
                            waitRecoveryLatch.set(null);
                            newServerInfoList = tmpMngInfo.getServerInfoList();
                    }
                    break;
                }
            }
            
            // 状態遷移を行った後のServerInfoを基にして、新規MngInfoを決定する
            // コンフィグ値については新規にDB側から読み込んだものを無条件に使う
            final MngInfo newMngInfo = new MngInfo(newServerInfoList, tmpMngInfo.getGlobalConfig(), tmpMngInfo.getLocalConfigMap());
            
            // 現在のMngInfoと新規に作ったMngInfoが論理的に異なる場合のみ変数を書き換える
            if (!currentMngInfo.equals(newMngInfo)) {
                if (mngInfo.compareAndSet(currentMngInfo, newMngInfo) == false) {
                    // AtomicReferenceのcompareAndSetがfalseで戻っている場合、
                    // この関数の頭でcurrentMngInfoを読み出してからここまでの間に
                    // 他者にmngInfoの参照を付け替えられていることを意味する。
                    // そのため、再度この関数を実行する必要があるので、
                    // 以下の待ちを行わず、即関数を終える（次のスケジュール実行を行う）
                    logWithoutException(Level.INFO, LogStr.INFO_MNGINFO_REFRESH_CONFLICT.toString());
                    return;
                }
                // MngInfoの切り替えに成功し、なおかつLogConfigが以前のものと論理的に
                // 異なっている場合には、ロガーを新しい設定で起動しなおす
                if (!currentMngInfo.getGlobalConfig().getLogConfig().equals(
                        newMngInfo.getGlobalConfig().getLogConfig())) {
                    initLogger();
                }
            }
            logWithoutException(Level.INFO, LogStr.INFO_MNGINFO_REFRESH_TASK_END.toString());
            try {
                final int duration = newMngInfo.getGlobalConfig().getMngdbReadDuration();
                // 現在のリフレッシュ間隔に従い、タスクを一時停止する
                Thread.sleep(duration * 1000);
            } catch (InterruptedException e) {
                // 一時停止中に割り込まれた場合、interruptをセットした上で
                // すみやかに終了する
                Thread.currentThread().interrupt();
            }
        }
    }
    
    /**
     * 与えられたPgUrlを使って、対応する管理データベースへの
     * コネクションを作成するためのタスククラス<br>
     * コンストラクタで与えられたPgUrlがnullの場合は何も行わない
     */
    private static final class CreateConnectionTask implements Callable<Connection> {
        private final PgUrl targetUrl;
        private final Properties prop;
        public CreateConnectionTask(final PgUrl targetUrl, final Properties prop) {
            this.targetUrl = targetUrl;
            this.prop = prop;
        }
        public Connection call() throws Exception {
            if (targetUrl == null) {
                return null;
            }
            // DriverManager.getConnectionを使わない理由は、
            // 1. 作るインスタンスが分かっているにも関わらずDriverManagerを
            //    経由するのはオーバーヘッドが大きい
            // 2. Java5のDriverManagerがロックを握る実装になっており、
            //    executorでDriverManager.getConnectionを実行すると、
            //    確実にデッドロックが発生してしまう
            // の2点
            final java.sql.Driver driver = new org.postgresql.Driver();
            
            // TODO （仕様検討） 管理情報へのアクセスユーザ名・パスワードをどうするか
            return driver.connect(targetUrl.getUrl(), prop);
            // 下記コードは以前の仕様（ユーザ名・パスワードを、管理DBのDB名と等しいもの
            // で固定する）という場合のものになる
            // return driver.connect(targetUrl.getUrl(), targetUrl.getProperty());
        }
    }
    
    /**
     * 与えられたコネクションを使用して、管理情報DBから情報を読み込んで
     * MngInfoを作成するためのタスククラス<br>
     * このクラスのcall()は、コネクションの解放を行わない<br>
     * また、コンストラクタで与えられたコネクションがnullの場合は何も行わずにnullを返す
     */
    private static final class ReadMngInfoTask implements Callable<MngInfo> {
        private final Connection con;
        public ReadMngInfoTask(final Connection con) {
            this.con = con;
        }
        public MngInfo call() throws Exception {
            if (con == null) {
                return null;
            }
            final Statement stmt = con.createStatement();
            try {
                // ServerInfoの生成
                // サーバIDは0か1なので、サイズが2のnullを格納した配列を最初に作り、
                // そこにそれぞれDBから読んだ値を格納する。
                final List<MngInfo.ServerInfo> serverInfoList = new ArrayList<MngInfo.ServerInfo>();
                serverInfoList.add(null);
                serverInfoList.add(null);
                final ResultSet svinfoRes = stmt.executeQuery("SELECT * FROM " + ConstStr.MNGDB_TBL_SERVERINFO.toString());
                // 各行をDBから読んで、ServerInfoのリストを構築する
                while (svinfoRes.next()) {
                    final int serverid = svinfoRes.getInt(ConstStr.MNGDB_COL_SERVERINFO_ID.toString());
                    if (serverid != 0 && serverid != 1) {
                        throw new ForestInvalidMngInfoException(ErrorStr.MNGDB_SERVERID_INVALID.toString());
                    }
                    final String udburl = svinfoRes.getString(ConstStr.MNGDB_COL_SERVERINFO_UDBURL.toString());
                    final UdbValidity udbValidity = 
                        UdbValidity.getEnum(svinfoRes.getInt(ConstStr.MNGDB_COL_SERVERINFO_VALID.toString()));
                    if (udbValidity == null) {
                        throw new ForestInvalidMngInfoException(ErrorStr.MNGDB_UDBVALIDITY_INVALID.toString());
                    }
                    serverInfoList.set(serverid, new MngInfo.ServerInfo(udburl, udbValidity));
                }
                // ServerInfoのリストに、インデックス0・1共に値がセットされているか確認する
                if (serverInfoList.get(0) == null || serverInfoList.get(1) == null) {
                    throw new ForestInvalidMngInfoException(ErrorStr.MNGDB_SERVERID_INVALID.toString());
                }
                
                // GlobalConfig（LogConfig含む）の作成
                final ResultSet gconfRes = stmt.executeQuery("SELECT * FROM " + ConstStr.MNGDB_TBL_GLOBALCONFIG.toString() + " LIMIT 1");
                if (gconfRes.next() == false) {
                    throw new ForestInvalidMngInfoException(ErrorStr.MNGDB_GCONFIG_INVALID.toString());
                }
                final int logLevel = gconfRes.getInt(ConstStr.MNGDB_COL_GCONF_LOGLEVEL.toString());
                final String logFormatter = gconfRes.getString(ConstStr.MNGDB_COL_GCONF_LOGFORMATTERNAME.toString());
                final boolean logConsoleEnable = gconfRes.getBoolean(ConstStr.MNGDB_COL_GCONF_LOGCONSOLEENABLE.toString());
                final boolean logFileEnable = gconfRes.getBoolean(ConstStr.MNGDB_COL_GCONF_LOGFILEENABLE.toString());
                final String logFileName = gconfRes.getString(ConstStr.MNGDB_COL_GCONF_LOGFILENAME.toString());
                final int logFileSize = gconfRes.getInt(ConstStr.MNGDB_COL_GCONF_LOGFILESIZE.toString());
                final int logFileRotate = gconfRes.getInt(ConstStr.MNGDB_COL_GCONF_LOGFILEROTATE.toString());
                final MngInfo.LogConfig logConfig = new MngInfo.LogConfig(logLevel, logFormatter, logConsoleEnable, logFileEnable, logFileName, logFileSize, logFileRotate);
                
                final int mngdbReadDuration = gconfRes.getInt(ConstStr.MNGDB_COL_GCONF_READDURATION.toString());
                final boolean enableSyncServerinfo = gconfRes.getBoolean(ConstStr.MNGDB_COL_GCONF_SYNCSVINFO.toString());
                final int mngConCreateTimeout = gconfRes.getInt(ConstStr.MNGDB_COL_GCONF_MNGCONCREATETO.toString());
                final int mngQueryExecTimeout = gconfRes.getInt(ConstStr.MNGDB_COL_GCONF_MNGQUERYEXECTO.toString());
                final boolean collectRuntimeInfo = gconfRes.getBoolean(ConstStr.MNGDB_COL_GCONF_COLLECTINFO.toString());
                final String brokenErrorPrefix = gconfRes.getString(ConstStr.MNGDB_COL_GCONF_BROKENPREFIX.toString());
                final MngInfo.GlobalConfig globalConfig = 
                    new MngInfo.GlobalConfig(mngdbReadDuration, enableSyncServerinfo, mngConCreateTimeout,
                            mngQueryExecTimeout, collectRuntimeInfo, brokenErrorPrefix, logConfig);
                
                // LocalConfigの作成
                final ResultSet lconfRes = stmt.executeQuery("SELECT * FROM " + ConstStr.MNGDB_TBL_LOCALCONFIG.toString());
                final Map<String, MngInfo.LocalConfig> localConfigMap = new HashMap<String, MngInfo.LocalConfig>();
                while (lconfRes.next()) {
                    final String configid = lconfRes.getString(ConstStr.MNGDB_COL_LCONF_CONFIGID.toString());
                    final int conCreateTimeout = lconfRes.getInt(ConstStr.MNGDB_COL_LCONF_CONCREATETO.toString());
                    final int queryExecTimeout = lconfRes.getInt(ConstStr.MNGDB_COL_LCONF_QUERYEXECTO.toString());
                    localConfigMap.put(configid, new MngInfo.LocalConfig(conCreateTimeout, queryExecTimeout));
                }
                
                return new MngInfo(serverInfoList, globalConfig, localConfigMap);
                
            } finally {
                stmt.close();
            }
        }
    }
    
    /**
     * 
     */
    private static final class WriteUdbInvalidTask implements Runnable {
        private final PgUrl pgUrl;
        private final int serverId;
        private final ExecutorService executor;
        private final Exception exception;
        private final String appThreadName;
        private final List<StackTraceElement> appThreadStack;
        private final String taskClassName;
        private final Properties prop;
        public WriteUdbInvalidTask(PgUrl pgUrl, int serverId, Exception exception, String appThreadName, List<StackTraceElement> appThreadStack, String taskClassName, ExecutorService executor, Properties prop) {
            this.pgUrl = pgUrl;
            this.serverId = serverId;
            this.executor = executor;
            this.exception = exception;
            this.appThreadName = appThreadName;
            this.taskClassName = taskClassName;
            this.appThreadStack = appThreadStack;
            this.prop = prop;
        }
        public void run() {
            // 書き込むためのコネクションを作成
            final Callable<Connection> createConTask = new CreateConnectionTask(pgUrl, prop);
            final Future<Connection> createConFuture = executor.submit(createConTask);
            final Connection con;
            try {
                con = createConFuture.get(10, TimeUnit.SECONDS);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return;
            } catch (ExecutionException e) {
                return;
            } catch (TimeoutException e) {
                return;
            }
            
            // 実際にDBに縮退情報を書き込む
            Statement stmt = null;
            PreparedStatement pstmt = null;
            try {
                
                // serverinfoの更新
                stmt = con.createStatement();
                final StringBuilder updateBuffer = new StringBuilder();
                updateBuffer.append("UPDATE ");                                           // UPDATE
                updateBuffer.append(ConstStr.MNGDB_TBL_SERVERINFO.toString());            // server_info
                updateBuffer.append(" SET ");                                             // SET
                updateBuffer.append(ConstStr.MNGDB_COL_SERVERINFO_VALID.toString());      // udb_validity
                updateBuffer.append(" = ");                                               // =
                updateBuffer.append(UdbValidity.INVALID.getInt());                        // -1
                updateBuffer.append(" WHERE ");                                           // WHERE
                updateBuffer.append(ConstStr.MNGDB_COL_SERVERINFO_ID.toString());         // serverid
                updateBuffer.append(" = ");                                               // =
                updateBuffer.append(serverId);                                            // 引数のID
                stmt.executeUpdate(updateBuffer.toString());
                
                // brokenlogの挿入
                final StringBuilder insertBuffer = new StringBuilder();
                insertBuffer.append("INSERT INTO ");
                insertBuffer.append(ConstStr.MNGDB_TBL_BROKENLOG.toString());
                insertBuffer.append(" (");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_SERVERID.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_TIME.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_CLIENT.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_APITASK.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_ERRMSG.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_ERRTYPE.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_ERRSTATE.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_ERRQUERY.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_ERRSTACK.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_APPTHREAD.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_APPSTACK.toString());
                insertBuffer.append(") VALUES (?, now(), ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                pstmt = con.prepareStatement(insertBuffer.toString());
                
                // serverid
                pstmt.setInt(1, serverId);
                // client
                final List<String> ipList = new ArrayList<String>();
                try {
                    for (final NetworkInterface nic : Collections.list(NetworkInterface.getNetworkInterfaces())) {
                        for (final InetAddress addr : Collections.list(nic.getInetAddresses())) {
                            ipList.add(addr.getHostAddress());
                        }
                    }
                } catch (Exception e) {
                    ipList.clear();
                    ipList.add("-");
                }
                Collections.sort(ipList);
                pstmt.setString(2, ipList.toString());
                // api_task
                pstmt.setString(3, taskClassName);
                // err_msg
                pstmt.setString(4, exception.toString());
                // err_type
                pstmt.setString(5, exception.getClass().getName());
                // err_state
                String sqlState;
                if (exception instanceof SQLException) {
                    sqlState = ((SQLException) exception).getSQLState();
                } else {
                    sqlState = "-";
                }
                pstmt.setString(6, (sqlState != null) ? sqlState : "-");
                // err_query
                // TODO （情報出力）縮退時のクエリを取得する？
                pstmt.setString(7, "-");
                // err_stack
                final StringBuilder errStackBuffer = new StringBuilder();
                for (final StackTraceElement elem : exception.getStackTrace()) {
                    errStackBuffer.append("\n\t");
                    errStackBuffer.append(elem.toString());
                }
                pstmt.setString(8, errStackBuffer.toString());
                // app_thread
                pstmt.setString(9, appThreadName);
                // app_stack
                final StringBuilder appStackBuffer = new StringBuilder();
                for (final StackTraceElement elem : appThreadStack) {
                    appStackBuffer.append("\n\t");
                    appStackBuffer.append(elem.toString());
                }
                pstmt.setString(10, appStackBuffer.toString());
                
                pstmt.executeUpdate();
                
            } catch (SQLException ignore) {
            } finally {
                if (stmt != null) {
                    try { stmt.close(); } catch (SQLException ignore) { }
                }
                if (pstmt != null) {
                    try { pstmt.close(); } catch (SQLException ignore) { }
                }
                try { con.close(); } catch (SQLException ignore) { }
            }
        }
    }
    
    /** recoveryListenerListを操作する場合に取得するロック。このロック取得無しに変更してはならない */
    private final Object lockListenerList = new Object();
    
    /**
     * リカバリ通知対象となるオブジェクトを保持するリスト。
     * MngInfoManagerからコネクションに関係するオブジェクトを握るのはまずい
     * （強参照による循環参照となる）ため、弱参照で保持することとする。
     * 弱参照なので、リストの要素を使用する際に必ずnullチェックが必要。
     */
    @GuardedBy("loclListenerList")
    private final List<WeakReference<RecoveryCompletedListener>> recoveryListenerList =
        new LinkedList<WeakReference<RecoveryCompletedListener>>();
    
    /**
     * リカバリが正常完了した際に、RecoveryCompletedListenerで定義された
     * コールバック関数を呼ぶように登録するための関数。
     * @param listener 登録したいコールバック関数を実装したオブジェクト
     */
    @GuardedBy("lockListenerList")
    public void setRecoveryCompletedListener(RecoveryCompletedListener newListener) {
        synchronized (lockListenerList) {
            final WeakReference<RecoveryCompletedListener> newWeakListener =
                new WeakReference<RecoveryCompletedListener>(newListener);
            
            recoveryListenerList.add(newWeakListener);
            
            // 肥大化を防ぐために、10の倍数の数に達した段階でリストを走査して、
            // 弱参照が消えているものをチェックしてそのエントリを削除する
            if (recoveryListenerList.size() % 10 == 0) {
                final ListIterator<WeakReference<RecoveryCompletedListener>> listIter =
                    recoveryListenerList.listIterator();
                
                while (listIter.hasNext()) {
                    final RecoveryCompletedListener listener = listIter.next().get();
                    if (listener == null) {
                        listIter.remove();
                    }
                }
            }
        }
    }
    
    /**
     * リカバリが正常完了した際に、setRecoveryCompletedListenerで登録された
     * リスナーに対して、setRecoveryCompleted関数を呼び出す。
     * @param serverId リカバリ処理が完了したサーバID
     */
    @GuardedBy("lockListenerList")
    public void notifyRecoveryCompleted(final int serverId) {
        synchronized (lockListenerList) {
            final ListIterator<WeakReference<RecoveryCompletedListener>> listIter =
                recoveryListenerList.listIterator();
            
            while (listIter.hasNext()) {
                final RecoveryCompletedListener listener = listIter.next().get();
                if (listener != null) {
                    listener.setRecoveryCompleted(serverId);
                } else {
                    listIter.remove();
                }
            }
        }
    }
}
