/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.collectiverun.composite;

import java.util.List;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.collectiverun.action.ModifyMaster;
import com.clustercontrol.collectiverun.bean.CommandParameterInfo;
import com.clustercontrol.collectiverun.bean.ItemInfo;
import com.clustercontrol.collectiverun.bean.TreeItem;
import com.clustercontrol.collectiverun.composite.action.MasterInfoModifyListener;
import com.clustercontrol.collectiverun.util.TreeItemUtil;
import com.clustercontrol.composite.StringVerifyListener;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.util.Messages;

/**
 * 一括制御階層ツリーコンポジット
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
public class CommandParameterComposite extends Composite {
    private Combo id = null;

    private Text option = null;
    
    private TreeItem selectItem = null;

    private TreeComposite composite = null;

    public CommandParameterComposite(Composite parent, int style, TreeComposite composite) {
        super(parent, style);
        this.composite = composite;
        this.selectItem = composite.getSelectItem();
        initialize();
    }

    private void initialize() {
        GridLayout layout = new GridLayout(10, true);
        this.setLayout(layout);
        layout.marginHeight = 5;
        layout.marginWidth = 5;

        GridData gridData = null;
        Label label = null;
        
        Group group = new Group(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 10;
        group.setLayoutData(gridData);
        group.setText(Messages.getString("parameter"));
        
        layout = new GridLayout(10, true);
        group.setLayout(layout);
        layout.marginHeight = 10;
        layout.marginWidth = 10;
        layout.verticalSpacing = 10;
        
        // パラメータID
        label = new Label(group, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 3;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("parameter.id") + " : ");
        
        this.id = new Combo(group, SWT.BORDER | SWT.READ_ONLY);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 7;
        this.id.setLayoutData(gridData);
        this.id.addModifyListener(new MasterInfoModifyListener(this.composite));
        List<String> list = 
        	TreeItemUtil.getParameterIdList(
        			TreeItemUtil.getTopTreeItem(selectItem));
        for(int i = 0; i < list.size(); i++){
        	this.id.add(list.get(i));
        }
        
        // オプション
        label = new Label(group, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 3;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("option") + " : ");
        
        this.option = new Text(group, SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 7;
        this.option.setLayoutData(gridData);
        this.option.addModifyListener(new MasterInfoModifyListener(this.composite));
        this.option.addVerifyListener(
        		new StringVerifyListener(DataRangeConstant.VARCHAR_64));
        
        // 適用ボタン
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 8;
        label.setLayoutData(gridData);
        
        Button buttonCreate = new Button(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 2;
        buttonCreate.setLayoutData(gridData);
        buttonCreate.setText(Messages.getString("apply"));
        buttonCreate.addSelectionListener(new SelectionAdapter() {
            @Override
			public void widgetSelected(SelectionEvent e) {
                ValidateResult result = validate();

                if (result != null) {
                    displayError(result);
                }
            }
        });
        
        setEnabled();
        
        update();
    }
    
    /**
     * コントロールの使用可/不可の設定処理
     * 
     */
    public void setEnabled() {
        if(this.composite.getCreateItem() == null){
        	this.id.setEnabled(false);
        }
        else{
        	this.id.setEnabled(true);
        }
    }

    /**
     * 更新処理
     * 
     */
    @Override
	public void update() {
    	if(selectItem instanceof TreeItem){
    		if(selectItem.getInfo() instanceof CommandParameterInfo){
    			CommandParameterInfo info = (CommandParameterInfo)selectItem.getInfo();
    			
    			//パラメータIDを設定
    			if(info.getId() != null){
    				this.id.setText(info.getId());
    			}
    			
    			//名前を設定
    			if(info.getPrefix() != null){
    				this.option.setText(info.getPrefix());
    			}
    		}
    	}
    	this.composite.setUpdateItem(null);
    }
    
	/**
	 * コマンドパラメータ情報作成
	 * 
	 * @return
	 */
	public ValidateResult createCommandParameterInfo() {
		ValidateResult result = null;
		
    	if(selectItem instanceof TreeItem){
    		if(selectItem.getInfo() instanceof CommandParameterInfo){
    			CommandParameterInfo info = (CommandParameterInfo)selectItem.getInfo();
    			CommandParameterInfo cloneInfo = (CommandParameterInfo)info.clone();
    			
    			ItemInfo parentInfo = selectItem.getParent().getInfo();
    			
		        //パラメータID取得
		        if (this.id.getText().length() > 0) {
		        	cloneInfo.setId(this.id.getText());
		        } else {
		            result = new ValidateResult();
		            result.setValid(false);
		            result.setID(Messages.getString("message.hinemos.1"));
		            result.setMessage(Messages.getString("message.collective.run.6"));		            return result;
		        }
		        
		        //パラメータID重複チェック
		        if (this.id.isEnabled() && 
		        		TreeItemUtil.findId(cloneInfo.getId(), selectItem.getParent())) {
		            result = new ValidateResult();
		            result.setValid(false);
		            result.setID(Messages.getString("message.hinemos.1"));
		            String args[] = { Messages.getString("parameter.id"), cloneInfo.getId() };
		            result.setMessage(Messages.getString("message.collective.run.16", args));
		            return result;
		        }
		        
		        //名前取得
		        cloneInfo.setName(TreeItemUtil.getParameterName(
		        		cloneInfo.getId(),
		        		TreeItemUtil.getTopTreeItem(selectItem)));
		        
		        //オプション取得
		        if (this.option.getText().length() > 0) {
		        	cloneInfo.setPrefix(this.option.getText());
		        }
		        else{
		        	cloneInfo.setPrefix("");
		        }
		        
		        ModifyMaster modify = new ModifyMaster();
                if(modify.update(cloneInfo, parentInfo)){
                	
                	selectItem.setInfo(cloneInfo);
                	
                	this.composite.setCreateItem(null);
                	this.composite.setUpdateItem(null);
                	this.composite.getTreeViewer().refresh(selectItem);
                }
                else{
                	//エラー時、エラーダイアログを表示する
                	Object[] args = { cloneInfo.getName() };
    				MessageDialog.openError(null, 
    						Messages.getString("message"), 
    						Messages.getString("message.collective.run.13", args));
                }
    		}
    	}
		
		return null;
	}
	
    /**
     * 入力値チェックを行います。
     * 
     * 必要に応じて、入力値チェックを実装して下さい。
     * 
     * @return ValidateResultオブジェクト
     */
    protected ValidateResult validate() {
        ValidateResult result = null;

        result = createCommandParameterInfo();
        if (result != null) {
            return result;
        }
        
        return null;
    }
	
    /**
     * エラー内容を通知します。
     * <p>
     * 
     * 警告メッセージボックスにて、クライアントに通知します。
     * 
     * @param result
     *            ValidateResultオブジェクト
     */
    protected void displayError(ValidateResult result) {
        MessageDialog.openWarning(
        		null, 
        		result.getID(), 
                result.getMessage());
    }
    
	/**
	 * @return
	 */
	public TreeItem getSelectItem() {
		return selectItem;
	}

	/**
	 * @param selectItem
	 */
	public void setSelectItem(TreeItem selectItem) {
		this.selectItem = selectItem;
	}
}