/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.composite;

import java.rmi.AccessException;
import java.util.ArrayList;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.MessageDialogWithToggle;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.bean.Property;
import com.clustercontrol.composite.CommonTableViewer;
import com.clustercontrol.jobmanagement.action.GetHistory;
import com.clustercontrol.jobmanagement.action.GetHistoryTableDefine;
import com.clustercontrol.jobmanagement.bean.HistoryTableDefine;
import com.clustercontrol.jobmanagement.bean.JobHistory;
import com.clustercontrol.jobmanagement.bean.JobHistoryList;
import com.clustercontrol.jobmanagement.composite.action.HistorySelectionChangedListener;
import com.clustercontrol.jobmanagement.composite.action.SessionJobDoubleClickListener;
import com.clustercontrol.jobmanagement.preference.JobManagementPreferencePage;
import com.clustercontrol.util.Messages;

/**
 * ジョブ[履歴]ビュー用のコンポジットクラスです。
 * 
 * @version 2.1.1
 * @since 1.0.0
 */
public class HistoryComposite extends Composite {
	/** テーブルビューア */
    protected CommonTableViewer m_viewer = null;
    /** ヘッダ用ラベル */
    protected Label m_labelType = null;
    /** 件数用ラベル */
    protected Label m_labelCount = null;
    /** セッションID */
    protected String m_sessionId = null;
    /** 所属ジョブユニットのジョブID */
    protected String m_jobunitId = null;
    /** ジョブID */
    protected String m_jobId = null;
    /** アクセス権関連通知フラグ */
    private boolean openAccessInformation = false;
    /** 表示数関連通知フラグ */
    private boolean openInformation = false;

    /**
     * コンストラクタ
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * 
     * @see org.eclipse.swt.SWT
     * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
     * @see #initialize()
     */
    public HistoryComposite(Composite parent, int style) {
        super(parent, style);
        initialize();
    }

    /**
     * コンポジットを配置します。
     */
    private void initialize() {
        GridLayout layout = new GridLayout(1, true);
        this.setLayout(layout);
        layout.marginHeight = 0;
        layout.marginWidth = 0;

        m_labelType = new Label(this, SWT.LEFT);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        m_labelType.setLayoutData(gridData);

        Table table = new Table(this, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.SINGLE);
        table.setHeaderVisible(true);
        table.setLinesVisible(true);

        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        table.setLayoutData(gridData);

        m_labelCount = new Label(this, SWT.RIGHT);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        m_labelCount.setLayoutData(gridData);

        m_viewer = new CommonTableViewer(table);
        m_viewer.createTableColumn(GetHistoryTableDefine.get(),
                HistoryTableDefine.SORT_COLUMN_INDEX,
                HistoryTableDefine.SORT_ORDER);

        m_viewer.addSelectionChangedListener(
        		new HistorySelectionChangedListener(this));
        
        m_viewer.addDoubleClickListener(
        		new SessionJobDoubleClickListener(this));
    }

    /**
     * テーブルビューアーを更新します。<BR>
     * ジョブ履歴一覧情報を取得し、共通テーブルビューアーにセットします。
     * <p>
	 * <ol>
	 * <li>監視管理のプレファレンスページより、ジョブ[履歴]ビューの表示履歴数を取得します。</li>
	 * <li>ジョブ履歴一覧情報を、表示履歴数分取得します。</li>
	 * <li>表示履歴数を超える場合、メッセージダイアログを表示します。</li>
	 * <li>共通テーブルビューアーにジョブ履歴一覧情報をセットします。</li>
	 * </ol>
     *  
     * @see com.clustercontrol.jobmanagement.action.GetHistory#getHistory(int)
     * @see #selectHistory(ArrayList)
     */
    @Override
	public void update() {
    	try {
    	JobHistoryList historyInfo = null;

        //ジョブ履歴情報取得
        GetHistory getHistory = new GetHistory();
        try {
        	int histories = ClusterControlPlugin.getDefault().getPreferenceStore().getInt(
        			JobManagementPreferencePage.P_HISTORY_MAX_HISTORIES);
            historyInfo = getHistory.getHistory(histories);
        } catch (AccessException e) {
        	if(!openAccessInformation){
        		openAccessInformation = true;
        		// アクセス権なしの場合、エラーダイアログを表示する
        		MessageDialog.openInformation(null, Messages.getString("message"),
	                    Messages.getString("message.accesscontrol.16"));
        	}
        }
        
        if (historyInfo == null) {
        	historyInfo = new JobHistoryList();
        	historyInfo.setList(new ArrayList<JobHistory>());
        }
        else if(historyInfo.getList() == null){
        	historyInfo.setList(new ArrayList<JobHistory>());
        }
        
        if(ClusterControlPlugin.getDefault().getPreferenceStore().getBoolean(
        		JobManagementPreferencePage.P_HISTORY_MESSAGE_FLG)){
        	if(historyInfo.getTotal() > historyInfo.getList().size()){ 
        		if(!openInformation){
        			openInformation = true;
        			// 最大表示件数を超える場合、エラーダイアログを表示する
        			MessageDialogWithToggle.openInformation(
        					null, 
        					Messages.getString("message"),
        					Messages.getString("message.job.33"), 
        					Messages.getString("message.will.not.be.displayed"), 
        					false, 
        					ClusterControlPlugin.getDefault().getPreferenceStore(), 
        					JobManagementPreferencePage.P_HISTORY_MESSAGE_FLG);
        			openInformation = false;
        		}
        	} 
        }
        
        ArrayList<Object> listInput = new ArrayList<Object>();
    	for (JobHistory history : historyInfo.getList()) {
    		ArrayList<Object> a = new ArrayList<Object>();
    		// HistoryTableDefineの順番でaddすること
    		a.add(history.getStatus());
    		a.add(history.getEndStatus());
    		a.add(history.getEndValue());
    		a.add(history.getSessionId());
    		a.add(history.getJobId());
    		a.add(history.getJobName());
    		a.add(history.getJobunitId());
    		a.add(history.getJobType());
    		a.add(history.getFacilityId());
    		a.add(history.getScope());
    		a.add(history.getScheduleDate());
    		a.add(history.getStartDate());
    		a.add(history.getEndDate());
    		a.add(history.getJobTriggerType());
    		a.add(history.getTriggerInfo());
    		listInput.add(a);
    	}
        m_viewer.setInput(listInput);
        
        selectHistory(listInput);

        // 表示件数をセット(最大件数以上に達しているか否かの分岐)
        m_labelType.setText("");
        Object[] args = null;
    	if(historyInfo.getTotal() > historyInfo.getList().size()){ 
    		args = new Object[]{ historyInfo.getList().size() };
    	} else {
    		args = new Object[]{ historyInfo.getTotal() };
    	}
        m_labelCount.setText(Messages.getString("records", args));
    	
    	} catch (Exception e) {
    		e.printStackTrace();
    	}
    }

    /**
     * テーブルビューアを更新します。<BR>
     * 引数で指定された条件に一致するジョブ履歴一覧情報を取得し、共通テーブルビューアーにセットします。
     * <p>
	 * <ol>
	 * <li>監視管理のプレファレンスページより、ジョブ[履歴]ビューの表示履歴数を取得します。</li>
	 * <li>引数で指定された条件に一致するジョブ履歴一覧情報を、表示履歴数分取得します。</li>
	 * <li>表示履歴数を超える場合、メッセージダイアログを表示します。</li>
	 * <li>共通テーブルビューアーにジョブ履歴一覧情報をセットします。</li>
	 * </ol>
     * 
     * @param condition 検索条件
     * 
     * @see com.clustercontrol.jobmanagement.action.GetHistory#getHistory(Property, int)
     * @see #selectHistory(ArrayList)
     */
    public void update(Property condition) {
    	JobHistoryList historyInfo = null;

        //ジョブ履歴情報取得
        GetHistory getHistory = new GetHistory();
        try {
        	int histories = ClusterControlPlugin.getDefault().getPreferenceStore().getInt(
        			JobManagementPreferencePage.P_HISTORY_MAX_HISTORIES);
            historyInfo = getHistory.getHistory(condition, histories);
        } catch (AccessException e) {
        	if(!openAccessInformation){
        		openAccessInformation = true;
        		// アクセス権なしの場合、エラーダイアログを表示する
        		MessageDialog.openInformation(null, Messages.getString("message"),
	                    Messages.getString("message.accesscontrol.16"));
        	}
        }
        
        if (historyInfo == null) {
        	historyInfo = new JobHistoryList();
        	historyInfo.setList(new ArrayList<JobHistory>());
        }
        else if(historyInfo.getList() == null){
        	historyInfo.setList(new ArrayList<JobHistory>());
        }
        
        if(ClusterControlPlugin.getDefault().getPreferenceStore().getBoolean(
        		JobManagementPreferencePage.P_HISTORY_MESSAGE_FLG)){
        	if(historyInfo.getTotal() > historyInfo.getList().size()){ 
        		if(!openInformation){
        			openInformation = true;
        			// 最大表示件数を超える場合、エラーダイアログを表示する
        			MessageDialogWithToggle.openInformation(
        					null, 
        					Messages.getString("message"),
        					Messages.getString("message.job.33"), 
        					Messages.getString("message.will.not.be.displayed"), 
        					false, 
        					ClusterControlPlugin.getDefault().getPreferenceStore(), 
        					JobManagementPreferencePage.P_HISTORY_MESSAGE_FLG);
        			openInformation = false;
        		}
        	} 
        }

    	ArrayList<Object> listInput = new ArrayList<Object>();
    	for (JobHistory history : historyInfo.getList()) {
    		ArrayList<Object> a = new ArrayList<Object>();
    		a.add(history.getStatus());
    		a.add(history.getEndStatus());
    		a.add(history.getEndValue());
    		a.add(history.getSessionId());
    		a.add(history.getJobId());
    		a.add(history.getJobunitId());
    		a.add(history.getJobName());
    		a.add(history.getJobType());
    		a.add(history.getFacilityId());
    		a.add(history.getScope());
    		a.add(history.getScheduleDate());
    		a.add(history.getStartDate());
    		a.add(history.getEndDate());
    		a.add(history.getJobTriggerType());
    		a.add(history.getTriggerInfo());
    		listInput.add(a);
    	}
        m_viewer.setInput(listInput);

        selectHistory(listInput);

        m_labelType.setText(Messages.getString("filtered.list"));
        Object[] args = { new Integer(historyInfo.getTotal()) };
        m_labelCount.setText(Messages.getString("filtered.records", args));
    }

    /**
     * ジョブ履歴の行を選択します。<BR>
     * 前回選択したセッションIDとジョブIDが一致する行を選択します。
     * 
     * @param historyInfo ジョブ履歴情報
     */
    public void selectHistory(ArrayList<Object> historyInfo) {
        if (m_sessionId != null && m_sessionId.length() > 0) {
            int index = -1;
            for (int i = 0; i < historyInfo.size(); i++) {
            	ArrayList line = (ArrayList) historyInfo.get(i);
                String sessionId = (String)line.get(HistoryTableDefine.SESSION_ID);
                
                if (m_sessionId.compareTo(sessionId) == 0) {
                    index = i;
                    break;
                }
            }
            if (index == -1) {
                m_sessionId = null;
                m_jobId = null;
            } else {
                m_viewer.setSelection(new StructuredSelection(historyInfo
                        .get(index)), true);
            }
        }
    }

    /**
     * このコンポジットが利用するテーブルビューアを返します。
     * 
     * @return テーブルビューア
     */
    public TableViewer getTableViewer() {
        return m_viewer;
    }

    /**
     * このコンポジットが利用するテーブルを返します。
     * 
     * @return テーブル
     */
    public Table getTable() {
        return m_viewer.getTable();
    }

    /**
     * セッションIDを返します。
     * 
     * @return セッションID
     */
    public String getSessionId() {
        return m_sessionId;
    }

    /**
     * セッションIDを設定します。
     * 
     * @param sessionId セッションID
     */
    public void setSessionId(String sessionId) {
        m_sessionId = sessionId;
    }

    /**
     * ジョブIDを返します。
     * 
     * @return ジョブID
     */
    public String getJobId() {
        return m_jobId;
    }

    /**
     * ジョブIDを設定します。
     * 
     * @param jobId ジョブID
     */
    public void setJobId(String jobId) {
        m_jobId = jobId;
    }

	/**
	 * 所属ジョブユニットのジョブIDを返します。
	 * 
	 * @return 所属ジョブユニットのジョブID
	 */
	public String getJobunitId() {
		return m_jobunitId;
	}

	/**
	 * 所属ジョブユニットのジョブIDを設定します。
	 * 
	 * @param jobunitId 所属ジョブユニットのジョブID
	 */
	public void setJobunitId(String jobunitId) {
		m_jobunitId = jobunitId;
	}




}