/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.composite.action;

import java.util.ArrayList;

import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;

import com.clustercontrol.bean.JobConstant;
import com.clustercontrol.jobmanagement.bean.JobTableDefine;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.composite.JobListComposite;
import com.clustercontrol.jobmanagement.util.JobUtil;
import com.clustercontrol.jobmanagement.view.JobListView;
import com.clustercontrol.util.LoginManager;

/**
 * ジョブ[一覧]ビューのテーブルビューア用のSelectionChangedListenerです。
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class JobListSelectionChangedListener implements ISelectionChangedListener {
    /** ジョブ[一覧]ビュー用のコンポジット */
	protected JobListComposite m_list;

    /**
     * コンストラクタ
     * 
     * @param list ジョブ[一覧]ビュー用のコンポジット
     */
    public JobListSelectionChangedListener(JobListComposite list) {
        m_list = list;
    }

    /**
     * 選択変更時に呼び出されます。<BR>
     * ジョブ[一覧]ビューのテーブルビューアを選択した際に、選択した行の内容でビューのアクションの有効・無効を設定します。
     * <P>
     * <ol>
	 * <li>選択変更イベントから選択行を取得し、選択行からジョブIDを取得します。</li>
	 * <li>ジョブ[一覧]ビュー用のコンポジットからジョブツリーアイテムを取得します。</li>
	 * <li>取得したジョブツリーアイテムから、ジョブIDが一致するジョブツリーアイテムを取得します。</li>
	 * <li>ジョブ[一覧]ビュー用のコンポジットに、ジョブIDが一致するジョブツリーアイテムを設定します。</li>
	 * <li>ジョブ[一覧]ビューのアクションの有効・無効を設定します。</li>
	 * </ol>
	 * 
     * @param event 選択変更イベント
     * 
     * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
     */
    public void selectionChanged(SelectionChangedEvent event) {
        JobTreeItem selectJobTreeItem = null;

        //ジョブ[登録]ビューのインスタンスを取得
        IWorkbenchPage page = PlatformUI.getWorkbench()
                .getActiveWorkbenchWindow().getActivePage();
        IViewPart viewPart = page.findView(JobListView.ID);
        

        //選択アイテムを取得
        StructuredSelection selection = (StructuredSelection) event.getSelection();
        
        if ( viewPart != null && selection != null ){
	        ArrayList item = (ArrayList) selection.getFirstElement();
	        JobListView view = (JobListView) viewPart.getAdapter(JobListView.class);
	        
	        if (item instanceof ArrayList) {
	            String jobId = (String) item.get(JobTableDefine.JOB_ID);
	
	            if (m_list.getJobTreeItem() instanceof JobTreeItem) {
	                JobTreeItem[] items = m_list.getJobTreeItem().getChildrenArray();
	
	                for (int i = 0; i < items.length; i++) {
	                    if (jobId.equals(items[i].getData().getId())) {
	                        selectJobTreeItem = items[i];
	                        break;
	                    }
	                }
	            }
	        }
	
	        if (selectJobTreeItem instanceof JobTreeItem) {
	            //選択ツリーアイテムを設定
	            m_list.setSelectJobTreeItem(selectJobTreeItem);	            
	
            	boolean isReferableJobunit = false;
            	
            	// ログインユーザで参照可能なジョブユニットかどうかチェックする
	            if (selectJobTreeItem.getData().getType() == JobConstant.TYPE_JOBUNIT) {

	            	// 参照できない場合、各種ボタン操作を無効化する
	            	if (JobUtil.isReferableJobunit(selectJobTreeItem, LoginManager.getContextManager().getUserId())) {
	            		view.setEnabledActionAll(true);
	            		view.setEnabledAction(selectJobTreeItem.getData().getType(), selection);
	            	} else {
	            		view.setEnabledActionAll(false);
	            	}
	            	
	            } else {
            		//ビューのアクションの有効/無効を設定
    	            view.setEnabledAction(selectJobTreeItem.getData().getType(), selection);
            	}
	        } else {
	            //選択ツリーアイテムを設定
	            m_list.setSelectJobTreeItem(null);

	            //ビューのアクションを全て無効に設定
	            view.setEnabledAction(-9, selection);
	        }
        } 
    }
}