/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.logtransfer.composite;

import java.util.ArrayList;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.composite.CommonTableViewer;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.logtransfer.action.GetLogTransferFileListTableDefine;
import com.clustercontrol.logtransfer.bean.LogTransferFileInfo;
import com.clustercontrol.logtransfer.bean.LogTransferFileTabelDefine;
import com.clustercontrol.logtransfer.bean.LogTransferInfo;
import com.clustercontrol.logtransfer.composite.action.LogTransferFileSelectionChangedListener;
import com.clustercontrol.logtransfer.dialog.LogTransferFileDialog;
import com.clustercontrol.util.Messages;

/**
 * 転送対象ログファイル一覧コンポジットクラス<BR>
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
public class LogTransferFileListComposite extends Composite {
	
	/** テーブルビューア */
	protected CommonTableViewer m_tableViewer = null;
	
	/** 追加ボタン */
	protected Button m_buttonAdd = null;
	
	/** 変更ボタン */
	protected Button m_buttonModify = null;
	
	/** 削除ボタン */
	protected Button m_buttonDelete = null;
	
	protected Shell m_shell = null;
	
	protected ArrayList m_selectItem = null;
	
	
	/**
	 * コンストラクタ
	 * 
	 * @param parent
	 * @param style
	 * @since 1.0.0
	 */
	public LogTransferFileListComposite(Composite parent, int style) {
		super(parent, style);
		initialize();
		m_shell = this.getShell();
	}
	
	/**
	 * 初期化処理
	 */
	private void initialize() {
		
		// 変数として利用されるラベル
		Label label = null;
		
		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		layout.numColumns = 15;
		this.setLayout(layout);
		
		/*
		 * 転送対象ログファイル一覧
		 */
		// ラベル
		label = new Label(this, SWT.NONE);
		label.setText(Messages.getString("logtransfer.file.list") + " : ");
		GridData gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		
		// テーブル
		Table table = new Table(this, SWT.BORDER | SWT.H_SCROLL
				| SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.SINGLE);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.minimumHeight = 200;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		table.setLayoutData(gridData);
		
		// ラベル
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 9;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		
		/*
		 * 操作ボタン
		 */
		// 追加
		m_buttonAdd = new Button(this, SWT.NONE);
		m_buttonAdd.setText(Messages.getString("add"));
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_buttonAdd.setLayoutData(gridData);
		m_buttonAdd.addSelectionListener(new SelectionAdapter() {
			@Override
			@SuppressWarnings("unchecked")
			public void widgetSelected(SelectionEvent e) {
				LogTransferFileDialog dialog = new LogTransferFileDialog(m_shell);
				if (dialog.open() == IDialogConstants.OK_ID) {
					ArrayList info = dialog.getInputData();
					ArrayList list = (ArrayList) m_tableViewer.getInput();
					if (list == null) {
						list = new ArrayList();
					}
					list.add(info);
					m_tableViewer.setInput(list);
				}
			}
		});
		
		// 変更
		m_buttonModify = new Button(this, SWT.NONE);
		m_buttonModify.setText(Messages.getString("modify"));
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_buttonModify.setLayoutData(gridData);
		m_buttonModify.addSelectionListener(new SelectionAdapter() {
			@Override
			@SuppressWarnings("unchecked")
			public void widgetSelected(SelectionEvent e) {
				LogTransferFileDialog dialog = new LogTransferFileDialog(m_shell);
				if (m_selectItem instanceof ArrayList) {
					dialog.setInputData(m_selectItem);
					if (dialog.open() == IDialogConstants.OK_ID) {
						ArrayList info = dialog.getInputData();
						ArrayList list = (ArrayList) m_tableViewer.getInput();
						list.remove(m_selectItem);
						list.add(info);
						m_selectItem = null;
						m_tableViewer.setInput(list);
					}
				} else {
                	MessageDialog.openWarning(
                			null, 
                			Messages.getString("warning"), 
                			Messages.getString("message.logtransfer.11"));
				}
			}
		});
		
		// 削除
		m_buttonDelete = new Button(this, SWT.NONE);
		m_buttonDelete.setText(Messages.getString("delete"));
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_buttonDelete.setLayoutData(gridData);
		m_buttonDelete.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				if (m_selectItem instanceof ArrayList) {
					ArrayList list = (ArrayList) m_tableViewer.getInput();
					String filePath =  (String)m_selectItem.get(LogTransferFileTabelDefine.FILE_PATH);
					
					String[] args = { filePath };
					if (MessageDialog.openConfirm(
		        			null, 
		        			Messages.getString("confirmed"), 
		        			Messages.getString("message.logtransfer.14", args))) {
						
						list.remove(m_selectItem);
						m_selectItem = null;
						m_tableViewer.setInput(list);
					}
					
				} else {
                	MessageDialog.openWarning(
                			null, 
                			Messages.getString("warning"), 
                			Messages.getString("message.logtransfer.11"));
				}
			}
		});
		
		// テーブルビューアの作成
		m_tableViewer = new CommonTableViewer(table);
		m_tableViewer.createTableColumn(GetLogTransferFileListTableDefine.get(),
				LogTransferFileTabelDefine.SORT_COLUMN_INDEX,
				LogTransferFileTabelDefine.SORT_ORDER);
		m_tableViewer.addSelectionChangedListener(
				new LogTransferFileSelectionChangedListener(this));
	}
	
	/**
	 * 各項目に入力値を設定します。
	 * 
	 * @param info 設定値として用いるログ転送設定情報
	 */
	@SuppressWarnings("unchecked")
	public void setInputData(LogTransferInfo info) {
		
		if(info != null){
			
			ArrayList tableData = new ArrayList();
			
			ArrayList fileList = info.getFileInfo();
			for (int index = 0; index < fileList.size(); index++) {
				
				LogTransferFileInfo fileInfo = (LogTransferFileInfo) fileList.get(index);
				ArrayList tableLineData = new ArrayList();
				
				tableLineData.add(fileInfo.getFilePath());
				tableLineData.add(Integer.valueOf(fileInfo.getRunInterval()));
				tableLineData.add(Boolean.valueOf(YesNoConstant.typeToBoolean(fileInfo.getExistenceFlg())));
				tableLineData.add(Integer.valueOf(fileInfo.getValid()));
				
				tableData.add(tableLineData);
			}
			
			// テーブル更新
			this.m_tableViewer.setInput(tableData);
		}
	}
	
	/**
	 * 入力値を用いて監視情報を生成します。
	 * 
	 * @return 検証結果
	 */
	public ValidateResult createInputData(LogTransferInfo info) {
		
		ArrayList tableData = (ArrayList) m_tableViewer.getInput();
		
		if(tableData != null && tableData.size() > 0){
			
			ArrayList<LogTransferFileInfo> list = new ArrayList<LogTransferFileInfo>();
			ArrayList<String> filePathList = new ArrayList<String>();
			String transferId = info.getTransferId();
			
			for(int index=0; index<tableData.size(); index++){
				
				ArrayList tableLineData = (ArrayList) tableData.get(index);
				
				String filePath = (String)tableLineData.get(LogTransferFileTabelDefine.FILE_PATH);
				
				// 転送対象ログファイル 重複チェック
				if(!filePathList.contains(filePath)){
					
					filePathList.add(filePath);
					
					LogTransferFileInfo fileInfo = new LogTransferFileInfo();
					fileInfo.setTransferId(transferId);
					fileInfo.setFilePath(filePath);
					fileInfo.setRunInterval(((Integer)tableLineData.get(LogTransferFileTabelDefine.RUN_INTERVAL)).intValue());
					fileInfo.setExistenceFlg(YesNoConstant.booleanToType(((Boolean)tableLineData.get(LogTransferFileTabelDefine.FILE_EXISTENCE_CHECK)).booleanValue()));
					fileInfo.setValid(((Integer)tableLineData.get(LogTransferFileTabelDefine.VALID_FLG)).intValue());
					
					list.add(fileInfo);
				}
				else{
					String[] args = { filePath };
					return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.logtransfer.15", args));
				}
			}
			// 転送対象ログファイルセット
			info.setFileInfo(list);
		}
		else{
			return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.logtransfer.11"));
		}
		return null;
	}
	
	/**
	 * 無効な入力値の情報を設定します
	 *  
	 */
	protected ValidateResult setValidateResult(String id, String message) {
		
		ValidateResult validateResult = new ValidateResult();
		validateResult.setValid(false);
		validateResult.setID(id);
		validateResult.setMessage(message);
		
		return validateResult;
	}
	
	/**
	 * 選択アイテムを取得
	 * 
	 * @return
	 */
	public ArrayList getSelectItem() {
		return m_selectItem;
	}
	
	/**
	 * 選択アイテムを設定
	 * 
	 * @param selectItem
	 */
	public void setSelectItem(ArrayList selectItem) {
		m_selectItem = selectItem;
	}
}