/*
 
Copyright (C) 2008 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.mail.dialog;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.composite.StringVerifyListener;
import com.clustercontrol.composite.StyledTextComposite;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.notify.mail.action.AddMailTemplate;
import com.clustercontrol.notify.mail.action.GetMailTemplate;
import com.clustercontrol.notify.mail.action.ModifyMailTemplate;
import com.clustercontrol.notify.bean.NotifySubstitutionParameterConstant;
import com.clustercontrol.notify.mail.ejb.entity.MailTemplateInfoData;
import com.clustercontrol.util.Messages;

/**
 * メールテンプレートID作成・変更ダイアログクラス<BR>
 * 
 * @version 2.4.0
 * @since 2.4.0
 */
public class MailTemplateCreateDialog extends CommonDialog {
	
	/** カラム数 */
	public static final int WIDTH	 = 15;
	
	/** カラム数（ラベル）。 */
	public static final int WIDTH_LABEL = 4;
	
	/** カラム数（テキスト）。 */
	public static final int WIDTH_TEXT = 10;
	
	
		
	
	/** 入力値を保持するオブジェクト。 */
	private MailTemplateInfoData inputData = null;
	
	/** 入力値の正当性を保持するオブジェクト。 */
	private ValidateResult validateResult = null;
	
	/** 参照フラグ。 */
	private boolean referenceFlg = false;
	
	/** 変更対象のメールテンプレートID。 */
	private String mailTemplateId = null;
	
	/** メールテンプレートID テキストボックス。 */
	private Text textMailTemplateId = null;
	
	/** 説明 テキストボックス。 */
	private Text textDescription = null;
	
	/** 件名 スタイルテキストコンポジット。 */
	private StyledTextComposite textSubject = null;
	
	/** 本文 スタイルテキストコンポジット。 */
	private StyledTextComposite textBody = null;	
	
	/**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 */
	public MailTemplateCreateDialog(Shell parent) {
		super(parent);
	}
	
	/**
	 * 変更用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 * @param mailTemplateId 変更するメールテンプレート情報のメールテンプレートID
	 * @param referenceFlg 参照フラグ（参照のみで変更不可の場合、<code> true </code>）
	 */
	public MailTemplateCreateDialog(Shell parent, String mailTemplateId, boolean referenceFlg) {
		super(parent);
		
		this.mailTemplateId = mailTemplateId;
		this.referenceFlg = referenceFlg;
	}
	
	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.notify.mail.action.GetMailTemplate#getMailTemplate(String)
	 * @see #setInputData(MailTemplateInfoData)
	 */
    @Override
	protected void customizeDialog(Composite parent) {
        Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages.getString("dialog.mail.template.create.modify"));

        // 変数として利用されるラベル
        Label label = null;
        // 変数として利用されるグリッドデータ
        GridData gridData = null;

        // レイアウト
        GridLayout layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 15;
        parent.setLayout(layout);
        
        /*
         * メールテンプレートID
         */
        // ラベル
        label = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_LABEL;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("mail.template.id") + " : ");
        // テキスト
        this.textMailTemplateId = new Text(parent, SWT.BORDER | SWT.LEFT);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_TEXT;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.textMailTemplateId.setLayoutData(gridData);
        if(this.mailTemplateId != null && !"".equals(this.mailTemplateId.trim())){
        	this.textMailTemplateId.setEnabled(false);
        }
        this.textMailTemplateId.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_64));

        /*
         * 説明
         */
        // ラベル
        label = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_LABEL;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("description") + " : ");
        // テキスト
        this.textDescription = new Text(parent, SWT.BORDER | SWT.LEFT);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_TEXT;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.textDescription.setLayoutData(gridData);
        this.textDescription.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_256));
        
        // 空白
        label = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        /*
         * メールテンプレート設定
         */
        label = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("mail.template.setting") + " : ");

        /*
         * メールテンプレート設定グループ 
         */
        Group groupMailTemplate = new Group(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 15;
        groupMailTemplate.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupMailTemplate.setLayoutData(gridData);
        
        /*
         * 件名
         */
        // ラベル
        label = new Label(groupMailTemplate, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 2;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("subject") + " : ");
        // テキスト
        this.textSubject = new StyledTextComposite(groupMailTemplate, SWT.BORDER | SWT.LEFT);
        gridData = new GridData();
        gridData.horizontalSpan = 13;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.textSubject.setLayoutData(gridData);
        this.textSubject.setColor(new Color(groupMailTemplate.getDisplay(), new RGB(0, 0, 255)));
        this.textSubject.setKeywords(NotifySubstitutionParameterConstant.getParameterList());
        this.textSubject.setInputUpper(DataRangeConstant.VARCHAR_256);
      
        /*
         * 本文
         */
        // ラベル
        label = new Label(groupMailTemplate, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("response.body") + " : ");
        // テキスト
        this.textBody = new StyledTextComposite(groupMailTemplate, SWT.MULTI | SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL | SWT.LEFT);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalSpan = 50;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        this.textBody.setLayoutData(gridData);
        this.textBody.setColor(new Color(groupMailTemplate.getDisplay(), new RGB(0, 0, 255)));
        this.textBody.setKeywords(NotifySubstitutionParameterConstant.getParameterList());
               
        // ラインを引く
        Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        line.setLayoutData(gridData);
        
        // サイズを最適化
        // グリッドレイアウトを用いた場合、こうしないと横幅が画面いっぱいになります。
        shell.pack();
        shell.setSize(new Point(550, shell.getSize().y));
        
        // 画面中央に
        Display display = shell.getDisplay();
        shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
                (display.getBounds().height - shell.getSize().y) / 2);

        // メールテンプレートIDが指定されている場合、その情報を初期表示する。
        MailTemplateInfoData info = null; 
        if(this.mailTemplateId != null){
        	info = new GetMailTemplate().getMailTemplate(this.mailTemplateId);        	
        }
        else{
        	info = new MailTemplateInfoData();
        }
        this.setInputData(info);
    }
    
	/**
	 * 入力値を保持したメールテンプレート情報を返します。
	 * 
	 * @return メールテンプレート情報
	 */
    public MailTemplateInfoData getInputData() {
        return this.inputData;
    }

	/**
	 * 引数で指定されたメールテンプレート情報の値を、各項目に設定します。
	 * 
	 * @param mailTemplate 設定値として用いるメールテンプレート情報
	 */
    protected void setInputData(MailTemplateInfoData mailTemplate) {

        this.inputData = mailTemplate;

        // 各項目に反映
        // メールテンプレートID
        if (mailTemplate.getMailTemplateId() != null) {
            this.textMailTemplateId.setText(mailTemplate.getMailTemplateId());
        }
        // 説明
        if (mailTemplate.getDescription() != null) {
            this.textDescription.setText(mailTemplate.getDescription());
        }
        // 件名
        if (mailTemplate.getSubject() != null) {
            this.textSubject.setText(mailTemplate.getSubject());
        }
        // 本文
        if (mailTemplate.getBody() != null) {
            this.textBody.setText(mailTemplate.getBody());
        }

        // 入力制御
        if(this.referenceFlg){
        	this.textMailTemplateId.setEnabled(false);
        	this.textDescription.setEnabled(false);
        	this.textSubject.setEnabled(false);
        	this.textBody.setEnabled(false);
        }

    }

	/**
	 * 入力値を設定したメールテンプレート情報を返します。<BR>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return メールテンプレート情報
	 */
    protected MailTemplateInfoData createInputData() {
    	MailTemplateInfoData info = new MailTemplateInfoData();

        if (this.textMailTemplateId.getText() != null
                && !"".equals((this.textMailTemplateId.getText()).trim())) {
        	info.setMailTemplateId(this.textMailTemplateId.getText());
        } else {
            this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.mail.template.1"));
            return null;
        }
        if (this.textDescription.getText() != null
                && !"".equals((this.textDescription.getText()).trim())) {
        	info.setDescription(this.textDescription.getText());
        }
        if (this.textSubject.getText() != null
                && !"".equals((this.textSubject.getText()).trim())) {
        	info.setSubject(this.textSubject.getText());
        }
        if (this.textBody.getText() != null
                && !"".equals((this.textBody.getText()).trim())) {
        	info.setBody(this.textBody.getText());
        }    
        
        return info;
    }

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
    @Override
	protected ValidateResult validate() {
        // 入力値生成
        this.inputData = this.createInputData();

        if (this.inputData != null) {
            return super.validate();
        } else {
            return validateResult;
        }
    }
    
    /**
     * 入力値をマネージャに登録します。
     * 
     * @return true：正常、false：異常
     * 
     * @see com.clustercontrol.dialog.CommonDialog#action()
     */
    @Override
	protected boolean action() {
    	boolean result = false;
    	
    	MailTemplateInfoData info = this.getInputData();
    	if(info != null){
            if(this.mailTemplateId == null){
            	// 作成の場合
            	result = new AddMailTemplate().add(info);
            }
            else{
            	// 変更の場合
            	result = new ModifyMailTemplate().modify(info);
            }
    	}
    	
    	return result;
    }

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
    @Override
	protected String getOkButtonText() {
        return Messages.getString("ok");
    }

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
    @Override
	protected String getCancelButtonText() {
        return Messages.getString("cancel");
    }
  
	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 */
    private void setValidateResult(String id, String message) {

        this.validateResult = new ValidateResult();
        this.validateResult.setValid(false);
        this.validateResult.setID(id);
        this.validateResult.setMessage(message);
    }
    
	/**
	 * ボタンを生成します。<BR>
	 * 参照フラグが<code> true </code>の場合は閉じるボタンを生成し、<code> false </code>の場合は、デフォルトのボタンを生成します。
	 * 
	 * @param parent ボタンバーコンポジット
	 * 
	 * @see #createButtonsForButtonBar(Composite)
	 */
    @Override
	protected void createButtonsForButtonBar(Composite parent) {
    	
        if(!this.referenceFlg){
        	super.createButtonsForButtonBar(parent);
        }
        else{
        	// 閉じるボタン
            this.createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CLOSE_LABEL, false);
        }
    }
}