/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.composite;

import java.awt.Color;
import java.awt.Frame;
import java.util.ArrayList;
import java.util.List;
import org.eclipse.swt.SWT;
import org.eclipse.swt.awt.SWT_AWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.StandardXYItemRenderer;
import org.jfree.chart.renderer.xy.XYAreaRenderer;
import org.jfree.data.time.Second;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.data.xy.XYDataset;

import com.clustercontrol.performance.bean.GraphConstant;
import com.clustercontrol.performance.bean.GraphProperty;
import com.clustercontrol.performance.util.Messages;
import com.clustercontrol.performance.util.RealtimeCollectorInfo;
import com.clustercontrol.performance.util.RealtimeCollectorItemInfo;
import com.clustercontrol.performance.util.CollectorItemCodeFactory;

/**
 * リアルタイムグラフ表示を行うViewを構成するコンポジットクラス グラフを配置する土台となるcomposite
 * 
 * @version 1.0
 * @since 1.0
 */
public class RealtimeGraphComposite extends Composite {
	// グラフを配置するcomposite.グラフ数分だけ生成される.
	private Composite graphComposite[] = null;

	public TimeSeries m_timeseries[];

	public ChartPanel chartPanel = null;

	private Label infoLabel;

	private boolean isCollect = false;

	private RealtimeCollectorInfo collectorInfo = null;

	private RealtimeCollectorItemInfo collectorItemInfo = null;

	// 後で、グラフ表示後にx軸のスケールを変更できる仕様にする.
	// private boolean isXRangeAuto = true;

	// グラフの表示プロット数をグラフを再作成せず変更できるようにするため.
	private ValueAxis valueaxis = null;

	private int graphPlotNum = 0;

	private int interval = 0;

	private JFreeChart jfreechart[] = null; // c -> C

	private Frame graphFrame = null;

	/**
	 * インスタンスを返します。
	 * 
	 * @param parent
	 *            親のコンポジット
	 * @param style
	 *            スタイル e.g. SWT.EMBEDDED
	 */
	public RealtimeGraphComposite(Composite parent, int style) {
		super(parent, style);
		initialize();
	}

	// ----- instance メソッド ----- //

	private void initialize() {
		graphComposite = new Composite[4];

		jfreechart = new JFreeChart[4];
	}

	/**
	 * graphComposite上にグラフを作成します。
	 * 
	 * @param itemCode
	 * @param subScopeNum
	 * @param displayType
	 * @param interval
	 */
	public void addGraph(int graphId, String itemCode, String displayName,
			int subScopeNum, String displayType, String displayForm, int interval,
			boolean isXRangeAuto, int graphPlotNum) {

		// 1インスタンスで複数グラフを生成するので変数をグラフ作成単位で初期化.
		isCollect = false;

		this.graphPlotNum = graphPlotNum;
		this.interval = interval;

		if (!displayType.equals("NonDisplay")) {
			createGraphComposite(graphId, itemCode, displayName, subScopeNum,
					displayType, displayForm, interval, isXRangeAuto, graphPlotNum);
		}
		// NonDisplayではisCollect = falseのまま.
	}

	private void createGraphComposite(int graphId, String itemCode,
			String displayName, int subScopeNum, String displayType, String displayForm,
			int interval, boolean isXRangeAuto, int graphPlotNum) {

		if (!collectorItemInfo.hasDevice(graphId)) {
			graphComposite[graphId] = new Composite(this, SWT.NULL);

			infoLabel = new Label(graphComposite[graphId], SWT.CENTER); // SWT.CENTERだめ.
			infoLabel.setText(Messages.getString("DEVICE") // "デバイス"
					+ " \"" + collectorItemInfo.getDisplayName(graphId) + "\" "
					+ Messages.getString("NOT_EXIST")); // "が存在しないので表示されません"
		} else if (displayType.equals("SubScope") && subScopeNum == 0) {
			// 「サブスコープ表示かつ、サブスコープをもたない」場合は、グラフ描画せず
			// 「サブスコープは存在しません」と表示する。(あとで実装)

			graphComposite[graphId] = new Composite(this, SWT.NULL);

			infoLabel = new Label(graphComposite[graphId], SWT.CENTER); // SWT.CENTERだめ.
			infoLabel.setText(Messages.getString("NOT_EXIST_SUBSCOPE")); // "サブスコープが存在しないので表示されません."
		} else if (displayType.equals("SubScope") && subScopeNum > 10) {

			// グラフ表示しない。(現行仕様)
			// ラベルで「スコープ数が10を超えているので表示しない」旨を表示する。
			// 今は仮決めで10を固定で条件指定している。あとで変数(定数)の持ち方を決める。

			graphComposite[graphId] = new Composite(this, SWT.NULL);

			infoLabel = new Label(graphComposite[graphId], SWT.NULL);
			infoLabel.setText(Messages
					.getString("MSG_SUB_SCOPE_COUNTS_OVER_10")); // "サブスコープが10を超えて存在するので表示されません."

		} else {
			graphComposite[graphId] = new Composite(this, SWT.EMBEDDED);

			setJFreeChart(graphId, interval, displayType, displayForm, itemCode, displayName,
					isXRangeAuto, graphPlotNum);

			// この変換を行なっているので、グラフ画面が自動的に表示(再描画)されない?
			graphFrame = SWT_AWT.new_Frame(graphComposite[graphId]);
			graphFrame.add(chartPanel);

			isCollect = true;
		}

		GridData graphGridData = new GridData();
		graphGridData.horizontalAlignment = GridData.FILL;
		graphGridData.verticalAlignment = GridData.FILL;
		graphGridData.grabExcessHorizontalSpace = true;
		graphGridData.grabExcessVerticalSpace = true;

		GridLayout graphLayout = new GridLayout(1, true);
		graphComposite[graphId].setLayout(graphLayout);
		graphLayout.marginHeight = 0;
		graphLayout.marginWidth = 0;

		graphComposite[graphId].setLayoutData(graphGridData); // レイアウトの指定を反映させる。
	}

	/**
	 * グラフ表示データ取得用基本情報生成処理
	 * <p>
	 * どのような情報をグラフ表示するかの情報を整理してまとめる処理。
	 * 現行仕様ではスレッドの制御までしているが、この点は見直す余地あり。...変更して分離した。
	 * </p>
	 * 
	 * @param interval
	 *            グラフ表示間隔(秒)
	 * @param displayType
	 *            グラフ表示種別
	 * @param displayForm
	 *            グラフ形式
	 * @param itemCode
	 *            グラフ表示情報コード
	 * @return なし
	 * @see なし
	 */
	private void setJFreeChart(int graphId, int interval, String displayType, String displayForm,
			String itemCode, String displayName, boolean isXRangeAuto,
			int graphPlotNum) {
		// 本処理は、以下のチェックを通ったことを前堤として処理する。
		// (1) サブスコープ代表値では、サブスコープ数が1以上であること。
		// (2) スコープ詳細では、itemCode数が1以上であること。
		// (3) 表示するグラフ(線)数が規定数以下(2次開発段階では10)であること。

		// グラフの内訳の凡例の表示名称を保持するために用いる。
		List graphObjectList = new ArrayList();

		if (displayType.equals("SelectedScope")) {
			// スコープ代表値では凡例は用いないが、ダミーでitemCodeをセットしておく。
			graphObjectList.add(0, itemCode);
		} else if (displayType.equals("SubScope")) {

			// facilityIdが10(2次開発での仕様)を超えるか否かのチェックは
			// ここでは行なわない。画面入力側で制御する。
			graphObjectList = collectorInfo.getSubScopeFacilityName();
		} else if (displayType.equals("Detail")) {
			// 内訳表示に必要な収集項目コードを取得
			graphObjectList = CollectorItemCodeFactory.getSubItemCode(itemCode);
		}

		// timeseries(グラフの線毎に必要)を入れる入れ物(グラフの表示毎に用意)を用意する。
		TimeSeriesCollection timeSeriesCollection = new TimeSeriesCollection();
		// グラフ表示用にTimeSeriesをグラフ線数だけ新規作成する
		int graphNum = 0;
		if (graphObjectList.size() > 10) {
			// サブスコープ数が10を超える場合は、facilityIDの昇順で先頭の10個をグラフ表示対象として登録する.
			// 但し、現行仕様では、10を超えるときはグラフを表示させないため、ここの段階で10を超えることはない.
			graphNum = 10;
		} else {
			graphNum = graphObjectList.size();
		}

		m_timeseries = new TimeSeries[graphNum];
		String displayNoteName = null;

		// サブスコープ代表値表示では、後から追加したものがグラフ表示(描画)の手前側に表示されることに留意のこと。
		if (graphNum > 0) {
			for (int i = 0; i < graphNum; i++) {
				if (displayType.equals("SubScope")) {
					displayNoteName = graphObjectList.get(i).toString();
				} else if (displayType.equals("Detail")) {
					// itemCodeを名称に置き換える。
					displayNoteName = CollectorItemCodeFactory
							.getFullItemName(graphObjectList.get(i).toString(), displayName);
				} else if (displayType.equals("SelectedScope")) {
					// スコープ代表値では凡例は用いないが、ダミーでitemCodeをセットしておく。
					displayNoteName = graphObjectList.get(i).toString();
				}
				m_timeseries[i] = new TimeSeries(displayNoteName, Second.class);

				// グラフの最大プロット数を指定
				m_timeseries[i]
						.setMaximumItemCount(GraphConstant.REALTIME_GRAPH_MAX_PLOT);

				timeSeriesCollection.addSeries(m_timeseries[i]);
			}
		}
		chartPanel = new ChartPanel(createChart(graphId, itemCode, displayName,
				displayType, displayForm, timeSeriesCollection, interval, isXRangeAuto,
				graphPlotNum));
	}

	// Chart作成のための共通処理(スコープ平均値、詳細(収集項目別)、詳細(サブスコープ別)
	private JFreeChart createChart(int graphId, String itemCode,
			String displayName, String displayType, String displayForm, XYDataset xydataset,
			int interval, boolean isXRangeAuto, int graphPlotNum) {
		// グラフの見出しの表示の決定
		String displayTypeName = Messages.getString("NULL"); // グラフの見出の指定
		boolean displayNote = false; // 汎例(各グラフ表示データの説明)
		if (displayType.equals("SelectedScope")) {
			displayTypeName = Messages.getString("SCOPE_REPRESENTING_VALUE");
		} else if (displayType.equals("Detail")) {
			displayTypeName = Messages
					.getString("DETAIL_SCOPE_REPRESENTING_VALUE");
			displayNote = true;
		} else if (displayType.equals("SubScope")) {
			displayTypeName = Messages.getString("SUB_SCOPE_REP_VAL");
			// スコープツリーの改造までは汎例表示で暫定対処する。
			displayNote = true;
		}

		String displayItemName = null;

		displayItemName = CollectorItemCodeFactory.getFullItemName(itemCode, displayName)
				+ " " + displayTypeName;

		// グラフ表示領域(chart)の作成。
		jfreechart[graphId] = ChartFactory.createTimeSeriesChart(displayItemName,
				Messages.getString("TIME"), CollectorItemCodeFactory
						.getMeasure(itemCode), xydataset, displayNote, // 汎例の表示
				true, false);

		XYPlot xyplot = jfreechart[graphId].getXYPlot();

		// ここをprivate化して外部から操作できるようにすれば表示中のグラフを動的に操作できる?
		valueaxis = xyplot.getDomainAxis(); // 何のためにgetしているのか明記する.
		valueaxis.setAutoRange(true); // 一旦autoRangeにしている.

		if (!isXRangeAuto) {
			valueaxis.setFixedAutoRange(interval * 1000D * graphPlotNum);// x軸の表示期間（mS）
																			// 12000msec=12sec
		}
		valueaxis = xyplot.getRangeAxis();
		
		int graphType = GraphProperty.INVISIBLE;
		if (displayType.equals("NonDisplay")) {
			graphType = GraphProperty.INVISIBLE;
		} else if (displayType.equals("SelectedScope")) {
			graphType = GraphProperty.TYPE1;
		} else if (displayType.equals("Detail")) {
			graphType = GraphProperty.TYPE2;
		} else if (displayType.equals("SubScope")) {
			graphType = GraphProperty.TYPE3;
		}
		
		int graphForm = GraphProperty.FORM1;
		if (displayForm.equals("FormLine")) {
			graphForm = GraphProperty.FORM1;
		} else if (displayForm.equals("FormStackArea")) {
			graphForm = GraphProperty.FORM2;
		}
		
		// 百分率の場合は0-100。しかし、(積み上げ面かつサブスコープ)の場合は除く。
		// それ以外は下限が0で、上限はオート
		if (CollectorItemCodeFactory.isRangeFixed(itemCode, graphType, graphForm) &&
				(graphForm != GraphProperty.FORM2 || graphType != GraphProperty.TYPE3)) {
			valueaxis.setRange(0.0D, 100D); // y軸のレンジを固定
		} else {
			NumberAxis valueaxis = (NumberAxis)(xyplot.getRangeAxis());
			valueaxis.setAutoRangeIncludesZero(true);
		}
		
		// レンダラと色を選択
		Color[] color = {Color.RED, Color.BLUE, Color.GREEN, Color.ORANGE, Color.CYAN,
				Color.MAGENTA, Color.YELLOW, Color.PINK, Color.GRAY, Color.BLACK,
				Color.LIGHT_GRAY, Color.DARK_GRAY, Color.WHITE};
		if (displayForm.equals("FormStackArea")) {
			XYAreaRenderer renderer = new XYAreaRenderer();
			/** グラフの描画色を固定する */
			for (int i = 0; i < 10; i++) {
				renderer.setSeriesPaint(i, color[i]);
			}
			xyplot.setRenderer(renderer); // グラフを積み上げ面表示にする。(初期値は折線グラフ)
		} else {
			StandardXYItemRenderer renderer = new StandardXYItemRenderer();
			/** グラフの描画色を固定する */
			for (int i = 0; i < 10; i++) {
				renderer.setSeriesPaint(i, color[i]);
			}
            xyplot.setRenderer(renderer); // グラフを折れ線グラフにする。
		}
		return jfreechart[graphId];
	}

	/**
	 * リアルタイムグラフ描画を行っているか否かを確認する。
	 * 
	 * @return グラフ描画を実行中の場合は、true
	 */
	public boolean isCollect() {
		return isCollect;
	}

	/**
	 * リアルタイム収集情報を設定します。
	 * 
	 * @param collectorInfo
	 *            リアルタイム収集情報
	 */
	public void setCollectorInfo(RealtimeCollectorInfo collectorInfo) {
		this.collectorInfo = collectorInfo;
	}

	/**
	 * 収集項目を設定します。
	 * 
	 * @param collectorItemInfo 収集項目
	 */
	public void setCollectorItemInfo(RealtimeCollectorItemInfo collectorItemInfo) {
		this.collectorItemInfo = collectorItemInfo;
	}

	/**
	 * グラフのレンジを設定します。
	 * 
	 * @param graphPlotNum プロット数
	 * @param interval 更新間隔
	 */
	public void setValueaxisRange(double graphPlotNum, int interval) {

		for (int i = 0; i < collectorInfo.getGraphNum(); i++) {
			if (collectorItemInfo.isCollect(i)) {
				// 一旦取得する必要がある.
				if (jfreechart[i] != null) {
					XYPlot xyplot = jfreechart[i].getXYPlot();
					valueaxis = xyplot.getDomainAxis();

					if (graphPlotNum != (valueaxis.getFixedAutoRange()
							* interval * 1000D)) {// graphPlotNum変更時.
						if (graphPlotNum == 0) {
							valueaxis.setAutoRange(true); // うまく自動調整にならない.
						} else if (graphPlotNum > 0
								&& interval == this.interval) {
							valueaxis.setFixedAutoRange(interval * 1000D
									* graphPlotNum);// x軸の表示期間（mS）
													// 12000msec=12sec
						}
					} else if (interval != this.interval && interval > 0) { // interval変更時.
						valueaxis.setFixedAutoRange(interval * 1000D
								* graphPlotNum);// x軸の表示期間（mS） 12000msec=12sec
						this.interval = interval;
					}
				}
			}
		}
	}
}