/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 */

package com.clustercontrol.performance.dialog;

import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.events.VerifyListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.composite.PositiveNumberVerifyListener;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.performance.composite.EstimationComposite;
import com.clustercontrol.performance.composite.tree.CollectorBasicScopeComposite;
import com.clustercontrol.performance.composite.tree.CollectorRuleComposite;
import com.clustercontrol.performance.composite.tree.CollectorTreeComposite;
import com.clustercontrol.performance.bean.CollectorItemInfo;
import com.clustercontrol.performance.bean.CollectorProperty;
import com.clustercontrol.performance.bean.CollectorType;
import com.clustercontrol.performance.ejb.bmp.RecordCollectorData;
import com.clustercontrol.util.Messages;

/**
 * 閾値監視作成・変更ダイアログクラス
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class RecordSetDialog extends CommonDialog {

	public static final int WIDTH_TITLE = 3;
	public static final int WIDTH_VALUE = 2;

	// ----- instance フィールド ----- //

	/** 入力値を保持するオブジェクト */
	private CollectorProperty inputData = null;

	/** 入力値の正当性を保持するオブジェクト */
	private ValidateResult validateResult = null;

	/** 変更対象の収集ID */
//	private String collectorId = null;

	/** 収集基本情報(収集ID、説明、対象スコープ) */
	private CollectorBasicScopeComposite m_collectorBasic = null;

	/** 収集間隔 */
	private CollectorRuleComposite m_collectorRule = null;

	/** 収集期間 */
	private Composite periodComposite = null;
	private Label periodLabel = null;
	private Label periodLabel2 = null;
	private Text hourText = null;
	private Label hourLabel = null;
	private Text minText = null;
	private Label minLabel = null;
	private Text dayText = null;
	private Label dayLabel = null;
	private boolean enabled = true;
	private Button limitButton = null; 
	private Button unlimitButton = null;
	private Label presaveLabel = null;
	private Text presaveText = null;
	private Label dayLabel2 = null;


	/** 収集項目設定用ツリーペイン */
	private CollectorTreeComposite itemTreeComposite;

	/** 
	 * 見積りデータの表示領域
	 */
	private EstimationComposite estimationComposite;
	
	// 見積りデータが空き領域を超えている場合に文字を赤くする処理を
	// 実施するか否かのフラグ
	// 既に収集している収集設定のプロパティ表示時にはチェックの必要がない
	private final boolean checkEstimationSize;
	
	// ----- コンストラクタ ----- //

	/**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent
	 *            親のシェルオブジェクト
	 */
	public RecordSetDialog(Shell parent) {
		super(parent);
		checkEstimationSize = true;
	}

	/**
	 * 変更用ダイアログのインスタンスを返します。
	 * 
	 * @param parent
	 *            親のシェルオブジェクト
	 * @param property
	 *            初期設定値
	 */
	public RecordSetDialog(Shell parent, CollectorProperty property, boolean enabled) {
		super(parent);

		this.inputData = property;
		this.enabled = enabled;
		this.checkEstimationSize = enabled;
	}

	// ----- instance メソッド ----- //

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent
	 *            親のインスタンス
	 */
	@Override
	protected void customizeDialog(Composite parent) {

		Shell shell = this.getShell();

		// タイトル
		if(enabled){
			shell.setText(Messages.getString(Messages.getString("dialog.performance.collector.create")));
		} else {
			shell.setText(Messages.getString(Messages.getString("dialog.performance.collector.property")));
		}

		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;

		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.numColumns = 15;
		parent.setLayout(layout);

		// 監視基本情報
		m_collectorBasic = new CollectorBasicScopeComposite(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_collectorBasic.setLayoutData(gridData);
		// 設定値が既に設定されている場合はそれを画面に設定
		if(inputData != null){
			// 収集基本情報
			m_collectorBasic.setInputData(inputData.getCollectorData());
		}

		// 監視基本情報領域に変更があった場合に通知を受ける
		m_collectorBasic.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				// 見積り情報を更新する
				checkEstimationSize();
			}
		});

		// ラベル
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("collection.setting") + " : ");

		// グループ
		Group groupRule = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 1;
		groupRule.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupRule.setLayoutData(gridData);

		// 収集間隔設定用コンボボックスを生成
		m_collectorRule = new CollectorRuleComposite(groupRule, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_collectorRule.setLayoutData(gridData);
		// 設定値が既に設定されている場合はそれを画面に設定
		if(inputData != null){
			m_collectorRule.setInputData(inputData.getCollectorData());
		}

		// 収集間隔設定用コンボボックスに変更があった場合に通知を受ける
		m_collectorRule.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				// 見積り情報を更新する
				checkEstimationSize();
			}
		});

		// 収集期間設定領域を生成
		createPeriodGroup(groupRule);

		/*
		 * グループ
		 */
		// ラベル
		label = new Label(groupRule, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("collection.item") + " : ");

		String collectorId = null;
		if(inputData != null){
			collectorId = inputData.getCollectorData().getCollectorID();
		}

		// 収集項目ツリー
		itemTreeComposite = new CollectorTreeComposite(
				groupRule,
				SWT.NONE,
				m_collectorBasic.getFacilityId(),
				collectorId
		);

		// スコープ設定が変更になると項目ツリーが更新されるように設定
		m_collectorBasic.setCollectorTreeComposite(itemTreeComposite);

		gridData = new GridData();
		gridData.horizontalSpan = 150;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.heightHint = 300;
		itemTreeComposite.setLayoutData(gridData);
		// 設定値が既に設定されている場合はそれを画面に設定
		if(inputData != null){
			// 収集項目のリストをツリーペインに設定
			this.itemTreeComposite.setItemList(inputData.getItemList());
		}
		// 選択されている収集項目が変更された場合は見積りを実行
		itemTreeComposite.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				// 見積り情報を更新する
				checkEstimationSize();
			}
		});

		// グループ
		Group groupEstimation = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 1;
		groupEstimation.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupEstimation.setLayoutData(gridData);

		// 見積りコンポジット（自身のサイズ見積りを表示）
		estimationComposite = new EstimationComposite(groupEstimation, SWT.NONE, checkEstimationSize);
		// スコープ設定が変更になると見積り領域が更新されるように設定
		m_collectorBasic.setEstimationComposite(estimationComposite);
		// 初期表示
		estimationComposite.update(getInputData());

		// サイズを最適化
		// グリッドレイアウトを用いた場合、こうしないと横幅が画面いっぱいになります。
		shell.pack();
		shell.setSize(new Point(580, shell.getSize().y));

		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);

		setEnabled(enabled);
	}

	/**
	 * This method initializes periodGroup 
	 *
	 */
	private void createPeriodGroup(Composite parent) {
		periodComposite = new Composite(parent, SWT.NONE);
		GridLayout compLayout = new GridLayout(1, true);
		compLayout.marginWidth = 0;
		compLayout.marginHeight = 0;
		compLayout.numColumns = 15;	
		periodComposite.setLayout(compLayout);

		GridData compositeData = new GridData();
		compositeData.horizontalSpan = 15;
		compositeData.horizontalAlignment = GridData.FILL;
		compositeData.grabExcessHorizontalSpace = true;
		periodComposite.setLayoutData(compositeData);

		GridData labelData = new GridData();
		labelData.horizontalSpan = 3;
		labelData.horizontalAlignment = GridData.FILL;
		labelData.grabExcessHorizontalSpace = true;
		GridData limitRadioData = new GridData();
		limitRadioData.horizontalSpan = 3;
		limitRadioData.horizontalAlignment = GridData.FILL;
		limitRadioData.grabExcessHorizontalSpace = true;
		GridData preiodDayTextData = new GridData();
		preiodDayTextData.horizontalSpan = 2;
		preiodDayTextData.horizontalAlignment = GridData.FILL;
		preiodDayTextData.grabExcessHorizontalSpace = true;
		GridData periodDayData  = new GridData();
		periodDayData.horizontalSpan = 1;
		periodDayData.horizontalAlignment = GridData.FILL;
		periodDayData.grabExcessHorizontalSpace = true;
		GridData periodHoutTextData = new GridData();
		periodHoutTextData.horizontalSpan = 2;
		periodHoutTextData.horizontalAlignment = GridData.FILL;
		periodHoutTextData.grabExcessHorizontalSpace = true;
		GridData periodHourData  = new GridData();
		periodHourData.horizontalSpan = 1;
		periodHourData.horizontalAlignment = GridData.FILL;
		periodHourData.grabExcessHorizontalSpace = true;
		GridData periodMinTextData = new GridData();
		periodMinTextData.horizontalSpan = 2;
		periodMinTextData.horizontalAlignment = GridData.FILL;
		periodMinTextData.grabExcessHorizontalSpace = true;
		GridData periodMinData  = new GridData();
		periodMinData.horizontalSpan = 1;
		periodMinData.horizontalAlignment = GridData.FILL;
		periodMinData.grabExcessHorizontalSpace = true;
		GridData unlimitRadioData = new GridData();
		unlimitRadioData.horizontalSpan = 3;
		unlimitRadioData.horizontalAlignment = GridData.FILL;
		unlimitRadioData.grabExcessHorizontalSpace = true;
		GridData presaveLabelData = new GridData();
		presaveLabelData.horizontalSpan = 3;
		presaveLabelData.horizontalAlignment = GridData.FILL;
		presaveLabelData.grabExcessHorizontalSpace = true;
		GridData presaveTextData = new GridData();
		presaveTextData.horizontalSpan = 2;
		presaveTextData.horizontalAlignment = GridData.FILL;
		presaveTextData.grabExcessHorizontalSpace = true;
		GridData presaveDay2Data = new GridData();
		presaveDay2Data.horizontalSpan = 4;
		presaveDay2Data.horizontalAlignment = GridData.FILL;
		presaveDay2Data.grabExcessHorizontalSpace = true;
		VerifyListener verifyListener = new PositiveNumberVerifyListener(0, DataRangeConstant.SMALLINT_HIGH);
		KeyListener checkEstimationListener = new KeyListener(){

			public void keyPressed(KeyEvent arg0) {
			}

			public void keyReleased(KeyEvent arg0) {
				// 見積り情報を更新する
				checkEstimationSize();
			}
		};


		periodLabel = new Label(periodComposite, SWT.NONE);
		periodLabel.setText(Messages.getString("collection.period") + " : ");
		periodLabel.setLayoutData(labelData);

		limitButton = new Button(periodComposite, SWT.RADIO);
		limitButton.setText(com.clustercontrol.performance.util.Messages.getString("PERIOD_SETTING")+"  ");
		limitButton.setLayoutData(limitRadioData);
		limitButton.setSelection(true);
		limitButton.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				dayText.setEnabled(true);
				hourText.setEnabled(true);
				minText.setEnabled(true);
				presaveText.setEnabled(false);

				// 見積り情報を更新する
				checkEstimationSize();
			}

			public void widgetDefaultSelected(SelectionEvent e) {
			}
		});

		dayText = new Text(periodComposite, SWT.BORDER);
		dayText.addVerifyListener(verifyListener);
		dayText.addKeyListener(checkEstimationListener);
		dayText.setLayoutData(preiodDayTextData);
		dayText.setText("0");  // 初期値 //$NON-NLS-1$

		dayLabel = new Label(periodComposite, SWT.NONE);
		dayLabel.setText(com.clustercontrol.performance.util.Messages.getString("DAY"));
		dayLabel.setLayoutData(periodDayData);

		hourText = new Text(periodComposite, SWT.BORDER);
		hourText.addVerifyListener(verifyListener);
		hourText.addKeyListener(checkEstimationListener);
		hourText.setLayoutData(periodHoutTextData);	
		hourText.setText("0");  // 初期値 //$NON-NLS-1$

		hourLabel = new Label(periodComposite, SWT.NONE);
		hourLabel.setText(com.clustercontrol.performance.util.Messages.getString("HOUR"));
		hourLabel.setLayoutData(periodHourData);

		minText = new Text(periodComposite, SWT.BORDER);
		minText.addVerifyListener(verifyListener);
		minText.addKeyListener(checkEstimationListener);
		minText.setLayoutData(periodMinTextData);
		minText.setText("10");  // 初期値 //$NON-NLS-1$

		minLabel = new Label(periodComposite, SWT.NONE);
		minLabel.setText(com.clustercontrol.performance.util.Messages.getString("MINUTE")); 
		minLabel.setLayoutData(periodMinData);

		periodLabel2 = new Label(periodComposite, SWT.NONE);
		periodLabel2.setLayoutData(labelData);

		unlimitButton = new Button(periodComposite, SWT.RADIO);
		unlimitButton.setText(com.clustercontrol.performance.util.Messages.getString("UNLIMIT_COLLECT"));
		unlimitButton.setLayoutData(unlimitRadioData);
		unlimitButton.setSelection(false);
		unlimitButton.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				dayText.setEnabled(false);
				hourText.setEnabled(false);
				minText.setEnabled(false);
				presaveText.setEnabled(true);
				// ラジオボタンはチェックされた場合でもはずした場合でも呼ばれるため、
				// limitのほうで見積り更新処理をすれば、こちらでする必要はない。
				// こちらでも処理すると2重に処理される。
			}

			public void widgetDefaultSelected(SelectionEvent e) {	
			}

		});
		presaveLabel = new Label(periodComposite, SWT.NONE);
		presaveLabel.setText(com.clustercontrol.performance.util.Messages.getString("PRESAVE_PERIOD")); 
		presaveLabel.setLayoutData(presaveLabelData);
		presaveText = new Text(periodComposite, SWT.BORDER);
		presaveText.addVerifyListener(verifyListener);
		presaveText.addKeyListener(checkEstimationListener);
		presaveText.setText("30");  //$NON-NLS-1$
		presaveText.setLayoutData(presaveTextData);

		dayLabel2 = new Label(periodComposite, SWT.NONE);
		dayLabel2.setText(com.clustercontrol.performance.util.Messages.getString("DAY")); 
		dayLabel2.setLayoutData(presaveDay2Data);


		// 設定値が既に設定されている場合はそれを画面に設定
		if(inputData != null){
			// 収集期間
			this.setPeriod(inputData.getCollectorData().getPeriodMin());
		}
	}

	/**
	 * 保存期間を指定します。 
	 */
	private void setPresave(int presave){
		String prev = Integer.toString(presave);
//		//		文字列は4文字以上でないといけない。
//		for (int i = 0 ; i <=(4-prev.length());i++)
//		prev = prev + " "; //$NON-NLS-1$
		presaveText.setText(prev);
	}
	/**
	 * 収集期間を取得します。
	 * @return
	 */
	private int getPresave(){

		String presave=presaveText.getText();
		int presaveInt;
		try {
			presaveInt = Integer.parseInt(presave.trim());
		}
		catch(NumberFormatException e) {
			return -2;
		}

		return presaveInt;
	}

	/**
	 * 収集期間設定領域に設定されている内容を整数化します。
	 * 
	 * @return 収集期間の整数値
	 */
	private int getPeriod(){

		int period;
		try {
			period =
				Integer.parseInt(dayText.getText().trim()) * 24 * 60 +
				Integer.parseInt(hourText.getText().trim()) * 60 +
				Integer.parseInt(minText.getText().trim());
		}
		catch (NumberFormatException e) {
			this.setValidateResult(Messages.getString("INPUT_ERROR"),Messages.getString("MSG_SPECIFY_NUM_COLLECTION_PERIOD"));
			return -2;
		}
		return period;

	}

	/**
	 * 収集期間設定領域に表示する収集期間を設定します。
	 * 
	 * @param period
	 */
	private void setPeriod(int period){
		//収集期間が-1であれば無期限収集
		if(period != -1){
			String day = Integer.toString(period / 60 / 24);
			String hour = Integer.toString(period / 60 % 24);
			String min = Integer.toString(period % 60);

//			//文字列は4文字以上でないといけない。
//			for(int i= 0; i <= (4-day.length());i++)
//			day = day+ " ";		 //$NON-NLS-1$
//			for(int i= 0; i <= (4-hour.length());i++)
//			hour = hour+ " "; //$NON-NLS-1$
//			for(int i= 0; i <= (4-min.length());i++)
//			min = min+ " "; //$NON-NLS-1$

			dayText.setText(day);
			hourText.setText(hour);
			minText.setText(min);

			this.setPresave(0);

		}else{
			this.limitButton.setSelection(false);
			this.unlimitButton.setSelection(true);
			this.setPresave(inputData.getCollectorData().getPresavePeriod());
			minText.setText("0"); //$NON-NLS-1$
		}
	}

	/**
	 * 入力内容を返します。
	 * 
	 * @return 入力内容を保持した設定情報
	 */
	public CollectorProperty getInputData() {
		return this.inputData;
	}

//	/**
//	 * 各項目に入力値を設定します。
//	 * 
//	 * @param property 収集設定
//	 */
//	private void setInputData(CollectorProperty property) {
//		inputData = property;
//	}

	/**
	 * 入力値を用いて通知情報を生成します。
	 * 
	 * @return 入力値を保持した通知情報
	 */
	protected CollectorProperty createInputData() {
		CollectorProperty info = new CollectorProperty();   	
		RecordCollectorData collectorData = new RecordCollectorData();
		info.setCollectorData(collectorData);

		// 収集ID、説明、対象スコープを設定
		validateResult = m_collectorBasic.createInputData(info.getCollectorData());
		if(validateResult != null){
			return null;
		}
		// 収集種別を実績収集に設定
		info.getCollectorData().setCollectorType(CollectorType.RECORD);

		// 収集間隔を設定
		validateResult = m_collectorRule.createInputData(info.getCollectorData());
		if(validateResult != null){
			return null;
		}

		//収集が期間設定であれば
		if(limitButton.getSelection()){
			// 収集期間を設定
			if(getPeriod() == 0){
				validateResult = new ValidateResult();
				validateResult.setMessage( // "収集期間を指定して下さい"
						com.clustercontrol.performance.util.Messages.getString("MSG_SPECIFY_COLLECTION_PERIOD"));
				validateResult.setValid(false);
				return null;
			}
			else if(getPeriod() == -2) {
				this.setValidateResult(com.clustercontrol.performance.util.Messages.getString("INPUT_ERROR"),
						com.clustercontrol.performance.util.Messages.getString("MSG_SPECIFY_NUM_COLLECTION_PERIOD"));
				return null;
			}
			else {
				info.getCollectorData().setPeriodMin(getPeriod());
				info.getCollectorData().setPresavePeriod(0);
			}
		}else{//無期限に収集する。
			if(getPresave()==0){
				validateResult = new ValidateResult();
				//	validateResult.setMessage( // "収集期間を指定して下さい"
				//			com.clustercontrol.performance.util.Messages.getString("MSG_SPECIFY_COLLECTION_PERIOD"));
				validateResult.setMessage(  com.clustercontrol.performance.util.Messages.getString("SET_PRESAVE_PERIOD")); 
				validateResult.setValid(false);
				return null;
			}
			else if(getPresave() == -2) {
				this.setValidateResult(com.clustercontrol.performance.util.Messages.getString("INPUT_ERROR"),
						com.clustercontrol.performance.util.Messages.getString("MSG_SPECIFY_NUM_COLLECTION_PERIOD"));
				return null;
			}

			else{
				info.getCollectorData().setPresavePeriod(getPresave());
				//閾値監視の収集期間が-1のため、無期限は-1とする。
				info.getCollectorData().setPeriodMin(-1);
			}
		}

		// 収集項目を設定
		List<CollectorItemInfo> itemList = itemTreeComposite.getItemList();
		if(itemList.size() == 0){
			validateResult = new ValidateResult();
			validateResult.setValid(false);
			validateResult.setMessage(  // "収集項目を設定して下さい"
					com.clustercontrol.performance.util.Messages.getString("MSG_SPECIFY_COLLECTION_ITEM")); //$NON-NLS-1$
			return null;
		} else {
			info.setItemList(itemList);
		}
		return info;
	}

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
	@Override
	protected ValidateResult validate() {
		// 入力値生成
		this.inputData = this.createInputData();

		if (this.inputData != null) {
			return super.validate();
		} else {
			return validateResult;
		}
	}

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		return Messages.getString("ok");
	}

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}

	/**
	 * 無効な入力値の情報を設定します
	 *  
	 */
	private void setValidateResult(String id, String message) {

		this.validateResult = new ValidateResult();
		this.validateResult.setValid(false);
		this.validateResult.setID(id);
		this.validateResult.setMessage(message);
	}

	public void setEnabled(boolean enabled){
		m_collectorBasic.setEnabled(enabled);
		m_collectorRule.setEnabled(enabled);

		this.dayText.setEnabled(enabled);
		this.hourText.setEnabled(enabled);
		this.minText.setEnabled(enabled);
		this.limitButton.setEnabled(enabled);
		this.unlimitButton.setEnabled(enabled);
		this.presaveText.setEnabled(false);
		itemTreeComposite.setEnabled(enabled);
	}

	/**
	 * 見積りように一時的にCollectorPropertyを作ります。
	 * 収集IDが設定されていなくても生成されるため、これをそのまま設定に
	 * 使うことはできません。
	 * 
	 * @return 入力値を保持した通知情報
	 */
	private CollectorProperty createPropertyForEstimation() {
		CollectorProperty info = new CollectorProperty();
		RecordCollectorData collectorData = new RecordCollectorData();
		info.setCollectorData(collectorData);

		// 収集IDを設定
		String collectorID = m_collectorBasic.getCollectorId();
		info.getCollectorData().setCollectorID(collectorID);
		
		// 対象スコープのファシリティIDを設定
		String facilityId = m_collectorBasic.getFacilityId();
		if(facilityId == null){
			return null;
		}
		info.getCollectorData().setFacilityID(facilityId);

		// 収集種別を実績収集に設定
		info.getCollectorData().setCollectorType(CollectorType.RECORD);

		// 収集間隔を設定
		validateResult = m_collectorRule.createInputData(info.getCollectorData());
		if(validateResult != null){
			return null;
		}

		//収集が期間設定であれば
		if(limitButton.getSelection()){
			// 収集期間を設定
			if(getPeriod() == 0){
				return null;
			} else if(getPeriod() == -2) {
				return null;
			} else {
				info.getCollectorData().setPeriodMin(getPeriod());
				info.getCollectorData().setPresavePeriod(0);
			}
		}else{//無期限に収集する。
			if(getPresave()==0){
				return null;
			} else if(getPresave() == -2) {
				return null;
			} else{
				info.getCollectorData().setPresavePeriod(getPresave());
				//閾値監視の収集期間が-1のため、無期限は-1とする。
				info.getCollectorData().setPeriodMin(-1);
			}
		}

		// 収集項目を設定
		List<CollectorItemInfo> itemList = itemTreeComposite.getItemList();
		if(itemList.size() == 0){
			return null;
		} else {
			info.setItemList(itemList);
		}
		return info;
	}
	
	// サイズ見積りを実行する
	private void checkEstimationSize(){
		if(estimationComposite != null){
			// 設定されている内容でサイズ見積りを実行
			estimationComposite.update(createPropertyForEstimation());
		}
	}
}
