/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.ping.dialog;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.PingRunCountConstant;
import com.clustercontrol.bean.PingRunIntervalConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.composite.PositiveNumberVerifyListener;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.composite.MonitorBasicScopeComposite;
import com.clustercontrol.monitor.run.composite.MonitorBlockComposite;
import com.clustercontrol.monitor.run.composite.MonitorRuleComposite;
import com.clustercontrol.monitor.run.composite.NumericValueInfoComposite;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.notify.composite.NotifyInfoComposite;
import com.clustercontrol.ping.action.AddPing;
import com.clustercontrol.ping.action.GetPing;
import com.clustercontrol.ping.action.ModifyPing;
import com.clustercontrol.ping.bean.PingCheckInfo;
import com.clustercontrol.util.Messages;

/**
 * ping監視作成・変更ダイアログクラス<BR>
 * 
 * @version 3.0.0
 * @since 2.0.0
 */
public class PingCreateDialog extends CommonDialog {
	
	public static final int WIDTH_TITLE = 5;
	public static final int WIDTH_VALUE = 2;

    // ----- instance フィールド ----- //

    /** 入力値を保持するオブジェクト */
    private MonitorInfo inputData = null;

    /** 入力値の正当性を保持するオブジェクト */
    private ValidateResult validateResult = null;
    
    /** 変更対象の監視項目ID */
    private String monitorId = null;
    
    /** 監視基本情報 */
    MonitorBasicScopeComposite m_monitorBasic = null;
    
    /** 監視条件 共通部分 */
    private MonitorRuleComposite m_monitorRule = null;
    
    /** 実行回数 */
	protected Combo m_comboRunCount = null;
	
	/** 実行間隔（秒） */
	protected Combo m_comboRunInterval = null;
		
	/** タイムアウト（ミリ秒） */
	protected Text m_textTimeout = null;
    
    /** 監視単位 */
    private MonitorBlockComposite m_monitorBlock = null;
    
    /** 数値監視判定情報 */
    private NumericValueInfoComposite m_numericValueInfo= null;
    
    /** 通知情報 */
    private NotifyInfoComposite notifyInfo = null;
    
    /** ジョブ実行 */
    private Button m_checkJobRun = null;
    
    
    /** この設定を有効にする */
    private Button confirmValid = null;

    // ----- コンストラクタ ----- //

    /**
     * 作成用ダイアログのインスタンスを返します。
     * 
     * @param parent
     *            親のシェルオブジェクト
     */
    public PingCreateDialog(Shell parent) {
        super(parent);
    }

    /**
     * 変更用ダイアログのインスタンスを返します。
     * 
     * @param parent
     *            親のシェルオブジェクト
     * @param notifyId
     *            変更する通知ID
     */
    public PingCreateDialog(Shell parent, String monitorId) {
        super(parent);

        this.monitorId = monitorId;
    }

    // ----- instance メソッド ----- //

    /**
     * ダイアログエリアを生成します。
     * 
     * @param parent
     *            親のインスタンス
     */
    @Override
	protected void customizeDialog(Composite parent) {
        Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages.getString("dialog.ping.create.modify"));

        // 変数として利用されるラベル
        Label label = null;
        // 変数として利用されるグリッドデータ
        GridData gridData = null;

        // レイアウト
        GridLayout layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 15;
        parent.setLayout(layout);
        
        // 監視基本情報
        m_monitorBasic = new MonitorBasicScopeComposite(parent, SWT.NONE);
        gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_monitorBasic.setLayoutData(gridData);
        
        /*
         * 監視条件グループ
         */
        // ラベル
        label = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("monitor.rule") + " : ");
        
        // グループ
        Group groupRule = new Group(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 1;
        groupRule.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupRule.setLayoutData(gridData);

        m_monitorRule = new MonitorRuleComposite(groupRule, SWT.NONE);
        gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_monitorRule.setLayoutData(gridData);
        
        /*
         * チェック条件グループ
         */
        // ラベル
        label = new Label(groupRule, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("check.rule") + " : ");
        
        // グループ
        Group groupCheckRule = new Group(groupRule, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 15;
        groupCheckRule.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupCheckRule.setLayoutData(gridData);
        
        /*
         * 実行回数
         */
        // ラベル
        label = new Label(groupCheckRule, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_TITLE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("run.count") + " : ");
        // コンボボックス
        this.m_comboRunCount = new Combo(groupCheckRule, SWT.DROP_DOWN | SWT.READ_ONLY);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_comboRunCount.setLayoutData(gridData);
        this.m_comboRunCount.add(PingRunCountConstant.STRING_COUNT_01);
        this.m_comboRunCount.add(PingRunCountConstant.STRING_COUNT_02);
        
        // 空白
        label = new Label(groupCheckRule, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 8;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        /*
         * 実行間隔
         */
        // ラベル
        label = new Label(groupCheckRule, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_TITLE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("run.interval") + " : ");
        // コンボボックス
        this.m_comboRunInterval = new Combo(groupCheckRule, SWT.DROP_DOWN | SWT.READ_ONLY);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_comboRunInterval.setLayoutData(gridData);
        this.m_comboRunInterval.add(PingRunIntervalConstant.STRING_SEC_01);
        this.m_comboRunInterval.add(PingRunIntervalConstant.STRING_SEC_02);
        this.m_comboRunInterval.add(PingRunIntervalConstant.STRING_SEC_05);
        
        // 空白
        label = new Label(groupCheckRule, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 8;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        /*
         * タイムアウト(ミリ秒)
         */
        // ラベル
        label = new Label(groupCheckRule, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_TITLE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("time.out.milli.sec") + " : ");
        // コンボボックス
        this.m_textTimeout = new Text(groupCheckRule, SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_textTimeout.addVerifyListener(
        		new PositiveNumberVerifyListener(1, DataRangeConstant.SMALLINT_HIGH));
        this.m_textTimeout.setLayoutData(gridData);
        
        // 空白
        label = new Label(groupCheckRule, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 8;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        /*
         * 監視単位グループ
         */
        // ラベル
        label = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("monitor.block") + " : ");
        
        // グループ
        Group groupMonitorBlock = new Group(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 1;
        groupMonitorBlock.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupMonitorBlock.setLayoutData(gridData);
        
        // 監視単位情報
        m_monitorBlock = new MonitorBlockComposite(groupMonitorBlock, SWT.NONE);
        gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_monitorBlock.setLayoutData(gridData);
        
		
		/*
         * ノード単位の結果判定の定義グループ
         */
        // ラベル
        label = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("def.result.node") + " : ");
        
        // グループ
        Group groupDefResultNode = new Group(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 1;
        groupDefResultNode.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupDefResultNode.setLayoutData(gridData);
        
        // 数値判定定義情報
        m_numericValueInfo = new NumericValueInfoComposite(groupDefResultNode, 
        		SWT.NONE,
        		false,
        		Messages.getString("ping.reach"),
        		Messages.getString("time.period.milli.sec"),
        		Messages.getString("less"),
        		Messages.getString("less"),
        		NumericValueInfoComposite.INPUT_VERIFICATION_POSITIVE_REAL,
        		NumericValueInfoComposite.INPUT_VERIFICATION_POSITIVE_INTEGER,
        		Double.valueOf(0d),
        		Double.valueOf(100d),
        		Double.valueOf(0d),
        		null);
        m_numericValueInfo.setInputRange1(0, 100);
        m_numericValueInfo.setInputRange2(0, DataRangeConstant.SMALLINT_HIGH);
        gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_numericValueInfo.setLayoutData(gridData);
		
		/*
         * 通知情報の属性グループ
         */
        // ラベル
        label = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("notify.attribute") + " : ");
        
        // グループ
        Group groupNotifyAttribute = new Group(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 1;
        groupNotifyAttribute.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupNotifyAttribute.setLayoutData(gridData);
        
        // 通知情報（通知ID，アプリケーションID）
		this.notifyInfo = new NotifyInfoComposite(groupNotifyAttribute, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 1;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		notifyInfo.setLayoutData(gridData);
	
		
        /*
         * 有効／無効
         */
        this.confirmValid = new Button(parent, SWT.CHECK);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.confirmValid.setLayoutData(gridData);
        this.confirmValid.setText(Messages.getString("setting.valid.confirmed"));

        // ラインを引く
        Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 15;
        line.setLayoutData(gridData);


        // サイズを最適化
        // グリッドレイアウトを用いた場合、こうしないと横幅が画面いっぱいになります。
        shell.pack();
        shell.setSize(new Point(550, shell.getSize().y));

        // 画面中央に
        Display display = shell.getDisplay();
        shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
                (display.getBounds().height - shell.getSize().y) / 2);

        // 通知IDが指定されている場合、その情報を初期表示する。
        MonitorInfo info = null; 
        if(this.monitorId != null){
        	info = new GetPing().getPing(this.monitorId);        	
        }
        else{
        	info = new MonitorInfo();
        }
        this.setInputData(info);
        
    }

    /**
     * 入力内容を返します。
     * 
     * @return 入力内容を保持した通知情報
     */
    public MonitorInfo getInputData() {
        return this.inputData;
    }

    /**
     * 各項目に入力値を設定します。
     * 
     * @param monitor
     *            設定値として用いる通知情報
     */
    protected void setInputData(MonitorInfo monitor) {

        this.inputData = monitor;
        
        // 監視基本情報
        m_monitorBasic.setInputData(monitor);
        
        // 監視条件
        m_monitorRule.setInputData(monitor);
        
        // 監視条件 ping監視情報
        PingCheckInfo pingInfo = (PingCheckInfo)monitor.getPingCheckInfo();
        if(pingInfo == null){
        	pingInfo = new PingCheckInfo();
        }
        if (pingInfo.getRunCount() != null) {
            this.m_comboRunCount.setText(PingRunCountConstant.typeToString(pingInfo.getRunCount().intValue()));
        }
        if (pingInfo.getRunInterval() != null) {
            this.m_comboRunInterval.setText(PingRunIntervalConstant.typeToString(pingInfo.getRunInterval().intValue()));
        }
        if (pingInfo.getTimeout() != null) {
            this.m_textTimeout.setText(pingInfo.getTimeout().toString());
        }
        
        // 監視単位
        m_monitorBlock.setInputData(monitor);
        
        // ノード単位の結果判定の定義
        m_numericValueInfo.setInputData(monitor);
    	
    	
        //通知情報の設定
        if(monitor.getNotifyId() != null)
        this.notifyInfo.setNotify(monitor.getNotifyId());
        
        //アプリケーション
        if (monitor.getApplication() != null) {
        	this.notifyInfo.setApplication(monitor.getApplication());
        }
        
        
        // 有効／無効
        if (monitor.getValid() == ValidConstant.TYPE_VALID) {
            this.confirmValid.setSelection(true);
        }
    }

    /**
     * 入力値を用いて通知情報を生成します。
     * 
     * @return 入力値を保持した通知情報
     */
    protected MonitorInfo createInputData() {
    	MonitorInfo info = new MonitorInfo();

    	info.setMonitorTypeId(HinemosModuleConstant.MONITOR_PING);
    	info.setMonitorType(MonitorTypeConstant.TYPE_NUMERIC);
    	
    	// 監視基本情報
    	validateResult = m_monitorBasic.createInputData(info);
    	if(validateResult != null){
        	return null;
        }
        
        // 監視条件 共通部分
        validateResult = m_monitorRule.createInputData(info);
        if(validateResult != null){
        	return null;
        }
        
        // 監視条件 ping監視情報
        PingCheckInfo pingInfo = new PingCheckInfo();
        pingInfo.setMonitorTypeId(HinemosModuleConstant.MONITOR_PING);
        pingInfo.setMonitorId(info.getMonitorId());
        if (this.m_comboRunCount.getText() != null
                && !"".equals((this.m_comboRunCount.getText()).trim())) {
        	pingInfo.setRunCount(PingRunCountConstant.stringToType(this.m_comboRunCount.getText()));
        } else {
            this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.ping.1"));
            return null;
        }
        if (this.m_comboRunInterval.getText() != null
                && !"".equals((this.m_comboRunInterval.getText()).trim())) {
        	pingInfo.setRunInterval(PingRunIntervalConstant.stringToType(this.m_comboRunInterval.getText()));
        } else {
            this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.ping.2"));
            return null;
        }
        if (this.m_textTimeout.getText() != null
                && !"".equals((this.m_textTimeout.getText()).trim())) {
        	
        	try{
        		Integer timeout = new Integer(this.m_textTimeout.getText().trim());
            	pingInfo.setTimeout(timeout);
        	}
        	catch(NumberFormatException e){
        		this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.ping.3"));
                return null;
        	}
        } else {
            this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.ping.3"));
            return null;
        }
        info.setPingCheckInfo(pingInfo);
        
        // 監視単位
        validateResult = m_monitorBlock.createInputData(info);
        if(validateResult != null){
        	return null;
        }
        
        // ノード単位の結果判定の定義
        validateResult = m_numericValueInfo.createInputData(info);
        if(validateResult != null){
        	return null;
        }
        
        /* 入力値判定 */  
        int runInterval = info.getRunInterval();
        int runCount = pingInfo.getRunCount();
        int interval = pingInfo.getRunInterval();
        int timeout = pingInfo.getTimeout();
        int infoUpper = (int) (info.getNumericValueInfo().get(0)).getThresholdUpperLimit();
        int warnUpper = (int) (info.getNumericValueInfo().get(1)).getThresholdUpperLimit();
        
        // 間隔よりチェック設定の「回数×タイムアウト＋間隔」が大きい場合
        double total = runCount * ((double)timeout / 1000) + ((double)interval / 1000);
        if (runInterval <= (int)total) {
        	this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.52"));
        	return null;
        }
        // タイムアウトより通知の上限が大きい場合
        if (timeout <= infoUpper) {
        	this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.50"));
        	return null;
        }
        // タイムアウトより警告の上限が大きい場合
        if (timeout <= warnUpper) {
        	this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.51"));
        	return null;
        }
        
        // 通知関連情報とアプリケーションの設定
		// 通知グループIDの設定
		info.setNotifyGroupId(NotifyRelationInfo.createNotifyGroupId(HinemosModuleConstant.MONITOR_PING, info.getMonitorId()));
		validateResult = notifyInfo.createInputData(info);
		if (validateResult != null) {
			if(validateResult.getID() == null){	// 通知ID警告用出力
				if(!displayQuestion(validateResult)){
					validateResult = null;
					return null;
				}
			}
			else{	// アプリケーション未入力チェック
				return null;
			}
		}
		
        // 有効/無効
        if (this.confirmValid.getSelection()) {
        	info.setValid(ValidConstant.TYPE_VALID);
        } else {
        	info.setValid(ValidConstant.TYPE_INVALID);
        }

        return info;
    }

    /**
     * 入力値チェックをします。
     * 
     * @return 検証結果
     */
    @Override
	protected ValidateResult validate() {
        // 入力値生成
        this.inputData = this.createInputData();

        if (this.inputData != null) {
            return super.validate();
        } else {
            return validateResult;
        }
    }
    
    /**
     * 入力値をマネージャに登録します。
     * 
     * @return true：正常、false：異常
     * 
     * @see com.clustercontrol.dialog.CommonDialog#action()
     */
    @Override
	protected boolean action() {
    	boolean result = false;
    	
    	MonitorInfo info = this.inputData;
    	if(info != null){
            if(this.monitorId == null){
            	// 作成の場合
            	result = new AddPing().add(info);
            }
            else{
            	// 変更の場合
            	result = new ModifyPing().modify(info);
            }
    	}
    	
    	return result;
    }

    /**
     * ＯＫボタンのテキストを返します。
     * 
     * @return ＯＫボタンのテキスト
     */
    @Override
	protected String getOkButtonText() {
        return Messages.getString("ok");
    }

    /**
     * キャンセルボタンのテキストを返します。
     * 
     * @return キャンセルボタンのテキスト
     */
    @Override
	protected String getCancelButtonText() {
        return Messages.getString("cancel");
    }
    
    /**
     * 無効な入力値の情報を設定します
     *  
     */
    private void setValidateResult(String id, String message) {

        this.validateResult = new ValidateResult();
        this.validateResult.setValid(false);
        this.validateResult.setID(id);
        this.validateResult.setMessage(message);
    }
}