/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.repository.composite;

import java.util.ArrayList;

import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.bean.Property;
import com.clustercontrol.composite.CommonTableViewer;
import com.clustercontrol.repository.action.GetNodeList;
import com.clustercontrol.repository.action.GetNodeListTableDefine;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.NodeListTabelDefine;
import com.clustercontrol.repository.view.NodeAttributeView;
import com.clustercontrol.repository.view.NodeScopeView;
import com.clustercontrol.util.Messages;

/**
 * ノード一覧コンポジットクラス<BR>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class NodeListComposite extends Composite {

    // ----- instance フィールド ----- //

    /** テーブルビューア */
    private CommonTableViewer tableViewer = null;

    /** 表示内容ラベル */
    private Label statuslabel = null;

    /** 合計ラベル */
    private Label totalLabel = null;

    /** 検索条件 */
    private Property condition = null;

    // ----- コンストラクタ ----- //

    /**
     * インスタンスを返します。
     * 
     * @param parent
     *            親のコンポジット
     * @param style
     *            スタイル
     */
    public NodeListComposite(Composite parent, int style) {
        super(parent, style);

        this.initialize();
    }

    // ----- instance メソッド ----- //

    /**
     * コンポジットを生成・構築します。
     */
    private void initialize() {
        GridLayout layout = new GridLayout(1, true);
        this.setLayout(layout);
        layout.marginHeight = 0;
        layout.marginWidth = 0;

        this.statuslabel = new Label(this, SWT.LEFT);
        this.statuslabel.setText("");
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        this.statuslabel.setLayoutData(gridData);

        Table table = new Table(this, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION);
        table.setHeaderVisible(true);
        table.setLinesVisible(true);

        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        table.setLayoutData(gridData);

        // テーブルビューアの作成
        this.tableViewer = new CommonTableViewer(table);
        this.tableViewer.createTableColumn(GetNodeListTableDefine.get(),
                NodeListTabelDefine.SORT_COLUMN_INDEX,
                NodeListTabelDefine.SORT_ORDER);

        this.tableViewer
                .addSelectionChangedListener(new ISelectionChangedListener() {
                    public void selectionChanged(SelectionChangedEvent event) {
                        // 各ビューの更新

                        IWorkbench workbench = ClusterControlPlugin.getDefault()
                                .getWorkbench();
                        IWorkbenchPage page = workbench
                                .getActiveWorkbenchWindow().getActivePage();

                        NodeScopeView scopeView = (NodeScopeView) page
                                .findView(NodeScopeView.ID);
                        if (scopeView != null) {
                            scopeView.update();
                        }
                        NodeAttributeView attributeView = (NodeAttributeView) page
                                .findView(NodeAttributeView.ID);
                        if (attributeView != null) {
                            attributeView.update();
                        }
                    }
                });

        this.totalLabel = new Label(this, SWT.RIGHT);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        this.totalLabel.setLayoutData(gridData);

        this.update();
    }

    /**
     * tableViewerを返します。
     * 
     * @return tableViewer
     */
    public CommonTableViewer getTableViewer() {
        return this.tableViewer;
    }
    
    /**
     * このコンポジットが利用するテーブルを返します。
     * 
     * @return テーブル
     */
    public Table getTable() {
        return this.tableViewer.getTable();
    }

    /**
     * コンポジットを更新します。
     * <p>
     * 
     * 検索条件が事前に設定されている場合、その条件にヒットするノードの一覧を 表示します <br>
     * 検索条件が設定されていない場合は、全ノードを表示します。
     */
    @Override
	public void update() {
        // データ取得
        ArrayList<NodeInfo> list = null;
        if (this.condition == null) {
            this.statuslabel.setText("");
            list = new GetNodeList().getAll();
        } else {
            this.statuslabel.setText(Messages.getString("filtered.list"));
            list = new GetNodeList().get(this.condition);
        }
        
        if(list == null){
        	list = new ArrayList<NodeInfo>();
        }
        
    	ArrayList<Object> listInput = new ArrayList<Object>();
    	for (NodeInfo node : list) {
    		ArrayList<Object> a = new ArrayList<Object>(); 
    		a.add(node.getFacilityId());
    		a.add(node.getFacilityName());
    		a.add(node.getDescription());
    		listInput.add(a);
    	}
    	
        // テーブル更新
        this.tableViewer.setInput(listInput);

        // 合計欄更新
        String[] args = { String.valueOf(list.size()) };
        String message = null;
        if (this.condition == null) {
            message = Messages.getString("records", args);
        } else {
            message = Messages.getString("filtered.records", args);
        }
        this.totalLabel.setText(message);
    }

    /**
     * 検索条件にヒットしたノードの一覧を表示します。
     * <p>
     * 
     * conditionがnullの場合、全ノードを表示します。
     * 
     * @param condition
     *            検索条件
     */
    public void update(Property condition) {
        this.condition = condition;

        this.update();
    }
}