/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.snmptrap.dialog;

import java.util.ArrayList;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.calendar.composite.CalendarIdListComposite;
import com.clustercontrol.composite.CommonTableViewer;
import com.clustercontrol.composite.StringVerifyListener;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.composite.MonitorBasicScopeComposite;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.notify.composite.NotifyInfoComposite;
import com.clustercontrol.snmptrap.action.AddSnmpTrap;
import com.clustercontrol.snmptrap.action.GetSnmpTrap;
import com.clustercontrol.snmptrap.action.ModifySnmpTrap;
import com.clustercontrol.snmptrap.bean.MonitorSnmpTrapInfo;
import com.clustercontrol.snmptrap.bean.MonitorSnmpTrapOidInfo;
import com.clustercontrol.snmptrap.composite.OidListComposite;
import com.clustercontrol.util.Messages;

/**
 * SNMPTRAP監視作成・変更ダイアログクラス<BR>
 * 
 * @version 3.0.0
 * @since 2.1.0
 */
public class SnmpTrapCreateDialog extends CommonDialog {
	
	public static final int WIDTH_TITLE = 5;
	public static final int WIDTH_VALUE = 2;
	
    // 後でpackするためsizeXはダミーの値。
    private final int sizeX = 750;
    private final int sizeY = 760;
	
	// ----- instance フィールド ----- //
	
	/** 入力値を保持するオブジェクト */
	private MonitorSnmpTrapInfo inputData = null;
	
	/** 入力値の正当性を保持するオブジェクト */
	private ValidateResult validateResult = null;
	
	/** 変更対象の監視項目ID */
	private String monitorId = null;
	
	/** 監視基本情報 */
	private MonitorBasicScopeComposite monitorBasicComposite = null;
	
	/** カレンダ設定 */
	CalendarIdListComposite calendarComposite = null;
	
	/** コミュニティ名 */
	private Text textCommunityName = null;
	
	/** OIDテーブル */
	private OidListComposite oidListComposite = null;
	
    /** テーブルビューア */
    private CommonTableViewer tableViewer = null;
    
	/** 通知情報 */
	private NotifyInfoComposite notifyInfoComposite = null;
	
	
	/** この設定を有効にする */
	private Button confirmValid = null;
	
	
	/****/
	
	// ----- コンストラクタ ----- //
	
	/**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 * @param monitorType 監視判定タイプ
	 */
	public SnmpTrapCreateDialog(Shell parent) {
		super(parent);
		setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);
	}
	
	/**
	 * 変更用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 * @param monitorId 変更する監視項目ID
	 * @param monitorType 監視判定タイプ
	 */
	public SnmpTrapCreateDialog(Shell parent, String monitorId) {
		super(parent);
		setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);
		
		this.monitorId = monitorId;
	}

	// ----- instance メソッド ----- //
    
    /**
     * ダイアログの初期サイズを返します。
     * 
     * @return 初期サイズ
     */
    @Override
	protected Point getInitialSize() {
        return new Point(sizeX, sizeY);
    }
	
	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent
	 *            親のインスタンス
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		Shell shell = this.getShell();
		
		// タイトル
		shell.setText(Messages.getString("dialog.snmptrap.create.modify"));
		
		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;
		
		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.numColumns = 15;
		parent.setLayout(layout);
		
		// 監視基本情報
		//SNMPトラップでは未登録ノードからのトラップを受け付けるようにするので、
		//第３引数をtrueとする。
		monitorBasicComposite = new MonitorBasicScopeComposite(parent, SWT.NONE ,true);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		monitorBasicComposite.setLayoutData(gridData);

		/*
		 * カレンダ
		 */
		calendarComposite = new CalendarIdListComposite(parent, SWT.NONE, true);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		calendarComposite.setLayoutData(gridData);
		
		/*
		 * トラップ定義グループ
		 */
		// ラベル
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("trap.definition") + " : ");
		
		// グループ
		Group groupCheckRule = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 15;
		groupCheckRule.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		groupCheckRule.setLayoutData(gridData);
		
		/*
		 * コミュニティ名
		 */
		// ラベル
		label = new Label(groupCheckRule, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("community.name") + " : ");
		// テキスト
		this.textCommunityName = new Text(groupCheckRule, SWT.BORDER | SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = 8;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.textCommunityName.setLayoutData(gridData);
		this.textCommunityName.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_64));
		// 空白
		label = new Label(groupCheckRule, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		
		/*
		 * OIDテーブル
		 */
		// テキスト
		this.oidListComposite = new OidListComposite(groupCheckRule, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		this.oidListComposite.setLayoutData(gridData);
        
		/*
		 * 通知情報の属性グループ
		 */
		// ラベル
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("notify.attribute") + " : ");
		
		// グループ
		Group groupNotifyAttribute = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 1;
		groupNotifyAttribute.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupNotifyAttribute.setLayoutData(gridData);
		
		// 通知情報（通知ID，アプリケーションID，実行）
		this.notifyInfoComposite = new NotifyInfoComposite(groupNotifyAttribute, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 1;
		gridData.horizontalAlignment = GridData.HORIZONTAL_ALIGN_CENTER;
		gridData.grabExcessHorizontalSpace = true;
		notifyInfoComposite.setLayoutData(gridData);		
		
		/*
		 * 有効／無効
		 */
		this.confirmValid = new Button(parent, SWT.CHECK);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.confirmValid.setLayoutData(gridData);
		this.confirmValid.setText(Messages.getString("setting.valid.confirmed"));
		
		// ラインを引く
		Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 15;
		line.setLayoutData(gridData);
		
		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);
		
        //ダイアログのサイズ調整（pack:resize to be its preferred size）
        shell.pack();
        shell.setSize(new Point(shell.getSize().x, sizeY));
		
		// 初期表示
		MonitorSnmpTrapInfo info = null; 
		if(this.monitorId == null){
			// 作成の場合
			info = new MonitorSnmpTrapInfo();
			info.setOidInfos(new ArrayList<MonitorSnmpTrapOidInfo>());
			
		}
		else{
			// 変更の場合、情報取得
			info = new GetSnmpTrap().getSnmp(this.monitorId);
		}
		this.setInputData(info);
		
	}
	
	/**
	 * 入力内容を返します。
	 * 
	 * @return 入力内容を保持した通知情報
	 */
	public MonitorSnmpTrapInfo getInputData() {
		return this.inputData;
	}
	
	/**
	 * 各項目に入力値を設定します。
	 * 
	 * @param monitor
	 *            設定値として用いる監視情報
	 */
	protected void setInputData(MonitorSnmpTrapInfo monitor) {
		
		// 監視基本情報
		monitorBasicComposite.setInputData(monitor);
		
		// カレンダID
		if (monitor.getCalendarId() != null) {
			calendarComposite.setText(monitor.getCalendarId());
		}
		
		// コミュニティ名
		if(monitor.getCommunityName() != null){
			textCommunityName.setText(monitor.getCommunityName());
		}
		
		
		//OID情報
		oidListComposite.setCheckMode(monitor.getCheckMode());
		oidListComposite.setMonitorId(monitor.getMonitorId());
		oidListComposite.setOidList(monitor.getOidInfos());
		oidListComposite.update();
		
        //通知情報の設定
        if(monitor.getNotifyId() != null)
        this.notifyInfoComposite.setNotify(monitor.getNotifyId());
        
        //アプリケーション
        if (monitor.getApplication() != null) {
        	this.notifyInfoComposite.setApplication(monitor.getApplication());
        }
        
        
		// 有効／無効
		if (monitor.getValid() == ValidConstant.TYPE_VALID) {
			this.confirmValid.setSelection(true);
		}
	}
	
	/**
	 * 入力値を用いて通知情報を生成します。
	 * 
	 * @return 入力値を保持した通知情報
	 */
	protected MonitorSnmpTrapInfo createInputData() {
		MonitorSnmpTrapInfo info = new MonitorSnmpTrapInfo();
		
		// ジョブ連携のため、重要度ごとに実行するジョブの情報を保持するテーブルが必要だが、
		// SNMPトラップ監視機能の設定情報を保持するテーブル cc_snmp_trap_info には、
		// ジョブ連携用の情報を保持するカラムがないため、
		// 数値監視用テーブルcc_monitor_numeric_value_infoにその情報を保持する。
    	info.setMonitorTypeId(HinemosModuleConstant.MONITOR_SNMPTRAP);
    	info.setMonitorType(MonitorTypeConstant.TYPE_NUMERIC);
		
		// 監視基本情報
		validateResult = monitorBasicComposite.createInputData(info);
		if(validateResult != null){
			return null;
		}
		
		// カレンダID
		if (calendarComposite.getText() != null
				&& !"".equals((calendarComposite.getText()).trim())) {
			info.setCalendarId(calendarComposite.getText());
		}
		
		// コミュニティ名
		if (this.textCommunityName.getText() != null
				&& !"".equals((this.textCommunityName.getText()).trim())) {
			info.setCommunityName(textCommunityName.getText());
		}
		else{
			this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.snmptrap.1"));
			return null;
		}
		
		//OIDのチェック方法を設定します。
		
		info.setCheckMode(oidListComposite.getCheckMode());
		
		//OID情報
		info.setOidInfos(oidListComposite.getOidList());
		for(int i = 0; i < info.getOidInfos().size(); i++){
			MonitorSnmpTrapOidInfo oidInfo = info.getOidInfos().get(i);
			if(oidInfo.getMonitorId() == null || 
					oidInfo.getMonitorId().equals(info.getMonitorId())){
				oidInfo.setMonitorId(info.getMonitorId());
			}
		}
		
        // 通知関連情報とアプリケーションの設定
		// 通知グループIDの設定
		info.setNotifyGroupId(NotifyRelationInfo.createNotifyGroupId(HinemosModuleConstant.MONITOR_SNMPTRAP, info.getMonitorId()));
		validateResult = this.notifyInfoComposite.createInputData(info);
		if (validateResult != null) {
			if(validateResult.getID() == null){	// 通知ID警告用出力
				if(!displayQuestion(validateResult)){
					validateResult = null;
					return null;
				}
			}
			else{	// アプリケーション未入力チェック
				return null;
			}
		}     

//		if (this.notifyInfoComposite.getApplication() != null
//				&& !"".equals((this.notifyInfoComposite.getApplication()).trim())) {
//			info.setApplication(notifyInfoComposite.getApplication());
//		}
//		else{
//			
//		}
		
		// 有効/無効
		if (this.confirmValid.getSelection()) {
			info.setValid(ValidConstant.TYPE_VALID);
		} else {
			info.setValid(ValidConstant.TYPE_INVALID);
		}
		
		return info;
	}
	
	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
	@Override
	protected ValidateResult validate() {
		// 入力値生成
		this.inputData = this.createInputData();
		
		if (this.inputData != null) { //&& this.validateResult == null) {
			return super.validate();
		} else {
			return validateResult;
		}
	}
	
    /**
     * 入力値をマネージャに登録します。
     * 
     * @return true：正常、false：異常
     * 
     * @see com.clustercontrol.dialog.CommonDialog#action()
     */
    @Override
	protected boolean action() {
    	boolean result = false;
    	
    	MonitorSnmpTrapInfo info = this.inputData;
    	if(info != null){
            if(this.monitorId == null){
            	// 作成の場合
            	result = new AddSnmpTrap().add(info);
            }
            else{
            	// 変更の場合
            	result = new ModifySnmpTrap().modify(info);
            }
    	}
    	
    	return result;
    }
	
	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		return Messages.getString("ok");
	}
	
	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}
	
	/**
	 * 無効な入力値の情報を設定します
	 *  
	 */
	private void setValidateResult(String id, String message) {
		
		this.validateResult = new ValidateResult();
		this.validateResult.setValid(false);
		this.validateResult.setID(id);
		this.validateResult.setMessage(message);
	}
	
    /**
     * tableViewerを返します。
     * 
     * @return tableViewer
     */
    public CommonTableViewer getTableViewer() {
        return this.tableViewer;
    }
}