/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.syslogng.action;

import java.rmi.AccessException;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.dialogs.MessageDialog;

import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.MonitorNotFoundException;
import com.clustercontrol.syslogng.bean.LogFilterInfo;
import com.clustercontrol.syslogng.ejb.session.MonitorSyslogNGController;
import com.clustercontrol.syslogng.util.EjbConnectionManager;
import com.clustercontrol.util.Messages;

/**
 * フィルタ情報を管理するアクションクラス<BR>
 * シングルトン。
 * 
 * @version 2.2.0
 * @since 1.0.0
 */
public class LogManager {

	/** シングルトン用インスタンス。 */
    private static LogManager INSTANCE = null;


    /**
	 * 本クラスのインスタンスを返します。<BR>
	 * シングルトン用インスタンスが<code> null </code>ならば、インスタンスを生成します。<BR>
	 * シングルトン用インスタンスが存在すれば、シングルトン用インスタンスを返します。
	 * 
	 * @return 唯一のインスタンス
	 */
    public static LogManager getInstance() {
        if (INSTANCE == null) {
            synchronized (LogManager.class) {
                if (INSTANCE == null) {
                    INSTANCE = new LogManager();
                }
            }
        }

        return INSTANCE;
    }


    /** フィルタ情報のキャッシュ。 */
    private Map<String, LogFilterInfo> cashe = null;

    /** 順序を管理するフィルタ情報のリスト。 */
    private List<LogFilterInfo> orderList = null;
    
    /** フィルタIDの採番カウンタ */
    private int _nextId = 0;


    /**
	 * コンストラクタ。<BR>
	 * アクセスを禁止します。
	 */
    private LogManager() {
    }

    /**
	 * 初期処理を行います。
	 * 
	 * @see #loadLog()
	 */
    public void initialize() {
    	_nextId = 0;
        this.loadLog();
    }

    /**
	 * 全てのフィルタ情報の配列を返します。
	 * <p>
	 * 順序の番号順に整列した配列を返します。
	 * 
	 * @return フィルタ情報一覧
	 */
    public Object[] get() {

        Object[] records = this.orderList.toArray();
        return records;
    }

    /**
	 * 引数で指定したフィルタIDのフィルタ情報を返します。
	 * <p>
	 * フィルタ情報キャッシュよりフィルタ情報を取得します。
	 * 
	 * @param logId フィルタID
	 * @return フィルタ情報
	 */
    public LogFilterInfo get(String logId) {
        return this.cashe.get(logId);
    }

    /**
	 * 引数で指定したフィルタ情報を追加します。
	 * <p>
	 * 順序とフィルタIDを割り当て、フィルタ情報をキャッシュとリストに保持します。
	 * 
	 * @param log フィルタ情報
	 * @return 成功した場合、<code> true </code>
	 */
    public boolean add(LogFilterInfo log) {
    	
    	if(!this.checkMonitorId(log.getMonitorId())){
			// 監視項目IDが重複している場合、エラーダイアログを表示する
    		String[] args = { log.getMonitorId() };
            MessageDialog.openInformation(
            		null, 
            		Messages.getString("message"),
                    Messages.getString("message.monitor.53", args));
            return false;
    	}
    	
        int order = this.cashe.size() + 1;

        // 順序は一番後ろとする。
        log.setOrderNo(order);
        // 暫定のログIDを振り分ける。
        log.setLogId(getNextId());

        this.cashe.put(log.getLogId(), log);
        this.orderList.add(log);
        return true;
    }
    
    /**
	 * 全てのフィルタ情報の監視項目IDの重複チェック。
	 * <p>
	 * フィルタ情報の配列を取得し、監視項目IDの重複チェックを行う。
	 * 
	 * @return true：重複なし、false：重複あり
	 */
    private boolean checkMonitorId(String monitorId) {
    	boolean check = true;
    	
    	if(monitorId == null){
    		return check;
    	}
    	
    	Object[] records = get();
    	for (int i = 0; i < records.length; i++) {
    		LogFilterInfo info = (LogFilterInfo)records[i];
    		if(monitorId.equals(info.getMonitorId())){
    			check = false;
    			break;
    		}
    	}
    	
    	return check;
    }

    /**
	 * 引数で指定したフィルタ情報を変更します。
	 * <p>
	 * フィルタ情報のキャッシュとリストを置き換えます。
	 * 
	 * @param log フィルタ情報
	 * @return 成功した場合、<code> true </code>
	 */
    public boolean modify(LogFilterInfo log) {
        if (!this.cashe.containsKey(log.getLogId())) {
            return false;
        }

        this.cashe.put(log.getLogId(), log);
        this.orderList.set(log.getOrderNo() - 1, log);

        return true;
    }

    /**
	 * 引数で指定したフィルタIDのフィルタ情報を削除します。
	 * <p>
	 * <ol>
	 * <li>フィルタ情報のキャッシュより、引数で指定されたフィルタ情報を削除します。</li>
	 * <li>フィルタ定情報のリストより、引数で指定されたフィルタ情報を削除します。</li>
	 * <li>リストに保持されているフィルタ情報の順序を振りなおします。</li>
	 * </ol>
	 * 
	 * @param logId フィルタID
	 * @return 成功した場合、<code> true </code>
	 */
    public boolean delete(String logId) {
        if (!this.cashe.containsKey(logId)) {
            return false;
        }

        LogFilterInfo log = this.cashe.remove(logId);

        this.orderList.remove(log.getOrderNo() - 1);

        // 順序を割り当てなおします。
        int order = 0;
        Iterator<LogFilterInfo> ite = this.orderList.iterator();
        while (ite.hasNext()) {
            (ite.next()).setOrderNo(++order);
        }

        return true;
    }

    /**
	 * 引数で指定したフィルタ情報の順序をひとつ上げます。
	 * 
	 * @param logId フィルタID
	 * @return 成功した場合、<code> true </code>
	 * 
	 * @see #change(int, int)
	 */
    public boolean upOrder(String logId) {
        if (!this.cashe.containsKey(logId)) {
            return false;
        }

        LogFilterInfo log = this.cashe.get(logId);
        int oldOrder = log.getOrderNo();
        int newOrder = oldOrder - 1;
        if (newOrder < 1) {
            return false;
        }

        return this.change(oldOrder, newOrder);
    }

    /**
	 * 引数で指定したフィルタ情報の順序をひとつ下げます。
	 * 
	 * @param logId フィルタID
	 * @return 成功した場合、<code> true </code>
	 * 
	 * @see #change(int, int)
	 */
    public boolean downOrder(String logId) {
        if (!this.cashe.containsKey(logId)) {
            return false;
        }

        LogFilterInfo log = this.cashe.get(logId);
        int oldOrder = log.getOrderNo();
        int newOrder = oldOrder + 1;
        if (newOrder > this.cashe.size()) {
            return false;
        }

        return this.change(oldOrder, newOrder);
    }

    /**
     * 現時点までの操作内容を確定します。
     * 
     * @return 成功した場合、<code> true </code>
     * @throws MonitorNotFoundException
     * @throws HinemosUnknownException 
     * 
     * @see com.clustercontrol.syslogng.util.EjbConnectionManager#getMonitorSyslogNGController()
     * @see com.clustercontrol.syslogng.ejb.session.MonitorSyslogNGController#createMonitorRuleList(java.util.ArrayList)
     */
    public boolean commit() throws MonitorNotFoundException, HinemosUnknownException{

        MonitorSyslogNGController syslog = EjbConnectionManager.getConnectionManager()
                .getMonitorSyslogNGController();

        try {
            return syslog.createMonitorRuleList((ArrayList<LogFilterInfo>) this.orderList);
        } catch (RemoteException e) {
			if(e instanceof AccessException){
				// アクセス権なしの場合、エラーダイアログを表示する
	            MessageDialog.openInformation(null, Messages.getString("message"),
	                    Messages.getString("message.accesscontrol.16"));
			}
        } catch (MonitorNotFoundException e) {
        	throw e;
		} catch (HinemosUnknownException e) {
        	throw e;
		}
        return true;
    }

    /**
     * 現時点までの操作内容をキャンセルします。
     * <p>
     * フィルタ情報をロードすることで、変更前のフィルタ情報に戻します。
     * 
     * @return 成功した場合、<code> true </code>
     * 
     * @see #loadLog()
     */
    public boolean rollback() {
        // キャッシュをクリア
        this.loadLog();

        return true;
    }

    /**
     * フィルタ情報をロードします。
     * <p>
     * フィルタ情報を取得し、フィルタ情報のキャッシュとリストに保持します。
     * 
     * @see com.clustercontrol.syslogng.util.EjbConnectionManager#getMonitorSyslogNGController()
     * @see com.clustercontrol.syslogng.ejb.session.MonitorSyslogNGController#getFilterInfoList()
     */
    private void loadLog() {
        this.cashe = new HashMap<String, LogFilterInfo>();
        this.orderList = new ArrayList<LogFilterInfo>();

        MonitorSyslogNGController syslog = EjbConnectionManager.getConnectionManager()
                .getMonitorSyslogNGController();

        ArrayList<LogFilterInfo> records = new ArrayList<LogFilterInfo>();
        try {
            records = syslog.getFilterInfoList();
        } catch (RemoteException e) {
			if(e instanceof AccessException){
				// アクセス権なしの場合、エラーダイアログを表示する
	            MessageDialog.openInformation(null, Messages.getString("message"),
	                    Messages.getString("message.accesscontrol.16"));
			}
        } catch (MonitorNotFoundException e) {
		} catch (HinemosUnknownException e) {
		}

        int index = 0;
        Iterator<LogFilterInfo> ite = records.iterator();
        while (ite.hasNext()) {
        	Object o = ite.next();
            LogFilterInfo log = (LogFilterInfo)o;

            log.setLogId(getNextId());
            this.cashe.put(log.getLogId(), log);
            this.orderList.add(log);
            index++;
        }
    }

    /**
	 * 引数で指定した順序のフィルタ情報同士の順序を入れ替えます。
	 * <p>
	 * 引数で指定する値は、リストのインデックス値ではなく、フィルタ情報の順序を指定します。<BR>
	 * フィルタ情報の順序を変更し、リストの位置を入れ替えてにセットします。
	 * 
	 * @param index1 フィルタ情報１の順序の値
	 * @param index2 フィルタ情報２の順序の値
	 * @return 正常に終了した場合、<code> true </code>
	 */
    private boolean change(int index1, int index2) {
        LogFilterInfo log1 = this.orderList.get(--index1);
        LogFilterInfo log2 = this.orderList.get(--index2);

        int order1 = log1.getOrderNo();
        int order2 = log2.getOrderNo();

        // 順序の値を入れ替えます。
        log1.setOrderNo(order2);
        log2.setOrderNo(order1);

        // リストの位置を入れ替えます。
        this.orderList.set(index1, log2);
        this.orderList.set(index2, log1);

        return true;
    }
    
    /**
     * 新しいフィルタに利用するIDを採番する
     * @return
     */
    private synchronized String getNextId() {
    	return "LOGID" + String.format("%06d", _nextId++);
    }
}