/*
 
Copyright (C) 2010 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.bean;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import com.clustercontrol.repository.bean.FacilityInfo;

/**
 * スコープ情報をツリー構造化するためのクラス<BR>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class FacilityTreeItem implements Serializable {

    //	 ----- static フィールド ----- //

    /**
	 * 
	 */
	private static final long serialVersionUID = 1663475778653652044L;

	/** パスセパレーター */
    public static final String SEPARATOR = ">";

    //	 ----- instance フィールド ----- //

    /** 親 */
    private FacilityTreeItem parent = null;

    /** 情報オブジェクト */
    private FacilityInfo data = null;

    /** 子の格納リスト */
    private List<FacilityTreeItem> childrens = null;

    // ----- コンストラクタ ----- //
    /**
     * JAXB(Webservice)のために、引数なしのコンストラクタを用意。
     */
    public FacilityTreeItem() {}

    /**
     * 引数の情報を保持したインスタンスを取得します。<BR>
     * 
     * @param parent
     *            親のオブジェクト
     * @param data
     *            スコープ情報オブジェクト
     */
    public FacilityTreeItem(FacilityTreeItem parent, FacilityInfo data) {

        this.setParent(parent);
        this.setData(data);

        if (parent != null) {
            parent.addChildren(this);
        }

        this.childrens = new ArrayList();
    }

    // ----- instance フィールド ----- //

    /**
     * 親ファシリティ（スコープ）を取得します。<BR>
     * 
     * @return 親
     */
    public FacilityTreeItem getParent() {
        return this.parent;
    }

    /**
     * 親ファシリティ（スコープ）を設定します。<BR>
     * <p>
     * 
     * インスタンス化の際に親へ関係付けているため、子を削除するメソッドを実装した 後に可視性(スコープ)を拡大して下さい。 <br>
     * また、新しい親への関係付けも行うように実装して下さい。
     * 
     * @return 親ファシリティ
     */
    public void setParent(FacilityTreeItem parent) {
        this.parent = parent;
    }

    /**
     * スコープ情報を取得します。<BR>
     * 
     * @return スコープ情報
     */
    public FacilityInfo getData() {
        return this.data;
    }

    /**
     * スコープ情報を設定します。<BR>
     * 
     * @param data
     *            スコープ情報
     */
    public void setData(FacilityInfo data) {
        this.data = data;
    }

    /**
     * 子ファシリティを追加します。<BR>
     * <p>
     * 
     * この際、childeの親はこのオブジェクトとして設定されます。
     * 
     * @param child
     *            子
     */
    public void addChildren(FacilityTreeItem child) {
        child.setParent(this);
        childrens.add(child);
    }

    /**
     * 子ファシリティの数を取得します。<BR>
     * 
     * @return 子の数
     */
    public int size() {
        return childrens.size();
    }

    /**
     * 全ての子ファシリティを取得します。<BR>
     * <p>
     * 
     * 並び順は、追加された順となっています。
     * 
     * @return 全ての子。
     */
    public FacilityTreeItem[] getChildren() {
        FacilityTreeItem[] result = new FacilityTreeItem[this.size()];
        return (FacilityTreeItem[]) childrens.toArray(result);
    }
    /**
     * 直下の子ファシリティのデータがあれば消します。<BR>
     * 1つだけ消します。
     * 
     */
    public boolean removeChild(String facilityId){
    	
    	
    	for(int i=0 ; i< childrens.size(); i++){
    		
    		if(facilityId.equals(((FacilityTreeItem)childrens.get(i)).getData().getFacilityId())){
    			
    			//マッチした場合にはその要素を消します。
    			childrens.remove(i);
    			return true;
    		}
    	}
    	
		return false;
    }
    
    /**
     * ノードを削除して、スコープだけのツリーにします。
     */
    public boolean removeNode() {
    	boolean flag = false;
    	// for(int i=0 ; i< childrens.size(); i++){ 
    	// removeするとインデックスが変わってしまうため、i--で検索する。
    	for(int i=childrens.size()-1 ; i >= 0; i--){
    		if (((FacilityTreeItem)childrens.get(i)).getData().getFacilityType() ==
    			FacilityConstant.TYPE_NODE) {
    			//マッチした場合にはその要素を消します。
    			childrens.remove(i);
    			flag = true;
    		} else {
    			boolean tmpFlag = childrens.get(i).removeNode();
    			flag = flag || tmpFlag;
    		}
    	}
    	return flag;
    }
    
    /**
     * 直下の子ファシリティのデータがあれば残します。<BR>
     * 引数と一致しない物を消します。
     * 一つ以上消せた場合、trueを返します。
     * 
     */
    public boolean keepChild(String facilityId){
    	
    	boolean ret = false;
    	if (facilityId == null) {
    		return ret;
    	}
    	for(int i=0 ; i< childrens.size(); i++){
    		if(!facilityId.equals(((FacilityTreeItem)childrens.get(i)).getData().getFacilityId())){
    			
    			//マッチしない場合にはその要素を消します。
    			childrens.remove(i);

    			//さらに消せるかどうかチェックします。
    			keepChild(facilityId);
    			
    			//１つ以上消せた場合はtrueを返します。
    			ret = true;
    			break;
    		}
    	}
    	
		return ret;
	}

    /**
     * ツリーのディープコピーを生成して返します
     * 
     * @return ディープコピー
     * @throws IOException
     * @throws ClassNotFoundException
     */
    public synchronized FacilityTreeItem deepCopy() throws IOException, ClassNotFoundException {
		ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
		ObjectOutputStream out = new ObjectOutputStream(byteOut);
		out.writeObject(this);
		out.flush();
		ByteArrayInputStream byteIn = new ByteArrayInputStream(byteOut.toByteArray());
		ObjectInputStream in = new ObjectInputStream(byteIn);
		return (FacilityTreeItem) in.readObject();
	}
}