/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.util;
/**
 * Hinemosより与えられた文字列をコマンドとしてRuntime.exec()に渡すための引数配列を生成するクラス
 * @author Hinemos
 *
 */
public class CommonCreateCommands {

	public static final int WINDOWS = 0; // Windowプラットフォーム指定
	public static final int UNIX = 1; // UNIX/Linuxプラットフォーム指定
	public static final int COMPATIBLE = 2; // 過去互換性モード
	public static final int AUTO = 3; // プラットフォーム自動識別指定
	
	/** Javaプロセス起動ユーザ */
	private String sysUser = null;
	
	/**
	 * 実効ユーザとコマンド文字列に対してRuntime.exec()に渡す文字列配列を作成する処理
	 * @param execUser
	 * @param execCommand
	 * @param platform
	 * @return
	 * @throws Exception
	 */
	public String[] createCommands(String execUser,
			String execCommand, int platform) throws Exception {
		
		String[] commands = null;
		
		/** 入力チェック */
		if(execUser == null ){
			throw new Exception("execUser is null");
		}
		if(execCommand == null ){
			throw new Exception("execCommand is null");
		}
		
		/** プロットフォーム毎の分岐 */
		switch (platform) {
		case WINDOWS:
			commands = _createCommandsForWindows(execUser, execCommand);
			break;

		case UNIX:
			commands = _createCommandsForUnix(execUser, execCommand);
			break;
			
		case COMPATIBLE:
			commands = _createCommandsCompatible(execUser, execCommand);
			break;
			
		case AUTO:
		default:
			commands = _createCommandsAutoDetect(execUser, execCommand);
			break;
		}
		
		return commands;
	}
	
	/**
	 * Windows環境ではCMD経由で起動する。<BR>
	 * Javaプロセス起動ユーザと実効ユーザが異なる場合はスイッチする手段がないためExceptionとする
	 * 
	 * @param execUser
	 * @param execCommand
	 * @return
	 * @throws Exception
	 */
	private String[] _createCommandsForWindows(String execUser, String execCommand) throws Exception {

		//System.out.println("_createCommandsForWindows() execUser = " + execUser + ", execCommand" + execCommand);

		String[] commands = null;
		sysUser = getSystemUser();
		
		/** Javaプロセス起動ユーザと実効ユーザが同じか否かで分岐する */
		if(execUser.equals(sysUser)){

			// ユーザが同じ場合はCMD経由で起動する
			commands = new String[3];
			commands[0] = "CMD";
			commands[1] = "/C";
			commands[2] = execCommand;
		}
		else {
			
			// ユーザが異なる場合はWindows環境ではスイッチできないためERRORとする
			throw new Exception("The execution user of the command and agent's user are different. " +
					"execUser=[" + execUser + "], agentUser=[" + sysUser + "]");
		}
		
		return commands;
	}
	
	/**
	 * UNIX/Linux環境ではsh/sudo経由で起動する。<BR>
	 * rootユーザで起動していなくてもsudoが可能な設定が行われている場合は動作するようにする。
	 * 
	 * @param execUser
	 * @param execCommand
	 * @return
	 * @throws Exception
	 */
	private String[] _createCommandsForUnix(String execUser, String execCommand) throws Exception {
		
		//System.out.println("_createCommandsForUnix() execUser = " + execUser + ", execCommand" + execCommand);
		
		String[] commands = null;
		sysUser = getSystemUser();
		
		/** Javaプロセス起動ユーザと実効ユーザが同じか否かで分岐する */
		if(execUser.equals(sysUser)){
			
			// ユーザが同じ場合はsh経由で起動する
			commands = new String[3];
			commands[0] = "sh";
			commands[1] = "-c";
			commands[2] = execCommand;
		}
		else {

			// ユーザが異なる場合はsudu経由で起動する
			commands = new String[6];
			commands[0] = "sudo";
			commands[1] = "-u";
			commands[2] = execUser;
			commands[3] = "sh";
			commands[4] = "-c";
			commands[5] = execCommand;
		}
		
		return commands;
	}
	
	/**
	 * 下位ver互換性のための生成ロジック。<BR>
	 * Hinemos ver3.1以前で使用している「起動コマンド」の解析ロジックを行う。
	 * 
	 * @param execUser
	 * @param execCommand
	 * @return
	 * @throws Exception
	 */
	private String[] _createCommandsCompatible(String execUser, String execCommand) throws Exception {
		
		//System.out.println("_createCommandsCompatible() execUser = " + execUser + ", execCommand" + execCommand);

		String[] commands = null;
		sysUser = getSystemUser();
		
		/** Javaプロセス起動ユーザと実効ユーザが同じか否かで分岐する */
		if (!execUser.equals(sysUser)) {
			if (sysUser != null && sysUser.equals("root")) {
				
				// 起動ユーザがrootの場合、su -c 'コマンド' ユーザ名 を実行する
				// コマンドパラメータは、一つずつ配列の要素となるようにする
				commands = new String[4];
				commands[0] = "su";
				commands[1] = execUser;
				commands[2] = "-c";
				commands[3] = execCommand;

			} else {
				// 起動ユーザがroot以外の場合、エラーとする
				throw new Exception("The execution user of the command and agent's user are different. " +
						"execUser=[" + execUser + "], agentUser=[" + sysUser + "]");
			}
		} else {
			// 起動ユーザと実行ユーザが一緒の場合はそのまま実行
			commands = execCommand.split(" ");
		}

		
		return commands;
	}
	
	/**
	 * Javaのシステムプロパティよりプラットフォームを自動判別する
	 * 
	 * @param execUser
	 * @param execCommand
	 * @return
	 * @throws Exception
	 */
	private String[] _createCommandsAutoDetect(String execUser, String execCommand) throws Exception {
		
		//System.out.println("_createCommandsAutoDetect() execUser = " + execUser + ", execCommand" + execCommand);

		String[] commands = null;
		String osName = System.getProperty("os.name");
		
		/** OSがWindowsか否(UNIX)かを自動判別する */
		if(osName != null && osName.startsWith("Windows")){
			commands = createCommands(execUser, execCommand, CommonCreateCommands.WINDOWS);
		}
		else {
			commands = createCommands(execUser, execCommand, CommonCreateCommands.UNIX);
		}
		
		return commands;
	}
	
	
	/**
	 * Javaプロセス起動ユーザの取得
	 * @return
	 */
	private String getSystemUser(){
		return System.getProperty("user.name");
	}

	/**
	 * for debug
	 * 単体試験用メインメソッド
	 * 
	 * @param args
	 */
	public static void main(String[] args) {
		
		CommonCreateCommands test = new CommonCreateCommands();
		String command = "hogehoge \"a b\" c , d";
		
		// Windows環境用
		System.setProperty("user.name", "Administrator");
		try {
			printArg(test.createCommands("Administrator", command, CommonCreateCommands.WINDOWS));
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		try {
			printArg(test.createCommands("hinemos", command, CommonCreateCommands.WINDOWS));
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}

		// UNIX/Linux環境用
		System.setProperty("user.name", "root");
		try {
			printArg(test.createCommands("root", command, CommonCreateCommands.UNIX));
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		try {
			printArg(test.createCommands("hinemos", command, CommonCreateCommands.UNIX));
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		
		// 下位互換性環境用
		System.setProperty("user.name", "root");
		try {
			printArg(test.createCommands("hinemos", command, CommonCreateCommands.COMPATIBLE));
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		System.setProperty("user.name", "Administrator");
		try {
			printArg(test.createCommands("hinemos", command, CommonCreateCommands.COMPATIBLE));
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		try {
			printArg(test.createCommands("Administrator", command, CommonCreateCommands.COMPATIBLE));
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		// 環境自動識別用
		System.setProperty("user.name", "Administrator");
		System.setProperty("os.name", "Windows Server 2008");
		try {
			printArg(test.createCommands("Administrator", command, CommonCreateCommands.AUTO));
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}

		System.setProperty("user.name", "root");
		System.setProperty("os.name", "Linux");
		try {
			printArg(test.createCommands("root", command, CommonCreateCommands.AUTO));
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}	
	}
	
	/**
	 * 標準出力用
	 * @param commands
	 */
	private static void printArg(String[] commands){
		if(commands != null){
			for (int i = 0; i < commands.length; i++) {
				System.out.println("commands[" + i + "] : " + commands[i]);
			}
		}
		System.out.println();
	}
}
