/*
 
Copyright (C) since 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.accesscontrol.factory;

import java.util.ArrayList;
import java.util.Collection;

import javax.ejb.EJBException;
import javax.ejb.FinderException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.ejb.entity.UserLocal;
import com.clustercontrol.accesscontrol.ejb.entity.UserRoleLocal;
import com.clustercontrol.accesscontrol.ejb.entity.UserUtil;
import com.clustercontrol.accesscontrol.util.UserValidator;
import com.clustercontrol.bean.Property;
import com.clustercontrol.commons.util.ObjectValidator;
import com.clustercontrol.util.PropertyUtil;

/**
 * ユーザ情報を検索するファクトリクラス<BR>
 *
 * @version 1.0.0
 * @since 3.2.0
 */
public class LoginUserSelector {

	/** ログ出力のインスタンス */
	protected static Log m_log = LogFactory.getLog(LoginUserSelector.class);
	
	/**
	 * 指定のユーザ情報を取得する。<BR>
	 * ユーザ一覧情報は、以下の形式で値が格納されている。<BR>
	 * <PRE>
	 * {
	 *   {ユーザID1, ユーザ名1, 説明1, 作成者（ユーザID）1, 作成日時1, 最終更新者（ユーザID）1, 最終更新日時1}
	 *   ...
	 * }
	 * </PRE>
	 * 
	 * @return 1人のユーザ情報が格納されたArrayListの2次元配列
	 */
	public static ArrayList getOwnData(String userId) {
		
		/** ローカル変数 */
		ArrayList userTable = null;
		
		/** メイン処理 */
		m_log.debug("getting all user...");
		
		try {
			userTable = new ArrayList();
			
			// 全ユーザを取得
			UserLocal user = UserUtil.getLocalHome().findByPrimaryKey(userId);
			
			// ユーザエンティティを配列に変換
			userTable.add(UserValidator.userToArrayList(user));
		} catch (FinderException e) {
			m_log.info("found no user.", e);
		} catch (Exception e) {
			m_log.warn("failure to get all user.", e);
			throw new EJBException("failure to get all user.", e);
		}
		
		m_log.debug("successful in getting all user.");
		return userTable;
	}
	
	/**
	 * ユーザ一覧情報を取得する。<BR>
	 * ユーザ一覧情報は、以下の形式で値が格納されている。<BR>
	 * <PRE>
	 * {
	 *   {ユーザID1, ユーザ名1, 説明1, 作成者（ユーザID）1, 作成日時1, 最終更新者（ユーザID）1, 最終更新日時1}, 
	 *   {ユーザID2, ユーザ名2, 説明2, 作成者（ユーザID）2, 作成日時2, 最終更新者（ユーザID）2, 最終更新日時2},
	 *   ...
	 * }
	 * </PRE>
	 * 
	 * @return ユーザ一覧情報が格納されたArrayListの2次元配列
	 */
	public static ArrayList getUserList() {
		/** ローカル変数 */
		ArrayList userTable = null;
		Collection users = null;
		
		/** メイン処理 */
		m_log.debug("getting all user...");
		
		try {
			userTable = new ArrayList();
			
			// 全ユーザを取得
			users = UserUtil.getLocalHome().findAllLoginUser();
			
			// ユーザエンティティを配列に変換
			for (UserLocal user : (Collection<UserLocal>)users) {
				userTable.add(UserValidator.userToArrayList(user));
			}
		} catch (FinderException e) {
			m_log.info("found no user.", e);
		} catch (Exception e) {
			m_log.warn("failure to get all user.", e);
			throw new EJBException("failure to get all user.", e);
		}
		
		m_log.debug("successful in getting all user.");
		return userTable;
	}
	
	/**
	 * ユーザIDに付与されたユーザ名を取得する。<BR>
	 * 
	 * @param userId ユーザID
	 * @return ユーザ名
	 */
	public static String getUserName(String userId) {
		/** ローカル変数 */
		UserLocal user = null;
		
		/** メイン処理 */
		m_log.debug("getting a user's name...");
		
		try {
			// 該当するユーザを取得
			user = UserUtil.getLocalHome().findByPrimaryKey(userId);
		} catch (Exception e) {
			m_log.warn("failure to get a user's name.", e);
			throw new EJBException("failure to get a user's name.", e);
		}
		
		m_log.debug("successful in getting a user's name.");
		return user.getUserName();
	}
	
	/**
	 * 条件に一致するユーザ一覧情報を取得する。<BR>
	 * 2次元配列の要素（ArrayList）は以下の順序で値が格納されている。<BR>
	 * ユーザ一覧情報は、以下の形式で値が格納されている。<BR>
	 * <PRE>
	 * {
	 *   {ユーザID1, ユーザ名1, 説明1, 作成者（ユーザID）1, 作成日時1, 最終更新者（ユーザID）1, 最終更新日時1}, 
	 *   {ユーザID2, ユーザ名2, 説明2, 作成者（ユーザID）2, 作成日時2, 最終更新者（ユーザID）2, 最終更新日時2},
	 *   ...
	 * }
	 * </PRE>
	 * 
	 * @param property 条件情報
	 * @return ユーザ一覧情報
	 */
	public static ArrayList getUserList(Property property) {
		/** ローカル変数 */
		ArrayList userTable = null;
		ArrayList inputValue = null;
		String userId = null;
		String userName = null;
		String description = null;
		ArrayList<String> roleList = null;
		Collection users = null;
		boolean userIdFlg = false;
		boolean userNameFlg = false;
		boolean descriptionFlg = false;
		int accessMatchCount = 0;
		
		/** メイン処理 */
		m_log.debug("getting users by using filter...");
		
		try {
			userTable = new ArrayList();
			
			// ユーザIDの取得
			inputValue = PropertyUtil.getPropertyValue(property, UserFilterProperty.UID);
			userId = (String)inputValue.get(0);
			
			// ユーザ名の取得
			inputValue = PropertyUtil.getPropertyValue(property, UserFilterProperty.NAME);
			userName = (String)inputValue.get(0);
			
			// 説明の取得
			inputValue = PropertyUtil.getPropertyValue(property, UserFilterProperty.DESCRIPTION);
			description = (String)inputValue.get(0);
			
			// ユーザ権限の取得
			roleList = new ArrayList<String>();
			for (String roleName : UserProperty.getRoleList().getRoles()) {
				inputValue = PropertyUtil.getPropertyValue(property, roleName);
				if ( inputValue.size() > 0 && ((Boolean)inputValue.get(0)).booleanValue() ) {
					roleList.add(roleName);
				}
			}
			
			// 検索条件に基づいた絞込み
			if (! ObjectValidator.isEmptyString(userId)) {
				userIdFlg = true;
			}
			if (! ObjectValidator.isEmptyString(userName)) {
				userNameFlg = true;
			}
			if (! ObjectValidator.isEmptyString(description)) {
				descriptionFlg = true;
			}
			users = UserUtil.getLocalHome().findAllLoginUser();
			for (UserLocal user : (Collection<UserLocal>)users) {
				// ユーザID・ユーザ名・説明が入力されている場合、部分一致しなかったらヒットしない
				if (userIdFlg && user.getUserId().indexOf(userId) == -1) {
					continue;
				}
				if (userNameFlg && user.getUserName().indexOf(userName) == -1) {
					continue;
				}
				if (descriptionFlg && user.getDescription().indexOf(description) == -1) {
					continue;
				}
				if (roleList.size() == 0) {
					// 権限が入力されていない場合、権限の保持に関するチェックは行わない
					userTable.add(UserValidator.userToArrayList(user));
				} else {
					// 選択された権限がユーザの保持する権限の集合に含まれない場合、ヒットしない
					accessMatchCount = 0;
					for (UserRoleLocal role : (Collection<UserRoleLocal>)user.getUserRole()) {
						if ( roleList.contains(role.getUserRole()) ) {
							accessMatchCount++;
						
						}
					}
					if (accessMatchCount == roleList.size()) {
						userTable.add(UserValidator.userToArrayList(user));
					}
				}
			}
		} catch (FinderException e) {
			m_log.info("found no user.", e);
		} catch (Exception e) {
			m_log.info("failure to get users by using filter.", e);
			throw new EJBException("failure to get users by using filter.", e);
		}
		
		m_log.debug("successful in getting users by using filter.");
		return userTable;
	}
}
