/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.calendar.factory;

import java.sql.Time;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.factory.AccessLock;
import com.clustercontrol.bean.DayOfWeekConstant;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.calendar.bean.AddConstant;
import com.clustercontrol.calendar.bean.CalendarConstant;
import com.clustercontrol.calendar.bean.WeekdayConstant;
import com.clustercontrol.calendar.ejb.entity.CalendarAddInfoLocal;
import com.clustercontrol.calendar.ejb.entity.CalendarAddInfoPK;
import com.clustercontrol.calendar.ejb.entity.CalendarAddInfoUtil;
import com.clustercontrol.calendar.ejb.entity.CalendarInfoLocal;
import com.clustercontrol.calendar.ejb.entity.CalendarInfoUtil;
import com.clustercontrol.calendar.ejb.entity.CalendarWeekdayInfoLocal;
import com.clustercontrol.calendar.ejb.entity.CalendarWeekdayInfoPK;
import com.clustercontrol.calendar.ejb.entity.CalendarWeekdayInfoUtil;
import com.clustercontrol.util.PropertyUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * カレンダ更新を行うファクトリークラス<BR>
 *
 * @version 2.0.0
 * @since 2.0.0
 */
public class ModifyCalendar {
	protected static Log m_log = LogFactory.getLog(ModifyCalendar.class);

	/**
	 * カレンダ追加
	 * 
	 * @param property
	 * @param userName
	 * @return
	 * @throws NamingException
	 * @throws CreateException
	 */
	public void addCalendar(Property property, String userName) throws NamingException, CreateException {
    	//アクセスロック
    	AccessLock.lock(AccessLock.CALENDER);
    	
    	String id = null;
	    try {
			ArrayList values = null;
			//現在日時を取得
			Date now = new Date();
			
			//ID取得
			values = PropertyUtil.getPropertyValue(property, CalendarConstant.ID);
			id = (String)values.get(0);
			//名前を取得
			values = PropertyUtil.getPropertyValue(property, CalendarConstant.NAME);
			String name = (String)values.get(0);
			//説明を取得
			values = PropertyUtil.getPropertyValue(property, CalendarConstant.DESCRIPTION);
			String description = (String)values.get(0);
			//有効期間(From)を取得
			values = PropertyUtil.getPropertyValue(property, CalendarConstant.VALID_TIME_FROM);
			Date validTimeFrom = (Date)values.get(0);
			//有効期間(To)を取得
			values = PropertyUtil.getPropertyValue(property, CalendarConstant.VALID_TIME_TO);
			Date validTimeTo = (Date)values.get(0);
			
			//FROMとTOをチェック
			if(validTimeFrom.after(validTimeTo) || validTimeFrom.equals(validTimeTo)){
				throw new java.lang.IllegalArgumentException();
			}
			
			//カレンダを作成
			CalendarInfoUtil.getLocalHome().create(
						id, 
						name,
						description, 
						null, 
						validTimeFrom, 
						validTimeTo, 
						now, 
						now, 
						userName, 
						userName);
			
		} catch (EJBException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            String[] args = {id};
            apllog.put("SYS", "001", args);
            
            m_log.debug("addCalendar() : " + e.getMessage());
			throw e;
		} catch (NamingException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            String[] args = {id};
            apllog.put("SYS", "001", args);
            
            m_log.debug("addCalendar() : " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * カレンダ（基本）情報を変更します。
	 * 
	 * @param property
	 * @param userName
	 * @return
	 * @throws NamingException
	 * @throws FinderException
	 */
	public void modifyCalendar(Property property, String userName) throws NamingException, FinderException {
    	//アクセスロック
    	AccessLock.lock(AccessLock.CALENDER);
    	
    	String id = null;
	    try {
			ArrayList values = null;
			Date now = new Date();
			
			//UID取得
			values = PropertyUtil.getPropertyValue(property, CalendarConstant.ID);
			id = (String)values.get(0);
			
			//calを検索し取得
			CalendarInfoLocal cal = CalendarInfoUtil.getLocalHome().findByPrimaryKey(id);
			//名前を取得し、calに設定
			values = PropertyUtil.getPropertyValue(property, CalendarConstant.NAME);
			cal.setCalendar_name((String)values.get(0));
			//説明を取得し、calに設定
			values = PropertyUtil.getPropertyValue(property, CalendarConstant.DESCRIPTION);
			cal.setDescription((String)values.get(0));
			//有効期間(From)を取得、calに設定
			values = PropertyUtil.getPropertyValue(property, CalendarConstant.VALID_TIME_FROM);
			cal.setValid_time_from((Date)values.get(0));
			//有効期間(To)を取得、calに設定
			values = PropertyUtil.getPropertyValue(property, CalendarConstant.VALID_TIME_TO);
			cal.setValid_time_to((Date)values.get(0));
			//更新者を取得し、calに設定
			cal.setUpdate_user(userName);
			//更新日時を取得し、calに設定
			cal.setUpdate_date(now);
			
		} catch (EJBException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            String[] args = {id};
            apllog.put("SYS", "002", args);
            
            m_log.debug("modifyCalendar() : " + e.getMessage());
			throw e;
		} catch (FinderException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            String[] args = {id};
            apllog.put("SYS", "002", args);
            
            m_log.debug("modifyCalendar() : " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * カレンダ（基本）情報を削除します。<BR>
	 * 
	 * @param id
	 * @return
	 * @throws NamingException
	 * @throws FinderException
	 * @throws NamingException
	 */
	public void deleteCalendar(String id) throws NamingException, RemoveException, FinderException {
    	//アクセスロック
    	AccessLock.lock(AccessLock.CALENDER);
    	
    	try {
			//カレンダ情報を検索し取得
			CalendarInfoLocal cal = CalendarInfoUtil.getLocalHome().findByPrimaryKey(id);
			
            //カレンダ情報を削除
			cal.remove();
		} catch (FinderException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            String[] args = {id};
            apllog.put("SYS", "003", args);
            
            m_log.debug("deleteCalendar() : " + e.getMessage());
			throw e;
		} catch (RemoveException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            String[] args = {id};
            apllog.put("SYS", "003", args);
            
            m_log.debug("deleteCalendar() : " + e.getMessage());
			throw e;
		} catch (NamingException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            String[] args = {id};
            apllog.put("SYS", "003", args);
            
            m_log.debug("deleteCalendar() : " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * カレンダ曜日別情報を登録します。<BR>
	 * 
	 * @param id
	 * @param property
	 * @param userName
	 * @return
	 * @throws NamingException
	 * @throws CreateException
	 * @throws FinderException
	 */
	public void addWeekday(String id, Property property, String userName) throws NamingException, CreateException, FinderException  {
    	//アクセスロック
    	AccessLock.lock(AccessLock.CALENDER);
    	
    	int weekday = 0;
		Date from = null;
		Date to = null;
		try {
			ArrayList values = null;
			//現在日時を取得
			Date now = new Date();
			
			//曜日取得
			values = PropertyUtil.getPropertyValue(property, WeekdayConstant.WEEKDAY);
			weekday = DayOfWeekConstant.stringToType((String)values.get(0));
			//説明を取得
			values = PropertyUtil.getPropertyValue(property, WeekdayConstant.DESCRIPTION);
			String description = (String)values.get(0);
			//有効期間(From)を取得
			values = PropertyUtil.getPropertyValue(property, WeekdayConstant.TIME_FROM);
			from = (Date)values.get(0);
			//有効期間(To)を取得
			values = PropertyUtil.getPropertyValue(property, WeekdayConstant.TIME_TO);
			to = (Date)values.get(0);
			
			//FROMとTOをチェック
			if(from.after(to) || from.equals(to)){
				throw new java.lang.IllegalArgumentException();
			}
			
			//時間帯の重複をチェック
			Collection ct = 
				CalendarWeekdayInfoUtil.getLocalHome().findByTimeFromTo(
						id, Integer.valueOf(weekday), new Time(from.getTime()), new Time(to.getTime()));
			if(ct.size() > 0){
				throw new DuplicateKeyException();
			}
			
			//カレンダ曜日別情報を作成
			CalendarWeekdayInfoUtil.getLocalHome().create(
						id, 
						Integer.valueOf(weekday),
						new Time(from.getTime()), 
						new Time(to.getTime()), 
						description, 
						now, 
						now, 
						userName, 
						userName);
			
		} catch (EJBException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, String.valueOf(weekday), formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "004", args);
			
			m_log.debug("addWeekday() : " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, String.valueOf(weekday), formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "004", args);
			
			m_log.debug("addWeekday() : " + e.getMessage());
			throw e;
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, String.valueOf(weekday), formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "004", args);
			
			m_log.debug("addWeekday() : " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * カレンダ曜日別情報を変更します。<BR>
	 * 
	 * @param id
	 * @param property
	 * @param userName
	 * @return
	 * @throws NamingException
	 * @throws RemoveException
	 * @throws FinderException
	 */
	public void modifyWeekday(String id, Integer weekday, Date from, Date to, Property property, String userName) throws NamingException, RemoveException, CreateException, FinderException  {
    	//アクセスロック
    	AccessLock.lock(AccessLock.CALENDER);
    	
    	try {
			ArrayList values = null;
			//現在日時を取得
			Date now = new Date();
			
			//カレンダ曜日別情報を削除
			CalendarWeekdayInfoLocal info = 
				CalendarWeekdayInfoUtil.getLocalHome().findByPrimaryKey(
						new CalendarWeekdayInfoPK(id, Integer.valueOf(weekday), new Time(from.getTime()), new Time(to.getTime())));
			Date createTimeStamp = info.getReg_date();
			String createUser = info.getReg_user();
			info.remove();
			
			//曜日取得
			values = PropertyUtil.getPropertyValue(property, WeekdayConstant.WEEKDAY);
			int newWeekday = DayOfWeekConstant.stringToType((String)values.get(0));
			//説明を取得
			values = PropertyUtil.getPropertyValue(property, WeekdayConstant.DESCRIPTION);
			String newDescription = (String)values.get(0);
			//有効期間(From)を取得
			values = PropertyUtil.getPropertyValue(property, WeekdayConstant.TIME_FROM);
			Date newFrom = (Date)values.get(0);
			//有効期間(To)を取得
			values = PropertyUtil.getPropertyValue(property, WeekdayConstant.TIME_TO);
			Date newTo = (Date)values.get(0);
			
			//時間帯の重複をチェック
			Collection ct = 
				CalendarWeekdayInfoUtil.getLocalHome().findByTimeFromTo(
						id, Integer.valueOf(newWeekday), new Time(newFrom.getTime()), new Time(newTo.getTime()));
			if(ct.size() > 0){
				throw new DuplicateKeyException();
			}
			
			//カレンダ曜日別情報を作成
			CalendarWeekdayInfoUtil.getLocalHome().create(
						id, 
						Integer.valueOf(newWeekday),
						new Time(newFrom.getTime()), 
						new Time(newTo.getTime()), 
						newDescription, 
						createTimeStamp, 
						now, 
						createUser, 
						userName);

		} catch (EJBException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, weekday.toString(), formatter.format(from), formatter.format(to)};
            apllog.put("SYS", "006", args);
            
            m_log.debug("modifyWeekday() : " + e.getMessage());
			throw e;
		} catch (RemoveException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, weekday.toString(), formatter.format(from), formatter.format(to)};
            apllog.put("SYS", "006", args);
            
            m_log.debug("modifyWeekday() : " + e.getMessage());
			throw e;
		} catch (FinderException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, weekday.toString(), formatter.format(from), formatter.format(to)};
            apllog.put("SYS", "006", args);
            
            m_log.debug("modifyWeekday() : " + e.getMessage());
			throw e;
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, weekday.toString(), formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "006", args);
			
			m_log.debug("modifyWeekday() : " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * カレンダ曜日別情報を削除します。<BR>
	 * 
	 * @param id
	 * @return
	 * @throws NamingException
	 * @throws FinderException
	 * @throws NamingException
	 */
	public void deleteWeekday(String id, Integer weekday, Date from, Date to) throws NamingException, RemoveException, FinderException {
    	//アクセスロック
    	AccessLock.lock(AccessLock.CALENDER);
    	
    	try {
			//カレンダ曜日別情報を削除
			CalendarWeekdayInfoLocal info = 
				CalendarWeekdayInfoUtil.getLocalHome().findByPrimaryKey(
						new CalendarWeekdayInfoPK(id, weekday, new Time(from.getTime()), new Time(to.getTime())));
			info.remove();

		} catch (FinderException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, String.valueOf(weekday), formatter.format(from), formatter.format(to)};
            apllog.put("SYS", "005", args);
            
            m_log.debug("deleteWeekday() : " + e.getMessage());
			throw e;
		} catch (RemoveException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, String.valueOf(weekday), formatter.format(from), formatter.format(to)};
            apllog.put("SYS", "005", args);
            
            m_log.debug("deleteWeekday() : " + e.getMessage());
			throw e;
		} catch (NamingException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, String.valueOf(weekday), formatter.format(from), formatter.format(to)};
            apllog.put("SYS", "005", args);
            
            m_log.debug("deleteWeekday() : " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * カレンダ追加情報(例外実行情報)を追加します。<BR>
	 * 
	 * @param id
	 * @param property
	 * @param userName
	 * @return
	 * @throws NamingException
	 * @throws CreateException
	 * @throws FinderException
	 */
	public void addAddInfo(String id, Property property, String userName) throws NamingException, CreateException, FinderException  {
    	//アクセスロック
    	AccessLock.lock(AccessLock.CALENDER);
    	
    	Date from = null;
		Date to = null;
		try {
			ArrayList values = null;
			//現在日時を取得
			Date now = new Date();
			
			//説明を取得
			values = PropertyUtil.getPropertyValue(property, AddConstant.DESCRIPTION);
			String description = (String)values.get(0);
			//有効期間(From)を取得
			values = PropertyUtil.getPropertyValue(property, AddConstant.TIME_FROM);
			from = (Date)values.get(0);
			//有効期間(To)を取得
			values = PropertyUtil.getPropertyValue(property, AddConstant.TIME_TO);
			to = (Date)values.get(0);
			//実行取得
			values = PropertyUtil.getPropertyValue(property, AddConstant.EXECUTE);
			int execute = ValidConstant.booleanToType((Boolean)values.get(0));
			
			//FROMとTOをチェック
			if(from.after(to) || from.equals(to)){
				throw new java.lang.IllegalArgumentException();
			}
			
			//時間帯の重複をチェック
			Collection ct = 
				CalendarAddInfoUtil.getLocalHome().findByTimeFromTo(
						id, from, to);
			if(ct.size() > 0){
				throw new DuplicateKeyException();
			}
			
			//カレンダ追加情報を作成
			CalendarAddInfoUtil.getLocalHome().create(
						id, 
						from, 
						to, 
						Integer.valueOf(execute),
						description, 
						now, 
						now, 
						userName, 
						userName);
			
		} catch (EJBException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
            apllog.put("SYS", "007", args);
            
            m_log.debug("addAddInfo() : " + e.getMessage());
			throw e;
		} catch (NamingException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
            apllog.put("SYS", "007", args);
            
            m_log.debug("addAddInfo() : " + e.getMessage());
			throw e;
		} catch (CreateException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
            apllog.put("SYS", "007", args);
            
            m_log.debug("addAddInfo() : " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * カレンダ追加情報(例外実行情報)を変更します。<BR>
	 * 
	 * @param id
	 * @param property
	 * @param userName
	 * @return
	 * @throws NamingException
	 * @throws CreateException
	 * @throws FinderException
	 */
	public void modifyAddInfo(String id, Date from, Date to, Property property, String userName) throws NamingException, RemoveException, CreateException, FinderException  {
    	//アクセスロック
    	AccessLock.lock(AccessLock.CALENDER);
    	
    	try {
			ArrayList values = null;
			//現在日時を取得
			Date now = new Date();
			
			//カレンダ追加情報を削除
			CalendarAddInfoLocal info = 
				CalendarAddInfoUtil.getLocalHome().findByPrimaryKey(
						new CalendarAddInfoPK(id, from, to));
			Date createTimeStamp = info.getReg_date();
			String createUser = info.getReg_user();
			info.remove();
			
			//説明を取得
			values = PropertyUtil.getPropertyValue(property, AddConstant.DESCRIPTION);
			String newDescription = (String)values.get(0);
			//有効期間(From)を取得
			values = PropertyUtil.getPropertyValue(property, AddConstant.TIME_FROM);
			Date newFrom = (Date)values.get(0);
			//有効期間(To)を取得
			values = PropertyUtil.getPropertyValue(property, AddConstant.TIME_TO);
			Date newTo = (Date)values.get(0);
			//実行取得
			values = PropertyUtil.getPropertyValue(property, AddConstant.EXECUTE);
			int newExecute = ValidConstant.booleanToType((Boolean)values.get(0));
			
			//時間帯の重複をチェック
			Collection ct = 
				CalendarAddInfoUtil.getLocalHome().findByTimeFromTo(
						id, newFrom, newTo);
			if(ct.size() > 0){
				throw new DuplicateKeyException();
			}
			
			//カレンダ追加情報を作成
			CalendarAddInfoUtil.getLocalHome().create(
						id, 
						newFrom, 
						newTo, 
						Integer.valueOf(newExecute),
						newDescription, 
						createTimeStamp, 
						now, 
						createUser, 
						userName);
			
		} catch (EJBException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
            apllog.put("SYS", "009", args);
            
            m_log.debug("modifyAddInfo() : " + e.getMessage());
			throw e;
		} catch (RemoveException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
            apllog.put("SYS", "009", args);
            
            m_log.debug("modifyAddInfo() : " + e.getMessage());
			throw e;
		} catch (FinderException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
            apllog.put("SYS", "009", args);
            
            m_log.debug("modifyAddInfo() : " + e.getMessage());
			throw e;
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "009", args);
			
			m_log.debug("modifyAddInfo() : " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * カレンダ追加情報(例外実行情報)を削除します。<BR>
	 * 
	 * @param id
	 * @return
	 * @throws NamingException
	 * @throws FinderException
	 * @throws NamingException
	 */
	public void deleteAddInfo(String id, Date from, Date to) throws NamingException, RemoveException, FinderException {
    	//アクセスロック
    	AccessLock.lock(AccessLock.CALENDER);
    	
    	try {
			//カレンダ追加情報を削除
			CalendarAddInfoLocal info = 
				CalendarAddInfoUtil.getLocalHome().findByPrimaryKey(
						new CalendarAddInfoPK(id, from, to));
			info.remove();
		} catch (FinderException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
            apllog.put("SYS", "008", args);
            
            m_log.debug("deleteAddInfo() : " + e.getMessage());
			throw e;
		} catch (RemoveException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
            apllog.put("SYS", "008", args);
            
            m_log.debug("deleteAddInfo() : " + e.getMessage());
			throw e;
		} catch (NamingException e) {
            AplLogger apllog = new AplLogger("CAL", "cal");
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
            apllog.put("SYS", "008", args);
            
            m_log.debug("deleteAddInfo() : " + e.getMessage());
			throw e;
		}
	}
}
