/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.hinemosagent.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import javax.ejb.EJBException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.MonitorDuplicateException;
import com.clustercontrol.MonitorNotFoundException;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.hinemosagent.factory.AddMonitorAgent;
import com.clustercontrol.hinemosagent.factory.DeleteMonitorAgent;
import com.clustercontrol.hinemosagent.factory.ModifyMonitorAgent;
import com.clustercontrol.hinemosagent.factory.SelectMonitorAgent;
import com.clustercontrol.monitor.run.bean.MonitorInfo;

/**
 * エージェント監視を制御するSession Bean<BR>
 * 
 * @ejb.bean name="MonitorAgentController"
 *           jndi-name="MonitorAgentController"
 *           type="Stateless"
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorAgentInfoLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 */
public abstract class MonitorAgentControllerBean implements SessionBean {
	
	protected static Log m_log = LogFactory.getLog( MonitorAgentControllerBean.class );
	
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	/**
	 * セッションコンテキストを取得します。<BR>
	 */
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * Hinemos Agent監視情報を登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="AgentMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 通知情報
	 * @return 登録に成功した場合、true
	 * @throws MonitorDuplicateException 
	 * @throws HinemosUnknownException 
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean addAgent(MonitorInfo info) throws MonitorDuplicateException, HinemosUnknownException {
		
		Principal pri = m_context.getCallerPrincipal();

		// Hinemos Agent監視情報を登録
		AddMonitorAgent agent = new AddMonitorAgent();
		boolean flag;
		try {
			flag = agent.add(info, pri.getName());
		} catch (MonitorDuplicateException e) {
			m_log.warn("addAgent() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("addAgent() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(),e);
		}
		return flag;
	}
	
	/**
	 * Hinemos Agent監視情報を更新します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="AgentMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 通知情報
	 * @return 変更に成功した場合、true
	 * @throws MonitorNotFoundException
	 * @throws HinemosUnknownException
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean modifyAgent(MonitorInfo info) throws MonitorNotFoundException, HinemosUnknownException {
		
		Principal pri = m_context.getCallerPrincipal();

		// Hinemos Agent監視情報を更新
		ModifyMonitorAgent agent = new ModifyMonitorAgent();
		boolean flag;
		try {
			flag = agent.modify(info, pri.getName());
		} catch (MonitorNotFoundException e) {
			m_log.warn("modifyAgent() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("modifyAgent() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(),e);
		}
		return flag;
	}
	
	/**
	 * Hinemos Agent監視情報を削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="AgentMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param monitorId 監視項目ID
	 * @return 削除に成功した場合、true
	 * @throws MonitorNotFoundException
	 * @throws HinemosUnknownException 
	 */
	public boolean deleteAgent(String monitorId) throws MonitorNotFoundException, HinemosUnknownException {

		// Hinemos Agent監視情報を削除
		DeleteMonitorAgent agent = new DeleteMonitorAgent();
		boolean flag;
		try {
			flag = agent.delete(HinemosModuleConstant.MONITOR_AGENT, monitorId);
		} catch (MonitorNotFoundException e) {
			m_log.warn("deleteAgent() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("deleteAgent() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(),e);
		}
		return flag;
	}
	
	/**
	 * Hinemos Agent監視情報を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="AgentMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param monitorId 状態監視ID
	 * @return Hinemos Agent監視情報
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException 
	 */
	public MonitorInfo getAgentInfo(String monitorId) throws MonitorNotFoundException, HinemosUnknownException {

		// Hinemos Agent監視情報を取得
		SelectMonitorAgent agent = new SelectMonitorAgent();
		MonitorInfo info = null;
		try {
			info = agent.getMonitor(HinemosModuleConstant.MONITOR_AGENT, monitorId);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getAgentInfo " + e.getMessage(),e);
			throw new HinemosUnknownException(e.getMessage(),e);
		}
		return info;
	}

	/**
	 * Hinemos Agent監視一覧リストを返します。
	 * 
	 * 戻り値はArrayListのArryListであり、内部のArrayListは以下のように格納される。
	 * 
	 * <Pre>
	 * ArrayList info = new ArrayList();
 	 *	info.add(monitor.getMonitorId());
	 *	info.add(monitor.getDescription());
	 *	info.add(facilityPath);
	 *	info.add(MonitorTypeConstant.typeToString(monitor.getMonitorType().intValue()));
	 *	info.add(monitor.getCalendarId());
	 *	info.add(RunIntervalConstant.typeToString(monitor.getRunInterval().intValue()));
	 *	info.add(schedule.getValid(monitor.getMonitorTypeId(), monitor.getMonitorId()));
	 *	info.add(monitor.getRegUser());
	 *	info.add(monitor.getRegDate() == null ? null:new Date(monitor.getRegDate().getTime()));
	 *	info.add(monitor.getUpdateUser());
	 *	info.add(monitor.getUpdateDate() == null ? null:new Date(monitor.getUpdateDate().getTime()));
	 *
	 *	list.add(info);
	 * </Pre>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="AgentMonitorRead"
     *     method-intf="Remote"
     *     
     * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFoundException
	 * @throws HinemosUnknownException 
	 */
	public ArrayList<MonitorInfo> getAgentList() throws MonitorNotFoundException, HinemosUnknownException {

		// Hinemos Agent監視一覧を取得
		SelectMonitorAgent agent = new SelectMonitorAgent();
		ArrayList<MonitorInfo> list = null;
		try {
			list = agent.getMonitorList(HinemosModuleConstant.MONITOR_AGENT);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getAgentList " + e.getMessage(),e);
			throw new HinemosUnknownException(e.getMessage(),e);
		}
		return list;
	}
}
