/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.http.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import javax.ejb.EJBException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.MonitorDuplicateException;
import com.clustercontrol.MonitorNotFoundException;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.http.factory.AddMonitorHttp;
import com.clustercontrol.http.factory.AddMonitorHttpString;
import com.clustercontrol.http.factory.DeleteMonitorHttp;
import com.clustercontrol.http.factory.ModifyMonitorHttp;
import com.clustercontrol.http.factory.ModifyMonitorHttpString;
import com.clustercontrol.http.factory.SelectMonitorHttp;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.factory.AddMonitor;
import com.clustercontrol.monitor.run.factory.DeleteMonitor;
import com.clustercontrol.monitor.run.factory.ModifyMonitor;
import com.clustercontrol.monitor.run.factory.SelectMonitor;

/**
 * HTTP監視を制御するSession Beanクラス<BR>
 * 
 * @ejb.bean name="MonitorHttpController"
 *           jndi-name="MonitorHttpController"
 *           type="Stateless"
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorHttpInfoLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 *     
 */
public abstract class MonitorHttpControllerBean implements SessionBean {
	
	protected static Log m_log = LogFactory.getLog( MonitorHttpControllerBean.class );
	
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * HTTP監視情報をマネージャに登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="HttpMonitorWrite"
	 *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 登録に成功した場合、true
	 * @throws MonitorDuplicateException 
	 * @throws HinemosUnknownException
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean addHttp(MonitorInfo info) throws MonitorDuplicateException, HinemosUnknownException {
		
		Principal pri = m_context.getCallerPrincipal();
		
		// HTTP監視情報を登録
		AddMonitor http = null;
		if(info.getMonitorType() == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			http = new AddMonitorHttp();
		}
		else if(info.getMonitorType() == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			http = new AddMonitorHttpString();
		}
		else{
			return false;
		}
		boolean flag;
		try {
			flag = http.add(info, pri.getName());
		} catch (MonitorDuplicateException e) {
			m_log.warn("addHttp() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("addHttp() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * HTTP監視情報をマネージャ上で更新します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="HttpMonitorWrite"
	 *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 変更に成功した場合、true
	 * @throws HinemosUnknownException 
	 * @throws MonitorNotFoundException
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean modifyHttp(MonitorInfo info) throws MonitorNotFoundException, HinemosUnknownException {
		
		Principal pri = m_context.getCallerPrincipal();
		
		// HTTP監視情報を更新
		ModifyMonitor http = null;
		if(info.getMonitorType() == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			http = new ModifyMonitorHttp();
		}
		else if(info.getMonitorType() == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			http = new ModifyMonitorHttpString();
		}
		else{
			return false;
		}
		boolean flag;
		try {
			flag = http.modify(info, pri.getName());
		} catch (MonitorNotFoundException e) {
			m_log.warn("modifyHttp() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("modifyHttp() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * HTTP監視情報をマネージャから削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="HttpMonitorWrite"
	 *     method-intf="Remote"
	 * 
	 * @param monitorId 監視項目ID
	 * @return 削除に成功した場合、true
	 * @throws HinemosUnknownException 
	 * @throws MonitorNotFoundException 
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean deleteHttp(String monitorId) throws MonitorNotFoundException, HinemosUnknownException {
		
		// HTTP監視情報を削除
		DeleteMonitor http = new DeleteMonitorHttp();
		boolean flag;
		try {
			flag = http.delete(HinemosModuleConstant.MONITOR_HTTP, monitorId);
		} catch (MonitorNotFoundException e) {
			m_log.warn("deleteHttp() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("deleteHttp() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * HTTP監視情報を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="HttpMonitorRead"
	 *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param monitorType 監視判定タイプ
	 * @return HTTP監視情報
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public MonitorInfo getHttpInfo(String monitorId) throws MonitorNotFoundException, HinemosUnknownException {
		
		// HTTP監視情報を取得
		SelectMonitor http = new SelectMonitorHttp();
		
		MonitorInfo info = null;
		try {
			info = http.getMonitor(HinemosModuleConstant.MONITOR_HTTP, monitorId);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getHttpInfo " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return info;
	}
	
	/**
	 * HTTP監視一覧リストを取得します。<BR>
	 * 
	 * 戻り値のArrayListはArrayListのArrayListであり、
	 * 内部のArrayListは以下のように生成される。<BR>
	 * 
	 * <Pre>
	 * ArrayList info = new ArrayList();
	 *  info.add(monitor.getMonitorId());
	 *  info.add(monitor.getDescription());
	 *  info.add(facilityPath);
	 *  info.add(MonitorTypeConstant.typeToString(monitor.getMonitorType().intValue()));
	 *  info.add(monitor.getCalendarId());
	 *  info.add(RunIntervalConstant.typeToString(monitor.getRunInterval().intValue()));
	 *  info.add(schedule.getValid(monitor.getMonitorTypeId(), monitor.getMonitorId()));
	 *  info.add(monitor.getRegUser());
	 *  info.add(monitor.getRegDate() == null ? null:new Date(monitor.getRegDate().getTime()));
	 *  info.add(monitor.getUpdateUser());
	 *  info.add(monitor.getUpdateDate() == null ? null:new Date(monitor.getUpdateDate().getTime()));
	 *	
	 *	list.add(info);
	 * </li>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="HttpMonitorRead"
	 *     method-intf="Remote"
	 *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public ArrayList<MonitorInfo> getHttpList() throws MonitorNotFoundException, HinemosUnknownException {
		
		// HTTP監視一覧を取得
		SelectMonitorHttp http = new SelectMonitorHttp();
		ArrayList<MonitorInfo> list = null;
		try {
			list = http.getMonitorList(HinemosModuleConstant.MONITOR_HTTP);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getHttpList " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return list;
	}
	
}
