/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.factory;

import java.sql.Timestamp;
import java.util.Date;
import java.util.Locale;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ConfirmConstant;
import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.PropertyConstant;
import com.clustercontrol.monitor.bean.EventInfoConstant;
import com.clustercontrol.notify.monitor.ejb.entity.EventLogLocal;
import com.clustercontrol.notify.monitor.ejb.entity.EventLogPK;
import com.clustercontrol.notify.monitor.ejb.entity.EventLogUtil;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * イベント詳細情報表示用プロパティを作成するクラス<BR>
 *
 * @version 2.0.0
 * @since 2.0.0
 */
public class EventInfoProperty {
	/** ログ出力のインスタンス */
	protected static Log m_log = LogFactory.getLog( EventInfoProperty.class );
	
	/**
	 * イベント詳細情報表示用プロパティを返します。<BR>
	 * <p>
	 * <ol>
	 *  <li>引数で指定された条件に一致するイベント情報を取得します。</li>
	 *  <li>フィルタ項目毎にID, 名前, 処理定数（{@link com.clustercontrol.bean.PropertyConstant}）を指定し、
	 *      プロパティ（{@link com.clustercontrol.bean.Property}）を生成します。</li>
	 *  <li>各項目のプロパティに値を設定し、ツリー状に定義します。</li>
	 * </ol>
	 * 
	 * <p>プロパティに定義する項目は、下記の通りです。
	 * <p>
	 * <ul>
	 *  <li>プロパティ（親。ダミー）</li>
	 *  <ul>
	 *   <li>重要度（子。テキスト）</li>
	 *   <li>受信日時（子。テキスト）</li>
	 *   <li>出力日時（子。テキスト）</li>
	 *   <li>プラグインID（子。テキスト）</li>
	 *   <li>監視項目ID（子。テキスト）</li>
	 *   <li>ファシリティID（子。テキスト）</li>
	 *   <li>スコープ（子。テキスト）</li>
	 *   <li>アプリケーション（子。テキスト）</li>
	 *   <li>メッセージID（子。テキスト）</li>
	 *   <li>メッセージ（子。テキスト）</li>
	 *   <li>オリジナルメッセージ（子。テキストエリア）</li>
	 *   <li>確認（子。テキスト）</li>
	 *   <li>確認済み日時（子。テキスト）</li>
 	 *   <li>確認ユーザ（子。テキスト）</li>
	 *   <li>重複カウンタ（子。テキスト）</li>
	 *  </ul>
	 * </ul>
	 * 
	 * @param monitorId 取得対象の監視項目ID
	 * @param pluginId 取得対象のプラグインID
	 * @param facilityId 取得対象のファシリティID
	 * @param outputDate 取得対象の受信日時
	 * @param locale ロケール情報
	 * @return イベント詳細情報表示用プロパティ
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.bean.Property
	 * @see com.clustercontrol.bean.PropertyConstant
	 * @see com.clustercontrol.bean.PriorityConstant
	 * @see com.clustercontrol.bean.ConfirmConstant
	 */
	public Property getProperty(
			String monitorId,
			String pluginId,
			String facilityId,
			Date outputDate,
			Locale locale) throws FinderException, NamingException {
		
		//重要度
		Property m_priority =
			new Property(EventInfoConstant.PRIORITY, Messages.getString("priority", locale), PropertyConstant.EDITOR_TEXT);
		//受信日時
		Property m_outputDate =
			new Property(EventInfoConstant.OUTPUT_DATE, Messages.getString("receive.time", locale), PropertyConstant.EDITOR_DATETIME);
		//出力日時
		Property m_generationDate = 
			new Property(EventInfoConstant.GENERATION_DATE, Messages.getString("output.time", locale), PropertyConstant.EDITOR_DATETIME);
		//プラグインID
		Property m_pluginId =
			new Property(EventInfoConstant.PLUGIN_ID, Messages.getString("plugin.id", locale), PropertyConstant.EDITOR_TEXT);
		//監視項目ID
		Property m_monitorId =
			new Property(EventInfoConstant.MONITOR_ID, Messages.getString("monitor.id", locale), PropertyConstant.EDITOR_TEXT);
		//ファシリティID
		Property m_facilityId =
			new Property(EventInfoConstant.FACILITY_ID, Messages.getString("facility.id", locale), PropertyConstant.EDITOR_TEXT);
		//スコープ
		Property m_scopeText =
			new Property(EventInfoConstant.SCOPE_TEXT, Messages.getString("scope", locale), PropertyConstant.EDITOR_FACILITY);
		//アプリケーション
		Property m_application =
			new Property(EventInfoConstant.APPLICATION, Messages.getString("application", locale), PropertyConstant.EDITOR_TEXT);
		//メッセージID
		Property m_messageId =
			new Property(EventInfoConstant.MESSAGE_ID, Messages.getString("message.id", locale), PropertyConstant.EDITOR_TEXT);
		//メッセージ
		Property m_message =
			new Property(EventInfoConstant.MESSAGE, Messages.getString("message", locale), PropertyConstant.EDITOR_TEXT);
		//オリジナルメッセージ
		Property m_messageOrg =
			new Property(EventInfoConstant.MESSAGE_ORG, Messages.getString("message.org", locale), PropertyConstant.EDITOR_TEXTAREA, DataRangeConstant.TEXT);
		//確認
		Property m_confirmed =
			new Property(EventInfoConstant.CONFIRMED, Messages.getString("confirmed", locale), PropertyConstant.EDITOR_TEXT);
		//確認済み日時
		Property m_confirmDate =
			new Property(EventInfoConstant.CONFIRM_DATE, Messages.getString("confirm.time", locale), PropertyConstant.EDITOR_DATETIME);
		//確認ユーザ
		Property m_confirmUser = new Property(EventInfoConstant.CONFIRM_USER, Messages.getString("confirm.user", locale), PropertyConstant.EDITOR_TEXT);
		//重複カウンタ
		Property m_duplicationCount = 
			new Property(EventInfoConstant.DUPLICATION_COUNT, Messages.getString("duplication.count", locale), PropertyConstant.EDITOR_NUM);
		
		try{
			// イベントログ情報を取得
			EventLogPK pk = new EventLogPK(monitorId,
					pluginId,
					facilityId,
					new Timestamp(outputDate.getTime()));
			EventLogLocal event = EventLogUtil.getLocalHome().findByPrimaryKey(pk);
			
			// 値を初期化
			m_priority.setValue(PriorityConstant.typeToString(event.getPriority().intValue()));
			m_outputDate.setValue(event.getOutputDate());
			m_generationDate.setValue(event.getGenerationDate()); 
			m_pluginId.setValue(event.getPluginId());
			m_monitorId.setValue(event.getMonitorId());
			m_facilityId.setValue(event.getFacilityId());
			m_scopeText.setValue(event.getScopeText());
			m_application.setValue(event.getApplication());
			m_messageId.setValue(event.getMessageId());
			m_message.setValue(event.getMessage());
			m_messageOrg.setValue(this.nullToSpace(event.getMessageOrg()));
			m_confirmed.setValue(ConfirmConstant.typeToString(event.getConfirmFlg().intValue()));
			m_confirmDate.setValue(event.getConfirmDate());
			m_confirmUser.setValue(event.getConfirmUser());
			m_duplicationCount.setValue(event.getDuplicationCount());
			
		} catch (FinderException e) {
            AplLogger apllog = new AplLogger("MON", "mon");
            String[] args = {facilityId, monitorId, pluginId, outputDate.toString()};
            apllog.put("SYS", "006", args);
			m_log.debug("getProperty():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
            AplLogger apllog = new AplLogger("MON", "mon");
            String[] args = {facilityId, monitorId, pluginId, outputDate.toString()};
            apllog.put("SYS", "006", args);
			m_log.debug("getProperty():" + e.getMessage());
			throw e;
		}
		
		//変更の可/不可を設定
		m_priority.setModify(PropertyConstant.MODIFY_NG);
		m_outputDate.setModify(PropertyConstant.MODIFY_NG);
		m_generationDate.setModify(PropertyConstant.MODIFY_NG); 
		m_pluginId.setModify(PropertyConstant.MODIFY_NG);
		m_monitorId.setModify(PropertyConstant.MODIFY_NG);
		m_facilityId.setModify(PropertyConstant.MODIFY_NG);
		m_scopeText.setModify(PropertyConstant.MODIFY_NG);
		m_application.setModify(PropertyConstant.MODIFY_NG);
		m_messageId.setModify(PropertyConstant.MODIFY_NG);
		m_message.setModify(PropertyConstant.MODIFY_NG);
		m_messageOrg.setModify(PropertyConstant.MODIFY_NG);
		m_confirmed.setModify(PropertyConstant.MODIFY_NG);
		m_confirmDate.setModify(PropertyConstant.MODIFY_NG);
		m_confirmUser.setModify(PropertyConstant.MODIFY_NG);
		m_duplicationCount.setModify(PropertyConstant.MODIFY_NG); 

		Property property = new Property(null, null, "");
		
		// 初期表示ツリーを構成。
		property.removeChildren();
		property.addChildren(m_priority);
		property.addChildren(m_outputDate);
		property.addChildren(m_generationDate); 
		property.addChildren(m_pluginId);
		property.addChildren(m_monitorId);
		property.addChildren(m_facilityId);
		property.addChildren(m_scopeText);
		property.addChildren(m_application);
		property.addChildren(m_messageId);
		property.addChildren(m_message);
		property.addChildren(m_messageOrg);
		property.addChildren(m_confirmed);
		property.addChildren(m_confirmDate);
		property.addChildren(m_confirmUser);
		property.addChildren(m_duplicationCount);
		return property;
	}
	
	/**
	 * Nullを空文字へ変換
	 * 
	 * @param target
	 * @return
	 */
	private String nullToSpace(String target){
		
		if(target == null){
			return "";
		}
		return target;
	}
}
