/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.monitor.run.factory;

import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.jms.JMSException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.SchedulerException;

import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.MonitorNotFoundException;
import com.clustercontrol.NotifyNotFoundException;
import com.clustercontrol.accesscontrol.factory.AccessLock;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoPK;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoUtil;
import com.clustercontrol.monitor.run.ejb.entity.MonitorStringValueInfoLocal;
import com.clustercontrol.notify.ejb.entity.MonitorStatusLocal;
import com.clustercontrol.notify.ejb.entity.MonitorStatusUtil;
import com.clustercontrol.notify.ejb.entity.NotifyHistoryLocal;
import com.clustercontrol.notify.ejb.entity.NotifyHistoryUtil;
import com.clustercontrol.notify.ejb.session.NotifyControllerLocal;
import com.clustercontrol.notify.ejb.session.NotifyControllerUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 監視情報を削除する抽象クラス<BR>
 * <p>
 * 監視種別（真偽値，数値，文字列）の各クラスで継承してください。
 *
 * @version 3.0.0
 * @since 2.0.0
 */
abstract public class DeleteMonitor {
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( DeleteMonitor.class );
	
	/** 監視情報のローカルコンポーネント。 */
	protected MonitorInfoLocal m_monitor;
	
	/** 監視対象ID */
	protected String m_monitorTypeId;
	
	/** 監視項目ID */
	protected String m_monitorId;
	
	/**
	 * トランザクションを開始し、監視情報を削除します。
	 * 
	 * @param monitorTypeId 監視対象ID
	 * @param monitorId 監視項目ID
	 * @return 削除に成功した場合、</code> true </code>
	 * @throws RemoveException
	 * @throws SchedulerException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws SystemException
	 * @throws CreateException 
	 * @throws HinemosUnknownException 
	 * @throws NotifyNotFoundException 
	 * @throws MonitorNotFoundException 
	 * @see #deleteMonitorInfo()
	 */
	public boolean delete(String monitorTypeId, String monitorId) throws RemoveException, SchedulerException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, InvalidTransactionException, IllegalStateException, SystemException, CreateException, FinderException, NotifyNotFoundException, HinemosUnknownException, MonitorNotFoundException{
		
		//アクセスロック
		AccessLock.lock(AccessLock.MONITOR);
		
		m_monitorTypeId = monitorTypeId;
		m_monitorId = monitorId;
		
		TransactionManager tm = null;
//		Transaction tx = null;
//		Transactionの利用を取りやめたため、関連箇所ふくめコメントアウトした : uchiyama
		Transaction oldTx = null;
		boolean result = false;
		
		try 
		{
			// TransactionManagerを取得
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			
			// 実行中のスレッドがトランザクションに関連付けられている場合は、トランザクションを退避 
			if(tm.getTransaction() != null){
				oldTx = tm.suspend();
			}
			
			try{
				// トランザクション開始
				if(tm.getTransaction() == null){
					tm.begin();	
//					tx = tm.getTransaction();
				}
				
				// 監視情報を削除
				result = deleteMonitorInfo();
				
				if(result){
					// コミット
//					tx.commit();
					tm.commit();
				}
				
			} catch(NotSupportedException e){
				outputLog(e, "delete()");
				throw e;
			} catch (RollbackException e) {
				outputLog(e, "delete()");
				throw e;
			} catch (HeuristicMixedException e) {
				outputLog(e, "delete()");
				throw e;
			} catch(HeuristicRollbackException e){
				outputLog(e, "delete()");
				throw e;
			} catch(SystemException e){
				outputLog(e, "delete()");
				throw e;
			} catch (CreateException e) {
				outputLog(e, "delete()");
				throw e;
			}
			
			finally{
				// トランザクション関連の例外が発生した場合は、ロールバック
//				if(tx != null){
//					if(!result){
//						tx.rollback();
//					}
//					tx = null;
//				}
				if(tm.getTransaction() != null){
					if(!result){
						tm.rollback();
					}
				}
			}
		}
		finally{
			
			// 一時停止していたトランザクションを再開
			if(oldTx != null){
				try{
					tm.resume(oldTx);
					
				} catch(InvalidTransactionException e){
					outputLog(e, "delete()");
					throw e;
				} catch(IllegalStateException e){
					outputLog(e, "delete()");
					throw e;
				} catch(SystemException e){
					outputLog(e, "delete()");
					throw e;
				}
			}
		}
		return result;
	}
	
	/**
	 * 判定情報を削除します。
	 * <p>
	 * 各監視種別（真偽値，数値，文字列）のサブクラスで実装します。
	 * 
	 * @return 削除に成功した場合、</code> true </code>
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException 
	 * @throws CreateException 
	 * @throws HinemosUnknownException
	 * @throws NotifyNotFoundException 
	 */
	public boolean deleteJudgementInfo() throws RemoveException, NamingException, CreateException, FinderException, HinemosUnknownException, NotifyNotFoundException {
		
		if (MonitorTypeConstant.TYPE_STRING == m_monitor.getMonitorType()) {
			Collection ct = m_monitor.getMonitorStringValueInfo();
			deleteStringValueInfo(ct);
		} else {
			// removing child entity is not required （because of CMR DELETE CASCADE）
		}
		
		return true;
	}
	
	/**
	 * チェック条件情報を削除します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return 削除に成功した場合、</code> true </code>
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws NotifyNotFoundException 
	 */
	public abstract boolean deleteCheckInfo() throws FinderException, RemoveException, NamingException, HinemosUnknownException, NotifyNotFoundException;
	
	/**
	 * 監視情報を削除します。
	 * <p>
	 * <ol>
	 * <li>監視対象IDと監視項目IDより、監視情報を取得します。</li>
	 * <li>チェック条件情報を削除します。各監視管理のサブクラスで実装します（{@link #deleteCheckInfo()}）。</li>
	 * <li>判定情報を削除します。各監視種別（真偽値，数値，文字列）のサブクラスで実装します（{@link #deleteJudgementInfo()}）。</li>
	 * <li>監視情報を削除します。</li>
	 * <li>Quartzから監視情報を削除します。</li>
	 * </ol>
	 * 
	 * @return 削除に成功した場合、</code> true </code>
	 * @throws RemoveException
	 * @throws SchedulerException
	 * @throws NamingException
	 * @throws CreateException 
	 * @throws HinemosUnknownException 
	 * @throws NotifyNotFoundException 
	 * @throws MonitorNotFoundException 
	 * @throws JMSException 
	 * 
	 * @see com.clustercontrol.monitor.run.ejb.entity.MonitorInfoBean
	 * @see #deleteCheckInfo()
	 * @see #deleteJudgementInfo()
	 * @see com.clustercontrol.monitor.run.factory.ModifySchedule#deleteSchedule(String, String)
	 */
	public boolean deleteMonitorInfo() throws RemoveException, SchedulerException, NamingException, CreateException, NotifyNotFoundException, HinemosUnknownException, MonitorNotFoundException{
		
		try 
		{
			// 監視情報を取得
			MonitorInfoPK pk = new MonitorInfoPK(m_monitorId, m_monitorTypeId);
			m_monitor = MonitorInfoUtil.getLocalHome().findByPrimaryKey(pk);
			
			NotifyControllerLocal nc = NotifyControllerUtil.getLocalHome().create();
			
			// 監視グループ情報を削除
			nc.deleteNotifyRelation(m_monitor.getNotifyGroupId());
			
			// チェック条件情報を削除
			if(deleteCheckInfo()){
				
				// 判定情報を削除
				if(deleteJudgementInfo()){
					
					// 監視情報を削除
					m_monitor.remove();
					
					// Quartzから削除
					ModifySchedule quartz = new ModifySchedule();
					quartz.deleteSchedule(m_monitorTypeId, m_monitorId);

					// この監視設定の監視結果状態を削除する
					Collection<MonitorStatusLocal> statusList = 
						MonitorStatusUtil.getLocalHome().findByPluginIdAndMonitorId(m_monitorTypeId, m_monitorId);
					for(MonitorStatusLocal status : statusList){
						status.remove();
					}
					
					// この監視設定の結果として通知された通知履歴を削除する
					Collection<NotifyHistoryLocal> historyList = 
						NotifyHistoryUtil.getLocalHome().findByPluginIdAndMonitorId(m_monitorTypeId, m_monitorId);
					for(NotifyHistoryLocal history : historyList){
						history.remove();
					}
					
					return true;
				}
			}
			
			
		} catch (FinderException e) {
			outputLog(e, "deleteMonitorInfo()");
			throw new MonitorNotFoundException(e.getMessage(),e);
		} catch (RemoveException e) {
			outputLog(e, "deleteMonitorInfo()");
			throw e;
		} catch (SchedulerException e) {
			outputLog(e, "deleteMonitorInfo()");
			throw e;
		} catch (NamingException e) {
			outputLog(e, "deleteMonitorInfo()");
			throw e;
		} catch (CreateException e) {
			outputLog(e, "deleteMonitorInfo()");
			throw e;
		} 
		return false;
	}
	
	/**
	 * 引数で指定された文字列監視判定情報を削除します。
	 * 
	 * @param ct 文字列監視判定情報のコレクション
	 * @throws RemoveException
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 * @throws HinemosUnknownException 
	 * @throws NotifyNotFoundException 
	 * 
	 * @see com.clustercontrol.monitor.run.ejb.entity.MonitorStringValueInfoBean
	 */
	private void deleteStringValueInfo(Collection ct) throws RemoveException, CreateException, NamingException, FinderException, NotifyNotFoundException, HinemosUnknownException {
		
		
		NotifyControllerLocal nc;
		try {
			nc = NotifyControllerUtil.getLocalHome().create();
		
		
		
			// 文字列監視判定情報を削除
		    if(ct != null && ct.size() > 0){
		        Object[] infoArray = ct.toArray();
		        for(int j = 0; j < infoArray.length; j++){
		        	MonitorStringValueInfoLocal info = (MonitorStringValueInfoLocal)infoArray[j];
		            if(info != null){
		            	
		            	//通知設定の削除
						nc.deleteNotifyRelation(info.getNotifyGroupId());
		            }
		        }
		    }
	    
		} catch (CreateException e) {
			e.printStackTrace();
			throw e;
		} catch (NamingException e) {
			e.printStackTrace();
			throw e;
		}
	}
	
	/**
	 * アプリケーションログにログを出力します。
	 * 
	 * @param e 例外
	 * @param method メソッド名
	 */
	private void outputLog(Exception e, String method) {
		AplLogger apllog = new AplLogger("MON", "mon");
		String[] args = {m_monitorTypeId, m_monitorId };
		apllog.put("SYS", "009", args);
		m_log.debug(method + ":" + e.getMessage());
	}
}
