/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performance.ejb.session;

import java.io.UnsupportedEncodingException;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.UsedFacilityException;
import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.bean.PluginConstant;
import com.clustercontrol.commons.ejb.session.CheckFacility;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorDeviceInfoData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorDeviceInfoLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorDeviceInfoUtil;
import com.clustercontrol.performance.util.FacilityTreeDB;
import com.clustercontrol.performance.util.JNDIConnectionManager;
import com.clustercontrol.performance.util.LockManager;
import com.clustercontrol.performance.util.Repository;
import com.clustercontrol.performance.bean.CollectedDataInfo;
import com.clustercontrol.performance.bean.CollectedDataSet;
import com.clustercontrol.performance.bean.CollectorItemInfo;
import com.clustercontrol.performance.bean.CollectorLockKey;
import com.clustercontrol.performance.bean.CollectorProperty;
import com.clustercontrol.performance.bean.CollectorRunStatus;
import com.clustercontrol.performance.bean.CollectorType;
import com.clustercontrol.performance.bean.DeviceDataSet;
import com.clustercontrol.performance.dao.CalculatedDataDAO;
import com.clustercontrol.performance.dao.CollectorItemDAO;
import com.clustercontrol.performance.ejb.bmp.CollectorLocal;
import com.clustercontrol.performance.ejb.bmp.CollectorLocalHome;
import com.clustercontrol.performance.ejb.bmp.CollectorPK;
import com.clustercontrol.performance.ejb.bmp.CollectorUtil;
import com.clustercontrol.performance.ejb.bmp.RecordCollectorData;
import com.clustercontrol.performance.util.code.CollectorItemCodeTable;
import com.clustercontrol.performance.util.code.CollectorItemTreeItem;
import com.clustercontrol.repository.util.RepositoryUpdateTime;

/**
 *　性能管理機能の管理を行うコントローラクラス
 * クライアントからの Entity Bean へのアクセスは、このSession Bean を介して行います。
 * 
 * @version 1.0
 * @since 1.0
 *
 * <!-- begin-xdoclet-definition --> 
 * @ejb.bean name="CollectorController"	
 *           jndi-name="CollectorController"
 *           type="Stateful" 
 *           transaction-type="Container"
 *           
 * <!-- @jboss.clustered -->
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorStatusLocalEJB"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorCalcMethodMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorCategoryCollectMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorCategoryMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorDeviceInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorItemCalcMethodMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorItemCodeMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorPollingMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorScopeRelationLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorScopeSnapLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=DeviceTypeMstLocal"
 *           
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 * <!-- end-xdoclet-definition --> 
 * @generated
 */
public abstract class CollectorControllerBean implements javax.ejb.SessionBean, CheckFacility {
	
	//	ログ出力
	protected static Log m_log = LogFactory.getLog(CollectorControllerBean.class);
	
	// コンテキスト情報。
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
//	/** 収集方法 */ ver3.1より使用しない
//	public static final String COLLECT_METHOD = PollerProtocolConstant.PROTOCOL_SNMP;
	
	private String m_collectorID;  // 収集ID
	private int m_collectorType;   // 収集種別（リアルタイム もしくは 実績）

	// 実績収集で1行あたりに必要な物理サイズ容量（Byte）
	// サイズ見積り時に使われる（ヘッダサイズ ＋ timestamp + double precision）
	private static long _fixRowSize = 24 + 8 + 8;

	// DBの文字セット
	private String DB_CHARSET_NAME ="UTF8";
	
	// リポジトリ情報の最終更新時刻
	private static long _lastRepositoryUpdateTime = RepositoryUpdateTime.getUpdateTime().getTime();
	private static final Object _refreshScopeTreeLock = new Object();

	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean の生成時に行う処理を実装します。
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}

	/**
	 * SessionBean生成時にコンテナから呼ばれます。
	 *
	 * <!-- begin-user-doc -->
	 * The  ejbCreate method.
	 * <!-- end-user-doc -->
	 *
	 * <!-- begin-xdoclet-definition --> 
	 * @ejb.create-method 
	 * <!-- end-xdoclet-definition --> 
	 * @generated
	 */
	public void ejbCreate() throws CreateException {
		// 収集のステータスを設定
		this.m_collectorType = CollectorType.UNKNOWN;		
	}
	
	/**
	 * SessionBean削除時にコンテナから呼ばれます。
	 **/
	public void ejbRemove() {
		m_log.debug("ejbRemove() start : " + m_collectorID);
		
		// リアルタイム収集中の場合
		if(this.m_collectorType == CollectorType.REALTIME){
			try {			
				CollectorLocalHome home = JNDIConnectionManager.getCollectorLocalHome();
				CollectorLocal myBean = home.findByPrimaryKey(new CollectorPK(m_collectorID));
				
				// 収集を停止
				myBean.stopCollect();
				
				// DBの収集設定を削除
				myBean.remove();
			} catch (FinderException e) {
				throw new EJBException(e.getMessage(), e);
			} catch (NamingException e) {
				throw new EJBException(e.getMessage(), e);
			} catch (EJBException e) {
				throw new EJBException(e.getMessage(), e);
			} catch (RemoveException e) {
				throw new EJBException(e.getMessage(), e);
			}
		}
		m_log.debug("ejbRemove() end : " + m_collectorID);
	}
	
	/**
	 * SessionBeanインスタンスが「非活性化」状態になる前にコンテナから呼ばれます。
	 **/
	public void ejbPassivate(){
	}
	
	/**
	 * リアルタイム収集用のEntityBeanを生成します。
	 * 
	 * @param facilityID ファシリティID
	 * @param interval 収集間隔
	 * @param colletorItemList　収集項目設定のリスト
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceExecute"
     *  method-intf="Remote"
     *  
	 **/
	public void createRealtimeColletor(String facilityID, int interval, List<CollectorItemInfo> colletorItemList) {
		m_collectorType = CollectorType.REALTIME;
		
		// 収集IDを生成して設定
		String collectorID = (CollectorUtil.generateGUID(this));
		
		// Stateful Session Bean のためIDを保持することができる
		m_collectorID = collectorID;
		
		// デバッグ出力
		m_log.debug("createRealtimeColletor() start : " + 
				Thread.currentThread().getName() + "\t" + collectorID);
		
		try {
			RecordCollectorData data = new RecordCollectorData();
			data.setCollectorID(collectorID);
			data.setCollectorType(CollectorType.REALTIME);
			data.setLabel(Thread.currentThread().getName());  // スレッド名を説明項目に入力
			data.setFacilityID(facilityID);
			data.setIntervalSec(interval);
			data.setPeriodMin(-1);  // 収集期間は無期限を指定

			// スコープツリーを取得
			Repository repository = new Repository();
			FacilityTreeItem facilityTree = repository.getFacilityTree(facilityID);
			data.setFacilityTree(facilityTree);

			// 対象スコープが存在しない場合はエラー
			if(facilityTree == null){
				// エラー処理
				return;				
			}
			
			// RealtimeCollector Entity Bean を生成
			CollectorLocalHome home = JNDIConnectionManager.getCollectorLocalHome();
			CollectorLocal myBean = home.create(data, colletorItemList);
			
			// 収集の開始を指示
			myBean.startCollect();
		} catch (CreateException e) {
			m_log.warn("createRealtimeColletor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new EJBException(e);
		} catch (NamingException e){
			m_log.warn("createRealtimeColletor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new EJBException(e);
		}
		
		m_log.debug("createRealtimeColletor() end : " + Thread.currentThread().getName() + "\t" + collectorID);
	}
	
	/**
	 * 指定スコープのリアルタイム収集の収集済みデータを取得します。
	 * @param facilityID ファシリティID
	 * @param itemID　収集項目ID
	 * @return　収集済み性能データ（CollectedDataInfo）のリスト
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceExecute"
     *  method-intf="Remote"
     *  
	 */
	public Collection<CollectedDataInfo> getRealtimeCollectedData(
			String facilityID, 
			String itemCode,
			String displayName) { 
		try {
			CollectorLocalHome home = JNDIConnectionManager.getCollectorLocalHome();
			CollectorLocal myBean = home.findByPrimaryKey(new CollectorPK(m_collectorID));
			
			// 性能値を取得
			CollectedDataInfo data = myBean.getValue(
					facilityID, 
					new CollectorItemInfo(
							m_collectorID,
							itemCode,
							displayName
							));
			ArrayList<CollectedDataInfo> ret = new ArrayList<CollectedDataInfo>();
			ret.add(data);
			return ret;
		} catch (FinderException e) {
			throw new EJBException(e);
		} catch (NamingException e){
			throw new EJBException(e);
		}
	}

	/**
	 * 指定スコープのサブスコープのリアルタイム収集済みデータを取得します。
	 * @param facilityID ファシリティID
	 * @param itemID　収集項目ID
	 * @return　収集済み性能データ（CollectedDataInfo）のリスト
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceExecute"
     *  method-intf="Remote"
     *  
	 */
	public Collection<CollectedDataInfo> getSubScopeRealtimeCollectedData(
			String facilityID, 
			String itemCode,
			String displayName) { 
		try {
			CollectorLocalHome home = JNDIConnectionManager.getCollectorLocalHome();
			CollectorLocal myBean = home.findByPrimaryKey(new CollectorPK(m_collectorID));
			
			// 性能値を取得
			return myBean.getSubScopeValues(
					facilityID, 
					new CollectorItemInfo(
							m_collectorID,
							itemCode,
							displayName
							));
		} catch (FinderException e) {
			throw new EJBException(e);
		} catch (NamingException e){
			throw new EJBException(e);
		}
	}
	
	
	/**
	 * リアルタイム収集の収集済みデータを取得します。
	 * 指定のファシリティで収集されている全ての収集項目の性能値データを取得します。
	 * 
	 * @param facilityID　ファシリティID
	 * @return 収集済み性能データ（CollectedDataInfo）のリスト
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceExecute"
     *  method-intf="Remote"
	 *
	 **/
	public Collection<CollectedDataInfo> getRealtimeCollectedDataAll(String facilityID) {
		try {
			CollectedDataInfo[] returnData = null;
			
			CollectorLocalHome home = JNDIConnectionManager.getCollectorLocalHome();
			CollectorLocal myBean = home.findByPrimaryKey(new CollectorPK(m_collectorID));
			
			// 性能値を取得
			returnData = myBean.getValueAll(facilityID);
			return Arrays.asList(returnData); 
		} catch (FinderException e) {
			throw new EJBException(e);
		} catch (NamingException e){
			throw new EJBException(e);
		}
	}
	
	/**
	 * 収集設定のプロパティ情報を取得します（指定の収集IDが存在しない場合はnullを返す）。
	 * 
	 * @param collectorID 収集ID
	 * @return 収集設定のプロパティ情報
	 * 
	 * @ejb.interface-method
	 *	view-type="remote"
	 *
	 *	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
	 *
	 **/
	public CollectorProperty getCollectorProperty(String collectorID) { 
		// 収集IDを設定
		m_collectorID = collectorID;
		
		CollectorProperty property = new CollectorProperty();
		
		RecordCollectorData collectorData = null;
		
		try {
			// Entity Bean を取得する
			CollectorLocalHome home = JNDIConnectionManager.getCollectorLocalHome();
			CollectorLocal myBean = home.findByPrimaryKey(new CollectorPK(collectorID));
			
			// 収集定義を取得する
			collectorData = myBean.getRecordColletorData();
			
			// デバイス情報を設定
			property.setDeviceSet(getDeviceSet(collectorID));
			
			// スコープツリー情報を設定
			collectorData.setFacilityTree(this.getFacilityTree(collectorID));
			
			// 収集設定を格納
			property.setCollectorData(collectorData);
			
			// 収集項目用のDAOを生成
			CollectorItemDAO dao = new CollectorItemDAO();
			
			
			ArrayList<CollectorItemInfo> itemList = new ArrayList(
					dao.selectCollectorItemByCollectorID(collectorID)
			);
			
			property.setItemList(itemList);
						
			return property;
		} catch (FinderException e) {
			// 指定の収集IDが存在しない場合はnullを返す
			return null;
		} catch (NamingException e){
			throw new EJBException(e);
		}
	}
	
	/**
	 * 実績情報収集用のEntityBeanを生成し、収集を開始します。
	 * @param collectorDef　収集の定義情報
	 * 
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceWrite"
     *  method-intf="Remote"
	 *
	 **/
	public void createRecordColletor(CollectorProperty collectorDef)
	throws DuplicateKeyException {
		// デバッグ出力
		m_log.debug("createRecordColletor() start : " + Thread.currentThread().getName() + "\t" + m_collectorID);
		
		// 収集種別を設定
		this.m_collectorType = CollectorType.RECORD;
		
		// このBeanに収集IDを設定
		m_collectorID = collectorDef.getCollectorData().getCollectorID();
		
		RecordCollectorData collectorData = collectorDef.getCollectorData();
		collectorData.setCollectorType(m_collectorType);
		
		// 設定の書き込みとEntity Beanの生成
		try {
			// 収集の開始を指示
			CollectorLocalHome home = JNDIConnectionManager.getCollectorLocalHome();
			CollectorLocal collector = home.create(collectorData, collectorDef.getItemList());
			
			collector.startCollect();
		} catch (DuplicateKeyException e) {
			m_log.warn("createRecordColletor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;  // キーが重複した場合はクライアントまで投げる
		} catch (CreateException e) {
			m_log.warn("createRecordColletor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new EJBException(e);
		} catch (NamingException e){
			m_log.warn("createRecordColletor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new EJBException(e);
		}
		
		m_log.debug("createRecordColletor() end : " + Thread.currentThread().getName()
				+ "\t" + collectorDef.getCollectorData().getCollectorID());
	}
	
	/**
	 * 実績収集を停止します。
	 * 
	 * @param collectorID 停止したい収集ID
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceExecute"
     *  method-intf="Remote"
	 *
	 */
	public void stopRecordCollector(String collectorID) {
		
		try {
			CollectorLocalHome home = JNDIConnectionManager.getCollectorLocalHome();
			CollectorLocal myBean = home.findByPrimaryKey(new CollectorPK(collectorID));
			
			// 収集の停止を指示
			myBean.stopCollect();
		} catch (FinderException e) {
			m_log.warn("stopRecordCollector() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new EJBException(e);
		} catch (NamingException e){
			m_log.warn("stopRecordCollector() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new EJBException(e);
		}
	}
	
	/**
	 * 実績収集で収集されたデータを取得します。
	 * @param collectorID 収集ID
	 * @param facilityID　ファシリティID
	 * @param item　収集項目
	 * @param startDate　取得したい始点の時刻
	 * @param endDate　　取得した終点の時刻
	 * @return　収集済み性能データ（CollectedDataInfo）のリスト
	 *  
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
	 *
	 */
	public List<CollectedDataInfo> getRecordCollectedData(
			CollectorItemInfo item, 
			String facilityID, 
			Date startDate, 
			Date endDate) {
		m_log.debug("getRecordCollectedData() " + facilityID);
		CalculatedDataDAO dao = new CalculatedDataDAO();
		
		List<CollectedDataInfo> data = dao.select(
				item.getCollectorId(), 
				item.getItemCode(), 
				item.getDisplayName(),
				facilityID,
				startDate, 
				endDate);
		
		return data;
	}
	
	/**
	 * 実績収集結果を収集ID、ファシリティID、検索期間（開始日時および終了日時）の組み合わせで検索し、収集性能値DTOクラスの配列として返す。
	 * 
	 * @param collectorID
	 *            収集ID
	 * @param facilityID
	 *            ファシリティID
	 * @param startDate
	 *            検索対象となる開始日時
	 * @param endDate
	 *            検索対象となる終了日時
	 * @return 該当する収集性能値DTOクラスの配列（順序は収集日時順とは限らない）
	 * 
	 * @ejb.interface-method view-type="remote"
	 * 
	 * @ejb.permission role-name="PerformanceRead" method-intf="Remote"
	 * 
	 * @author NTT DATA takahatat
	 */
	public List<CollectedDataInfo> getRecordCollectedData(
			String collectorID, 
			String facilityID, 
			Date startDate,
			Date stopDate) {

		m_log.debug("getRecordCollectedData() " + facilityID);
		CalculatedDataDAO dao = new CalculatedDataDAO();

		ArrayList<CollectedDataInfo> data = dao.select(collectorID, facilityID, startDate, stopDate);

		return data;
	}
	
//	/**
//	 * 実績収集で収集されたデータを取得します。
//	 * @param collectorID 収集ID
//	 * @param facilityIDs　ファシリティIDの配列
//	 * @param items　収集項目の配列
//	 * @param startDate　取得したい始点の時刻
//	 * @param endDate　　取得した終点の時刻
//	 * @return　収集データのデータセット
//	 *  
//	 * @ejb.interface-method
//	 *	view-type="remote" 
//	 *
//	 * @ejb.permission
//     *  role-name="PerformanceRead"
//     *  method-intf="Remote"
//	 *
//	 */
//	public CollectedDataSet getRecordCollectedData(
//			CollectorItemInfo[] items, 
//			String[] facilityIDs, 
//			Date startDate, 
//			Date endDate) {
//		CollectedDataSet ret = new CollectedDataSet();
//
//		for(int j=0; j<facilityIDs.length; j++){
//			for(int i=0; i<items.length; i++){
//				List<CollectedDataInfo> dataList = getRecordCollectedData(
//						items[i], 
//						facilityIDs[j],
//						startDate,
//						endDate);
//				ret.setCollectedDataList(facilityIDs[j], items[i], dataList);
//			}
//		}
//		
//		return ret;
//	}
	
	/**
	 * 実績収集で収集されたデータを取得します。
	 * @param facilityID　ファシリティIDの配列
	 * @param itemInfos　収集項目の配列
	 * @param startDate　取得したい始点の時刻
	 * @param endDate　　取得した終点の時刻
	 * @return　収集データのデータセット
	 *  
	 * @ejb.interface-method
	 *  view-type="remote"
	 *
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
	 *
	 */
	public CollectedDataSet getRecordCollectedData(
			String[] facilityIDs, 
			CollectorItemInfo[] itemInfos, 
			Date startDate, 
			Date endDate) {
		CollectedDataSet ret = new CollectedDataSet();

//		CollectorItemDAO dao = new CollectorItemDAO();
		
		for(int j=0; j<facilityIDs.length; j++){
			for(int i=0; i<itemInfos.length; i++){
				// 収集項目をDBから取得します
//				CollectorItemInfo item = dao.selectCollectorItem(collectorId, itemInfos[i].getItemCode(), itemInfos[i].getDeviceName());
				
				List<CollectedDataInfo> dataList = getRecordCollectedData(
						itemInfos[i], 
						facilityIDs[j],
						startDate,
						endDate);
				ret.setCollectedDataList(facilityIDs[j], itemInfos[i], dataList);
			}
		}
		
		return ret;
	}
	
	/**
	 * 実績収集設定及び、収集した性能情報を全て削除します。
	 * @param collectorID　削除したい収集ID
	 * 
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceWrite"
     *  method-intf="Remote"
	 *
	 **/
	public void deleteRecordCollector(String collectorID) throws RemoveException {
		// デバッグ出力
		m_log.debug("deleteRecordCollector() start : " + Thread.currentThread().getName() + "\t" + collectorID);
		
		// 削除ロックされているか否かをチェック
		if(LockManager.isLocked(collectorID)){
			m_log.debug("deleteRecordCollector()  : " + collectorID + " is locked.");
			throw new RemoveException(collectorID + " is locked.");
		}
		
		try {
			// 指定のCollectorIDのEntityBeanを取得
			CollectorLocalHome home = JNDIConnectionManager.getCollectorLocalHome();
			CollectorLocal myBean = home.findByPrimaryKey(new CollectorPK(collectorID));
			
			// まだ動作中の場合
			if(myBean.getStatus() == CollectorRunStatus.RUNNING){
				m_log.debug("deleteRecordCollector()  : " + collectorID + " is running.");
				throw new RemoveException(collectorID + " is running.");
			}
			
			myBean.remove();
		} catch (FinderException e) {
			m_log.warn("deleteRecordCollector() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new RemoveException(e.getMessage());
		} catch (NamingException e){
			m_log.warn("deleteRecordCollector() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new RemoveException(e.getMessage());
		} catch (RemoveException e) {
			m_log.warn("deleteRecordCollector() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
		
		m_log.debug("deleteRecordCollector() end : " + Thread.currentThread().getName() + "\t" + collectorID);
	}
	
	/**
	 * 収集設定時のファシリティツリー情報を取得します。
	 * @param collectorID 収集ID
	 * @return FacilityTreeItem ファシリティツリー
	 * 
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
	 *
	 **/
	public FacilityTreeItem getFacilityTree(String collectorID){
		return new FacilityTreeDB().getFacilityTree(collectorID);
	}
	
	/**
	 * 現在のリポジトリに登録されているデバイス情報から
	 * 指定のファシリティID以下の階層のノードのデバイスのリストを取得します。
	 * @param ファシリティID
	 * @return デバイス情報のセット
	 * 
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
	 *
	 **/
	public DeviceDataSet getCurrentDeviceSet(String facilityID) {
		DeviceDataSet dataSet = new DeviceDataSet();
		
		Repository repository = new Repository();
		List<CollectorDeviceInfoData> deviceList = repository.getDeviceList(facilityID);
		Iterator<CollectorDeviceInfoData> itr = deviceList.iterator();
		
		// デバイス情報を設定
		while(itr.hasNext()){
			dataSet.putDeviceList(itr.next());
		}
		
		return dataSet;
	}

	/**
	 * 収集設定に含まれたファシリティのデバイス情報を取得します。
	 * @param collectorID　収集ID
	 * @return デバイス情報のセット
	 * 
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
	 *
	 **/
	public DeviceDataSet getDeviceSet(String collectorId)
	throws FinderException, NamingException {
		DeviceDataSet dataSet = new DeviceDataSet();
		
		// デバイス情報をDBから取得する
		Collection<CollectorDeviceInfoLocal> col = 
			CollectorDeviceInfoUtil.getLocalHome().findByCollectorId(collectorId);
		
		// デバイス情報を設定
		Iterator<CollectorDeviceInfoLocal> itr = col.iterator();
		while(itr.hasNext()){
			CollectorDeviceInfoLocal bean = itr.next();
			dataSet.putDeviceList(
					new CollectorDeviceInfoData(
							bean.getCollectorId(),
							bean.getFacilityId(),
							bean.getDeviceName(),
							bean.getDisplayName(),
							bean.getIndex(),
							bean.getDeviceId(),
							bean.getDeviceType()
					));
		}
		
		return dataSet;
	}
	
	/**
	 * 指定の収集IDのデータに対して削除ロックをかけます。
	 * 
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
     *  unchecked="true"
     *  method-intf="Remote"
	 *
	 **/
	public void lock(String collectorID, CollectorLockKey key) {
		LockManager.lock(collectorID, key);
	}

	/**
	 * 指定の収集IDのデータに対する削除ロックを解除します。
	 * 
	 * @ejb.interface-method
	 *	view-type="remote"
	 *
	 * @ejb.permission
     *  unchecked="true"
     *  method-intf="Remote"
	 *
	 **/
	public void unlock(CollectorLockKey key){
		LockManager.unlock(key);
	}
	
	
	/**
	 * 実績収集のリストを返します。
	 * 
	 * 戻り値のArrayListはArrayLisyのArrayListとなる。
	 *　実行状態、ステータス、収集ID、説明、スコープのパス、収集開始時刻、収集終了時刻、収集間隔、収集回数
	 *　が格納されているArrayListのリストとなる。
	 *
	 * 下記のような構造。
	 *　インデックス1:実行状態、ステータス、収集ID、説明、スコープのパス、収集開始時刻、収集終了時刻、収集間隔、収集回数
	 *　インデックス2:実行状態、ステータス、収集ID、説明、スコープのパス、収集開始時刻、収集終了時刻、収集間隔、収集回数
	 *　インデックス3:実行状態、ステータス、収集ID、説明、スコープのパス、収集開始時刻、収集終了時刻、収集間隔、収集回数
	 *　インデックス4:実行状態、ステータス、収集ID、説明、スコープのパス、収集開始時刻、収集終了時刻、収集間隔、収集回数
	 * …
	 * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
	 * 
	 * @return 実績収集の一覧（Objectの2次元配列）
	 * @throws NamingException 
	 * @throws FinderException 
	 */
	public ArrayList<ArrayList<Object>> getCollectionList() {
		m_log.debug("getCollectionList() start : ");
		
		ArrayList<ArrayList<Object>> list = new ArrayList<ArrayList<Object>>();
		
		try {
			// 収集種別が実績であるEntityBeanを取得
			CollectorLocalHome home = JNDIConnectionManager.getCollectorLocalHome();
			Collection<CollectorLocal> myBeans = home.findByCollectorType(CollectorType.RECORD);
			
			Iterator<CollectorLocal> itr = myBeans.iterator();
			while(itr.hasNext())
			{
				CollectorLocal myBean = itr.next();
				
				// 収集設定を取得する
				RecordCollectorData data = myBean.getRecordColletorData();
				
				ArrayList<Object> info = new ArrayList<Object>();
				info.add(new Integer(data.getStatus()));
				info.add(new Integer(data.getEndStatus()));
				info.add(data.getCollectorID());
				info.add(data.getLabel());
				info.add(data.getScopePath());
				info.add(data.getStartDate());
				if(data.getStopDate() == null){
					// まだ終了していない場合
					// 期限指定の場合は、終了予定時刻を表示
					// 無期限収集の場合は、何も表示しない
					
					// 無期限収集の場合は、終了時刻を表示しない
					if(data.getPeriodMin() == -1){
						info.add(null);
					} else {
						// 終了予定時刻を設定
						info.add(data.getScheduledStopDate());
					}
				} else {
					// 終了時刻を設定
					info.add(data.getStopDate());
				}
				info.add(data.getIntervalSec());
				info.add(data.getCount());
				list.add(info);
			}
		} catch (FinderException e) {
			throw new EJBException(e);
		} catch (NamingException e) {
			throw new EJBException(e);
		}
		
		m_log.debug("getCollectionList() end   : ");
		return list;
	}
	
	/**
	 * 収集項目コードの一覧を取得します
	 * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @ejb.permission
     *  unchecked="true"
     *  method-intf="Remote"
     *   
	 * @return 収集項目IDをキーとしCollectorItemTreeItemが格納されているHashMap
	 */
	public HashMap<String, CollectorItemTreeItem> getItemCodeMap(){
		return CollectorItemCodeTable.getItemCodeMap();
	}
	
	/**
	 * 収集項目ツリーを返します
	 * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
	 * 
	 * @param facilityId ファシリティID
	 * @param collectorId 収集ID
	 * @return 収集項目ツリー（GUIで収集項目をカテゴリ毎にツリー構造で表示するためのデータ構造）
	 */
    public CollectorItemTreeItem[] getCollectorTree(String facilityId, String collectorId) {
    	return CollectorItemCodeTable.getCollectorTree(facilityId, collectorId);
    }
    
    /**
     * 指定のファシリティで収集可能な項目のリストを返します
     * デバイス別の収集項目があり、ノード毎に登録されているデバイス情報が異なるため、
     * 取得可能な収集項目はファシリティ毎に異なる。
     * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
     *
     * @param facilityId ファシリティID
     * @return 指定のファシリティで収集可能な項目のリスト
     */
    public List<CollectorItemInfo> getAvailableCollectorItemList(String facilityId) {
    	ArrayList<CollectorItemInfo> itemInfo = new ArrayList<CollectorItemInfo>();
    	
    	CollectorItemTreeItem[] itemTree = 
    		CollectorItemCodeTable.getCollectorTree(facilityId, null);

    	for(int i=0; i<itemTree.length; i++){
    		CollectorItemTreeItem item = itemTree[i];

   			// 自身をリストにセットし子要素も再帰的にリストにセットする
   			setAvailableCollectorItemList(item, itemInfo);
    	}

    	return itemInfo;
    }
    
    private void setAvailableCollectorItemList(
    		CollectorItemTreeItem item, 
    		List<CollectorItemInfo> itemInfoList){

    	// カテゴリは追加しない
    	if(item.isCategoryItem() == false){   	
    		// 自身をリストに追加
    		String displayName = null;
    		CollectorDeviceInfoData deviceData = item.getDeviceData();
    		if(deviceData != null){
    			displayName = deviceData.getDisplayName();
    		}
    		else {
    			displayName = "";
    		}
    		
    		itemInfoList.add(
    				new CollectorItemInfo(
    						null, //collectorId
    						item.getItemCode(), 
    						displayName));
    	}
    	
		// 子要素を再帰的にセットする
    	// 子要素を格納するリスト
    	List<CollectorItemInfo> childrenList = new ArrayList<CollectorItemInfo>();
    	
		List<CollectorItemTreeItem> children = item.getChildren();
		Iterator<CollectorItemTreeItem> itr = children.iterator();
		while(itr.hasNext()){
			CollectorItemTreeItem childItem = itr.next();
			setAvailableCollectorItemList(childItem, childrenList);
		}
		
		// 項目コード順にソート
		CollectorItemInfo[] list = childrenList.toArray(new CollectorItemInfo[0]);
		Arrays.sort(list, new Comparator<CollectorItemInfo>(){
			public int compare(CollectorItemInfo o1, CollectorItemInfo o2) {
				//比較
				return o1.getItemCode().compareTo(o2.getItemCode());
			}
		});
		
		itemInfoList.addAll(Arrays.asList(list));
	}

    /**
     * アクセス可能かどうかをチェックするためのメソッド
     * 読み込み権限のあるroleで呼び出すとtrueを返す
     * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @ejb.permission
     *  role-name="PerformanceRead"
     *  method-intf="Remote"
     *
     * @return true
     */
    public boolean checkPermissionRead() {
    	return true;
    }
    
    /**
     * アクセス可能かどうかをチェックするためのメソッド
     * 書き込み権限のあるroleで呼び出すとtrueを返す
     * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @ejb.permission
     *  role-name="PerformanceWrite"
     *  method-intf="Remote"
     *
     * @return true
     */
    public boolean checkPerformanceWrite() {
    	return true;
    }
    
    /**
     * アクセス可能かどうかをチェックするためのメソッド
     * 書き込み権限のあるroleで呼び出すとtrueを返す
     * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @ejb.permission
     *  role-name="PerformanceExecute"
     *  method-intf="Remote"
     *
     * @return true
     */
    public boolean checkPermissionExecute() {
    	return true;
    }
    
	/**
	 * 指定のファシリティIDが収集設定で使用されているかチェックします。
	 * <P>
	 * 使用されていたら、UsedFacilityException がスローされる。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="PerformanceRead"
	 *     method-intf="Remote"
	 *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param facilityId ファシリティ
	 * @throws UsedFacilityException
	 * 
	 * @see com.clustercontrol.commons.ejb.session.CheckFacility#isUseFacilityId(java.lang.String)
	 * @see com.clustercontrol.bean.PluginConstant;
	 */
	public void isUseFacilityId(String facilityId) throws UsedFacilityException {
		try {
			Collection ct = null;
			
			// ファシリティIDが使用されている、収集種別がリアルタイムであるEntityBeanを取得する。
			CollectorLocalHome home = JNDIConnectionManager.getCollectorLocalHome();
			ct = home.findByCollectorTypeAndFacilityId(CollectorType.REALTIME, facilityId);
			if(ct != null && ct.size() > 0) {
				throw new UsedFacilityException(PluginConstant.TYPE_PERFORMANCE);
			}
			
			// ファシリティIDが使用されている、収集種別が実績であるEntityBeanを取得する。
			ct = null;
			ct = home.findByCollectorTypeAndFacilityId(CollectorType.RECORD, facilityId);
			if(ct != null && ct.size() > 0) {
				Iterator itr = ct.iterator();
				while(itr.hasNext()){
					CollectorLocal collector = (CollectorLocal)itr.next();
					// 収集中の場合、UsedFacilityExceptionをスロー
					if(collector.getStatus() == CollectorRunStatus.RUNNING){
						throw new UsedFacilityException(PluginConstant.TYPE_PERFORMANCE);
					}
				}
			}
			
		} catch (FinderException e) {
			m_log.debug("isUseFacilityId() : " + e.getMessage());
		} catch (NamingException e) {
			m_log.debug("isUseFacilityId() : " + e.getMessage());
		}
	}
	
	/**
	 * 実績情報収集用のEntityBeanを生成し、収集を一時停止します。
	 * @param collectorId　収集ID
	 * 
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
	 *  role-name="PerformanceWrite"
	 *  method-intf="Remote"
	 *
	 **/
	public void suspendCollect(String collectorId){
		m_log.debug("suspendCollect() start");
		try {
			CollectorLocalHome home = JNDIConnectionManager.getCollectorLocalHome();
			CollectorLocal myBean = home.findByPrimaryKey(new CollectorPK(collectorId));
			
			// 収集の停止を指示
			myBean.suspendCollect();
		} catch (FinderException e) {
			m_log.warn("suspendCollect() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new EJBException(e);
		} catch (NamingException e){
			m_log.warn("suspendCollect() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new EJBException(e);
		}
		m_log.debug("suspendCollect() end");
	}
	
	/**
	 * 実績情報収集用のEntityBeanを生成し、収集を再開します。
	 * @param collectorId　収集ID
	 * 
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
	 *  role-name="PerformanceWrite"
	 *  method-intf="Remote"
	 *
	 **/
	public void restartCollect(String collectorId){
		m_log.debug("restartCollect() start");
		try {
			CollectorLocalHome home = JNDIConnectionManager.getCollectorLocalHome();
			CollectorLocal myBean = home.findByPrimaryKey(new CollectorPK(collectorId));
			
			// 収集の停止を指示
			myBean.restartCollect();
		} catch (FinderException e) {
			m_log.warn("restartCollect() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new EJBException(e);
		} catch (NamingException e){
			m_log.warn("restartCollect() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new EJBException(e);
		}
		m_log.debug("restartCollect() end");
	}

	/**
	 * 指定の収集を設定した際の実績収集データのサイズを見積ります。
	 * （見積りだけで実際に収集は設定されません）
	 * 
	 * @return 見積り結果（見積りに失敗した場合は、-1を返します）
	 * 
	 * @ejb.interface-method
	 *	view-type="remote" 
	 *
	 * @ejb.permission
	 *  role-name="PerformanceWrite"
	 *  method-intf="Remote"
	 */
	public long getEstimation(CollectorProperty property){
		// 収集設定を取得する
		RecordCollectorData data = property.getCollectorData();
		
		// 収集期間
		long period;
		
		// 期限指定の収集の場合
		if(!(data.getPresavePeriod() > 0)){
			period = data.getPeriodMin() * 60;
		} else {
			// 無期限収集の保存期間（日単位のため秒単位に換算）
			// 削除処理は1日に1度のみの実行であるため、保存期間+1日分のサイズを見積もる必要がある
			period = (data.getPresavePeriod() + 1) * 24 * 60 * 60;
		}
		
		// 指定期間の最後まで収集された場合に収集される予定の回数
		long maxCount = period / data.getIntervalSec();

		// 1回の収集で蓄積されるデータサイズを算出
		long emstimationSize = getEstimationSizePerCollect(property);
		if(emstimationSize == -1){
			// 算出出来なかった場合は-1を返す
			return -1l;
		}
		long totalEmstimationSize = emstimationSize * maxCount;
		
		m_log.debug("CollectorId=" + data.getCollectorID()
				+ ", MaxCount=" + maxCount
				+ ", EstimationSize=" + totalEmstimationSize);
		
		return totalEmstimationSize;
	}

	/**
	 * 指定の収集の1回あたりの実績収集データのサイズを見積ります。
	 * テーブルサイズとインデックスサイズの合計の値となります。
	 * 
	 * @return 見積り結果（見積りに失敗した場合は、-1を返します）
	 */
	private long getEstimationSizePerCollect(CollectorProperty property){
		// 一項目あたりの見積もりサイズ
		long totalItemSize = 0;
		int blockSize = 8; // 最小ブロック単位
		
		// 収集項目設定分のサイズを求める
		// 収集項目（CollectorItemInfo）毎のcollectorIdは、登録前は設定されていないため、
		// getCollectorData()で取得した、CollectorIDを持ちること。
		String collectorId = property.getCollectorData().getCollectorID();
		try {
			int collectorIdSize = collectorId.getBytes(DB_CHARSET_NAME).length;
			m_log.debug(collectorId + ", Size=" + collectorIdSize);

			for (CollectorItemInfo itemInfo : property.getItemList()) {
				long itemSize = _fixRowSize  // ヘッダ及びdate_timeとvalueのサイズ
						+ ((collectorIdSize / blockSize + 1) * blockSize)
						+ ((itemInfo.getItemCode().getBytes(DB_CHARSET_NAME).length / blockSize + 1) * blockSize)
						+ ((itemInfo.getDisplayName().getBytes(DB_CHARSET_NAME).length / blockSize + 1) * blockSize);
				m_log.debug("ItemCode=" + itemInfo.getItemCode()
						+ ", DisplayName=" + itemInfo.getDisplayName()
						+ ", ItemSize=" + itemSize
						+ ", total=" + totalItemSize);
				totalItemSize = totalItemSize + itemSize;
			}
		} catch (UnsupportedEncodingException e) {
			return -1l;
		}

		// 収集対象のファシリティツリーを取得
		FacilityTreeItem tree = property.getCollectorData().getFacilityTree();
		
		if(tree == null){
			// クライアントからの依頼で見積りを実施する場合、
			// ツリー情報が設定されておらず、ファシリティIDのみが設定されている。
			// リポジトリから取得
			// リポジトリの情報が変更されたなどの理由で指定のファシリティIDが存在しない場合 null が返る
			tree = new Repository().getFacilityTree(property.getCollectorData().getFacilityID());
			if(tree == null){
				return -1l;
			}
		}
		
		// 1回の収集で増加するサイズ
		long sizePerCollect = 
			addChildrenFacilityDataSize(tree, totalItemSize, property.getItemList().size(), blockSize);
		
		// インデックス分は、デーブルサイズの3倍のサイズとする（大きめの見積もり）
		m_log.debug("SizePerCollect=" + (sizePerCollect * 4));
		return sizePerCollect * 4;
	}
	
	/**
	 * 指定の要素のファシリティ配下のファシリティ毎に1回の蓄積データサイズを足し込みます。
	 * （スコープ、ノードの合計となります。）
	 * @param item ファシリティツリーのトップの要素
	 * @return 指定の要素以下のファシリティ（スコープ、ノード）の数の合計
	 */
	private long addChildrenFacilityDataSize(FacilityTreeItem item, long itemSize, long itemCount, int blockSize){
		long totalDataSize = 0;
		
		// 子要素を取得する
		FacilityTreeItem[] children = item.getChildren();
		String facilityId = item.getData().getFacilityId();
		try {
			long facilityIdSize = ((facilityId.getBytes(DB_CHARSET_NAME).length / blockSize + 1) * blockSize);
			// 1レコードのサイズは、1回の収集でのファシリティIDを除いたサイズ（itemSize）に、
			// ファシリティIDのサイズ×監視項目数を加えたもの
			long dataSize = itemSize + facilityIdSize * itemCount;  // 1レコードのサイズ
			
			if(children == null || children.length == 0){
				// 子要素がない場合（自身がノード、もしくは配下に要素を含まないスコープ）は、自身のサイズを返す
				m_log.debug("Leaf=" + facilityId + ", ItemSize=" + itemSize + ", Size=" + dataSize);
				return dataSize;
			} else {
				for(int i=0; i < children.length; i++){
					// 再帰的に子要素のデータサイズを合計する
					long childDataSize = addChildrenFacilityDataSize(children[i], itemSize, itemCount, blockSize);
					if(childDataSize == -1){
						return -1l;
					}
					totalDataSize = totalDataSize + childDataSize;
				}
				// 子要素がなくなった時点で、スコープ自身をカウントする
				if(!FacilityConstant.STRING_COMPOSITE.equals(facilityId)){
					totalDataSize = totalDataSize + dataSize;
					m_log.debug("Branch=" + facilityId + ", Size=" + dataSize + ", TotalSize = " + totalDataSize);
				}
				return totalDataSize;
			}
		} catch (UnsupportedEncodingException e) {
			m_log.warn("addChildrenFacilityDataSize() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			return -1l;
		}
	}

	/**
	 * 実績収集で1行あたりに必要な物理サイズ容量（Byte）を返します。
	 * サイズ見積り時に使われます。 
	 * @return 物理サイズ容量（Byte）
	 */
	public static long getFixRowSize() {
		return _fixRowSize;
	}

	/**
	 * 実績収集で1行あたりに必要な物理サイズ容量（Byte）を設定します。
	 * サイズ見積り時に使われます。
	 * @param fixRowSize 物理サイズ容量（Byte）
	 */
	public static void setFixRowSize(long fixRowSize) {
		_fixRowSize = fixRowSize;
	}
	
	/**
	 * 全ての収集実行中のCollectorのScopeTreeキャッシュを更新し、ポーラーを起動する。
	 * @throws NamingException 
	 * @throws FinderException 
	 * 
	 * @ejb.interface-method
	 *	view-type="local" 
	 *
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Local"
	 */
	public void refreshScopeTree() throws FinderException, NamingException{
		synchronized (_refreshScopeTreeLock) {
			// リポジトリ情報の変更をチェックする
			if(RepositoryUpdateTime.getUpdateTime().getTime() > _lastRepositoryUpdateTime){
				// 更新があった場合は全ての収集のScopeTreeをリフレッシュ
				
				Collection<CollectorLocal> cols = null;
				try {
					cols = CollectorUtil.getLocalHome().findAll();
				} catch (FinderException e) {
					m_log.warn("refreshScopeTree() : " + e.getClass().getSimpleName() +
							", " + e.getMessage());
					m_context.setRollbackOnly();
					throw e;
				} catch (NamingException e) {
					m_log.warn("refreshScopeTree() : " + e.getClass().getSimpleName() +
							", " + e.getMessage());
					m_context.setRollbackOnly();
					throw e;
				}
				for(CollectorLocal collector : cols){
					// スコープツリー情報を更新
					collector.updateScopeTree();
					
					if(collector.getStatus() == CollectorRunStatus.RUNNING){
						// ポーラーを再実行
						try {
							collector.startPoller();
						} catch (NamingException e) {
							m_log.warn("refreshScopeTree() : " + e.getClass().getSimpleName() +
									", " + e.getMessage());
							m_context.setRollbackOnly();
							throw e;
						}
					}
				}
				
				_lastRepositoryUpdateTime = RepositoryUpdateTime.getUpdateTime().getTime();
			}
		}
	}
}