/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performance.util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.FacilityNotFoundException;
import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.FacilityPath;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorDeviceInfoData;
import com.clustercontrol.performance.util.Repository;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.ejb.entity.DeviceInfoData;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.repository.bean.FacilityInfo;

/**
 * リポジトリからのデータ取得を提供するクラス
 * @version 1.0
 * @since 1.0
 */
public class Repository {
	//ログ出力
	private static Log m_log = LogFactory.getLog( Repository.class );
	
	/**
	 *指定のファシリティIDのプラットフォームIDを返します
	*/
	public static String getPlatformId(String facilityId) {
		// リポジトリ機能のコントローラを取得
		try {
			RepositoryControllerLocal bean = RepositoryControllerUtil.getLocalHome().create();

			// プラットフォームIDを取得するためのキー
			String key = FacilityAttributeConstant.PLATFORM;
			
			// プラットフォームIDを問い合わせる
			ArrayList<String> attributeList = new ArrayList<String>();
			attributeList.add(key);
			HashMap facilityAttrMap = bean.getNodeDetail(facilityId, attributeList);

			String platformId = (String)facilityAttrMap.get(key);
			
			return platformId;
		} catch (CreateException e) {
			m_log.error(e.getMessage(), e);
			throw new EJBException(e);
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
			throw new EJBException(e);
		} catch (FacilityNotFoundException e) {
			m_log.error(e.getMessage(), e);
			throw new EJBException(e);
		}
	}
	
	/**
	 *指定のファシリティIDのサブプラットフォームIDを返します
	 *サブプラットフォームIDは現状、「仮想化ソリューション(VIRTSOLUTION)」とする。
	 *将来、「仮想化ソリューション」以外で使用する事を想定して、
	 *サブプラットフォームIDという名前にしておく。
	*/
	public static String getSubPlatformId(String facilityId) {
		// リポジトリ機能のコントローラを取得
		try {
			RepositoryControllerLocal bean = RepositoryControllerUtil.getLocalHome().create();

			// プラットフォームIDを取得するためのキー
			String key = FacilityAttributeConstant.VIRTSOLUTION;
			
			// プラットフォームIDを問い合わせる
			ArrayList<String> attributeList = new ArrayList<String>();
			attributeList.add(key);
			HashMap facilityAttrMap = bean.getNodeDetail(facilityId, attributeList);

			String subPlatformId = (String)facilityAttrMap.get(key);
			
			if (subPlatformId == null) {
				subPlatformId = "";
			}
			return subPlatformId;
		} catch (CreateException e) {
			m_log.error(e.getMessage(), e);
			throw new EJBException(e);
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
			throw new EJBException(e);
		} catch (FacilityNotFoundException e) {
			m_log.error(e.getMessage(), e);
			throw new EJBException(e);
		}
	}	
	
	/**
	*指定のファシリティIDのノード種別を返します
	*/
	public static String getVirtNodeType(String facilityId){
		// リポジトリ機能のコントローラを取得
		try {
			RepositoryControllerLocal bean = RepositoryControllerUtil.getLocalHome().create();
			
			// プラットフォームIDを取得するためのキー
			String key = FacilityAttributeConstant.VIRTNODETYPE;
			
			// プラットフォームIDを問い合わせる
			ArrayList<String> attributeList = new ArrayList<String>();
			attributeList.add(key);
			HashMap facilityAttrMap = bean.getNodeDetail(facilityId, attributeList);
			
			String subPlatformId = (String)facilityAttrMap.get(key);
			
			if (subPlatformId == null) {
				subPlatformId = "";
			}
			return subPlatformId;
		} catch (CreateException e) {
			m_log.error(e.getMessage(), e);
			throw new EJBException(e);
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
			throw new EJBException(e);
		} catch (FacilityNotFoundException e) {
			m_log.error(e.getMessage(), e);
			throw new EJBException(e);
		}
	}

	
	/**
	 * リポジトリから現在のファシリティツリー情報を取得します。
	 * @param facilityID 取得したいツリーの最上位のファシリティID
	 * @return ファシリティツリーの最上位の要素
	 */
	public FacilityTreeItem getFacilityTree(String facilityID) {
		FacilityTreeItem top = null;
		// ツリー情報を取得
		try {
			RepositoryControllerLocal bean = RepositoryControllerUtil.getLocalHome().create();
			FacilityTreeItem originalFacilityTree = bean.getExecTargetFacilityTree(Locale.getDefault());
			bean.remove();
			
			// ツリーのコピーを作成する
			FacilityTreeItem facilityTree = null;
			try {
				facilityTree = originalFacilityTree.deepCopy();
			} catch (IOException e) {
				m_log.error(e.getMessage(), e);
			} catch (ClassNotFoundException e) {
				m_log.error(e.getMessage(), e);
			}
			
			// 指定のファシリティID以下のツリーを取得する
			FacilityTreeItem subFacilityTree = selectFacilityTreeItem(facilityTree, facilityID);
			
			if(subFacilityTree == null){
				// エラー処理
				String message= "Facility not found : " + facilityID;
				return null;
			}
			
			//FacilityTreeの最上位インスタンスを作成
			FacilityInfo info = new FacilityInfo();
			info.setFacilityName(FacilityConstant.STRING_COMPOSITE);
			info.setFacilityType(FacilityConstant.TYPE_COMPOSITE);
			top = new FacilityTreeItem(null, info);
			
			// 取得したファシリティツリーをコンポジットアイテムに繋ぐ
			subFacilityTree.setParent(top);
			top.addChildren(subFacilityTree);
		} catch (CreateException e) {
			m_log.error(e.getMessage(), e);
			throw new EJBException(e);
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
			throw new EJBException(e);
			/*
		} catch (FinderException e) {
			// リポジトリの内容が変更になっているなどの理由で取得できなかった場合
			// Exception は throw しない
			// 収集時以外にも収集用EntityBeanの参照時(収集停止時など)に参照されるため
			// Exception を throw すると処理が終了してしまい，収集停止ができなくなる
			m_log.error(e.getMessage(), e);
			*/
		} catch (RemoveException e) {
			m_log.error(e.getMessage(), e);
			throw new EJBException(e);
		}
		return top;
	}
	
	/**
	 * 指定のファシリティIDのパスを取得します。
	 * @param facilityID パスを取得したいファシリティのファシリティID
	 * @return ファシリティパス
	 */
	public String getFacilityPath(String facilityID) {
		// ツリー情報を取得
		try {
			RepositoryControllerLocal bean = RepositoryControllerUtil.getLocalHome().create();
			FacilityTreeItem facilityTree = bean.getFacilityTree(Locale.getDefault());
			bean.remove();

			FacilityTreeItem treeItem = selectFacilityTreeItem(facilityTree, facilityID);
			
			if(treeItem == null){
				// エラー処理
				String message= "Facility not found : " + facilityID;
				throw new EJBException(message);
			}
			
			FacilityPath fpath = new FacilityPath(">");
			
			return fpath.getPath(treeItem);
		} catch (CreateException e) {
			throw new EJBException(e);
		} catch (NamingException e) {
			throw new EJBException(e);
		} catch (RemoveException e) {
			throw new EJBException(e);
		}
	}

	/**
	 * リポジトリから指定のファシリティID以下の階層のノードのデバイス情報を取得します。
	 * 
	 * @param collectorID 収集ID
	 * @param facilityID 対象ファシリティID
	 * @return　デバイス情報のリスト
	 */
	public List<CollectorDeviceInfoData> getDeviceList(String facilityID) {
		try {
			RepositoryControllerLocal bean = RepositoryControllerUtil.getLocalHome().create();

			ArrayList<CollectorDeviceInfoData> deviceList = new ArrayList<CollectorDeviceInfoData>();
			
			List<String> nodeList = null;
			if(facilityID == null){  // ファシリティIDがnullの場合
				// スコープにふくまれる全てのノードのIDのリストを取得
				nodeList = bean.getNodeFacilityIdList();
			}
			
//			デバイス情報は、管理対象チェックボックスの内容を問わず、
//			該当のスコープすべてのデバイス情報を参照するように変更 Ver3.1.0 Uchiyama
//			else {
//				nodeList = bean.getExecTargetFacilityIdList(facilityID);
//			}
			
			else if(bean.isNode(facilityID)){	// 対象ファシリティがノードの場合
				// 指定のファシリティIDのデバイスを取得
				nodeList = new ArrayList<String>();
				nodeList.add(facilityID);
			} else {	// 対象ファシリティがスコープの場合
				// スコープにふくまれるノードのIDのリストを取得
				nodeList = bean.getNodeFacilityIdList(facilityID, RepositoryControllerBean.ALL);
				
			}
			
			Iterator<String> itr = nodeList.iterator();
			while(itr.hasNext()){
				String nodeID = itr.next();

				// デバイス情報を取得しリストに追加
				List<DeviceInfoData> dl = bean.getDeviceList(nodeID);
				Iterator<DeviceInfoData> itrDl = dl.iterator();
				
				while(itrDl.hasNext()) {
					DeviceInfoData did = itrDl.next();
					CollectorDeviceInfoData device = 
						new CollectorDeviceInfoData(
								null, 
								did.getFacilityId(), 
								did.getDeviceName(), 
								did.getDisplayName(),
								did.getDeviceIndex().longValue(), 
								did.getSnmpOID(), 
								did.getDeviceType());
					deviceList.add(device);
				}
			}
			
			bean.remove();

			return deviceList;
		} catch (FacilityNotFoundException e) {
			m_log.error("facility not found. (facilityId = " + facilityID + ")", e);
			return new ArrayList<CollectorDeviceInfoData>();
		} catch (CreateException e) {
			throw new EJBException(e);
		} catch (NamingException e) {
			throw new EJBException(e);
		} catch (RemoveException e) {
			throw new EJBException(e);
		}
	}
	
	/**
	 * ファシリティツリーの中で指定のファシリティIDを持つファシリティを再帰的に探します
	 * 
	 * @param facilityTree 対象のファシリティツリー
	 * @param facilityID パスを取得したいファシリティのファシリティID
	 * @return ファシリティ情報
	 */
	private FacilityTreeItem selectFacilityTreeItem(FacilityTreeItem facilityTree, String facilityID){
		if(facilityTree.getData().getFacilityId().equals(facilityID)){
			return facilityTree;
		} else {
			for(int i=0; i<facilityTree.getChildren().length; i++){
				FacilityTreeItem target = facilityTree.getChildren()[i];
				FacilityTreeItem temp = selectFacilityTreeItem(target, facilityID);  // 再帰的
				if(temp != null){
					return temp;  
				}
			}
		}
		return null;
	}
	
	/**
	 * 指定ファシリティIDのノードのIPアドレスを取得します。
	 * 
	 * @param facilityID 対象のファシリティID
	 */
	public String getIPAddress(String facilityID) {
		HashMap nodeInfo = new HashMap();
		
		// IPアドレスを取得
		try {
			RepositoryControllerLocal bean = RepositoryControllerUtil.getLocalHome().create();
			ArrayList attributes = new ArrayList();
			attributes.add(FacilityAttributeConstant.IPNETWORKNUMBER);
			nodeInfo = bean.getNodeDetail(facilityID, attributes);
			bean.remove();
		} catch (CreateException e) {
			throw new EJBException(e);
		} catch (NamingException e) {
			throw new EJBException(e);
		} catch (FacilityNotFoundException e) {
			throw new EJBException(e);
		} catch (RemoveException e) {
			throw new EJBException(e);
		}

		if(nodeInfo != null){
			return (String)nodeInfo.get(FacilityAttributeConstant.IPNETWORKNUMBER);
		} else {
			// エラー処理
			String message= "Couldn't get an IP Address : " + facilityID;
			throw new EJBException(message);
		}
	}
	
	/**
	 * 指定のファシリティIDがノードであるか否かを判定します。
	 * 
	 * @param facilityID ファシリティID
	 * @return ノードの場合は true
	 */
	public boolean isNode(String facilityID){
		RepositoryControllerLocal bean = null;
		try {
			bean = RepositoryControllerUtil.getLocalHome().create();
			return bean.isNode(facilityID);
		} catch (CreateException e) {
			throw new EJBException(e);
		} catch (NamingException e) {
			throw new EJBException(e);
		} catch (FacilityNotFoundException e) {
			throw new EJBException(e);
		} finally {
			try {
				bean.remove();
			} catch (EJBException e) {
				throw new EJBException(e);
			} catch (RemoveException e) {
				throw new EJBException(e);
			}
		}
	}
	
	
	/**
	 * 指定のファシリティID配下のノード一覧を取得します。
	 * @param facilityID ファシリティID
	 * @return nodeList ノードのリスト 
	 */
	public ArrayList<NodeInfo> getNodeList(String facilityID){
		
		RepositoryControllerLocal bean = null;
		ArrayList<NodeInfo> nodeList = null;
		
		try {
	
			bean = RepositoryControllerUtil.getLocalHome().create();
			
			// facilityID配下のノードをリストで取得する。
			nodeList = bean.getNodeList(facilityID, RepositoryControllerBean.ALL);
			

		} catch (CreateException e) {
			throw new EJBException(e);
		} catch (NamingException e) {
			throw new EJBException(e);
		} finally {
			try {
				bean.remove();
			} catch (EJBException e) {
				throw new EJBException(e);
			} catch (RemoveException e) {
				throw new EJBException(e);
			}
		}
		return nodeList;
	}
}
