/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.snmp.util;

import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.snmp.bean.SnmpCheckInfo;
import com.clustercontrol.snmp.ejb.entity.MonitorSnmpInfoLocal;
import com.clustercontrol.snmp.ejb.entity.MonitorSnmpInfoPK;
import com.clustercontrol.snmp.ejb.entity.MonitorSnmpInfoUtil;
import com.clustercontrol.snmp.ejb.entity.MonitorSnmpValueLocal;
import com.clustercontrol.snmp.ejb.entity.MonitorSnmpValueUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * SNMP監視 判定情報を管理するクラス<BR>
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class ControlSnmpInfo {
    protected static Log m_log = LogFactory.getLog( ControlSnmpInfo.class );
    
    /** 監視対象ID */
	protected String m_monitorTypeId;
	
	/** 監視ID */
	protected String m_monitorId;
	
	/**
	 * コンストラクタ
	 * 
	 * @param monitorId 監視項目ID
	 * @param monitorTypeId 監視対象ID
	 * @version 2.1.0
	 * @since 2.1.0
	 */
	public ControlSnmpInfo(String id, String typeId) {
		m_monitorId = id;
		m_monitorTypeId = typeId;
	}
	
	/**
	 * SNMP監視情報を取得
	 * 
	 * @return SNMP監視情報
	 * @throws FinderException
	 * @throws NamingException
	 * @version 2.1.0
	 * @since 2.1.0
	 */
	public SnmpCheckInfo get() throws FinderException, NamingException{
		
		// SNMP監視情報を取得
		MonitorSnmpInfoPK pk = new MonitorSnmpInfoPK(m_monitorId, m_monitorTypeId);
		MonitorSnmpInfoLocal snmpLocal = MonitorSnmpInfoUtil.getLocalHome().findByPrimaryKey(pk);
		
		SnmpCheckInfo snmp = new SnmpCheckInfo();
		snmp.setMonitorTypeId(m_monitorTypeId);
		snmp.setMonitorId(m_monitorId);
		snmp.setCommunityName(snmpLocal.getCommunityName());
		snmp.setSnmpOid(snmpLocal.getSnmpOid());
		snmp.setConvertFlg(snmpLocal.getConvertFlg());
		snmp.setSnmpPort(snmpLocal.getSnmpPort());
		snmp.setSnmpVersion(snmpLocal.getSnmpVersion());
		
		return snmp;
	}

	/**
	 * SNMP監視情報を追加
	 * 
	 * @param snmp SNMP監視情報
	 * @return 成功した場合、true
	 * @throws CreateException
	 * @throws NamingException
	 * @version 2.1.0
	 * @since 2.1.0
	 */
	public boolean add(SnmpCheckInfo snmp) throws CreateException, NamingException {
		
		// SNMP監視情報を追加
		MonitorSnmpInfoUtil.getLocalHome().create(
				m_monitorTypeId,
				m_monitorId,
				snmp.getCommunityName(),
				snmp.getSnmpOid(),
				snmp.getConvertFlg(),
				snmp.getSnmpPort(),
				snmp.getSnmpVersion());
		
		return true;
	}
	
	/**
	 * SNMP監視情報を変更
	 * 
	 * @param snmp SNMP監視情報
	 * @param deleteValueFlg 前回値情報削除フラグ
	 * @return 成功した場合、true
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * @version 2.1.0
	 * @since 2.1.0
	 */
	public boolean modify(SnmpCheckInfo snmp, boolean deleteValueFlg) throws CreateException, FinderException, NamingException {
		
		// SNMP監視情報を取得
		MonitorSnmpInfoPK pk = new MonitorSnmpInfoPK(m_monitorId, m_monitorTypeId);
		MonitorSnmpInfoLocal snmpLocal = MonitorSnmpInfoUtil.getLocalHome().findByPrimaryKey(pk);
		
		// 更新前のOIDを取得
		String prevSnmpOid = snmpLocal.getSnmpOid();
		
		snmpLocal.setCommunityName(snmp.getCommunityName());
		snmpLocal.setSnmpOid(snmp.getSnmpOid());
		snmpLocal.setConvertFlg(snmp.getConvertFlg());
		
		snmpLocal.setSnmpPort(snmp.getSnmpPort());
		snmpLocal.setSnmpVersion(snmp.getSnmpVersion());
		
		// SNMP前回値情報を削除
		if(deleteValueFlg || !prevSnmpOid.equals(snmp.getSnmpOid())){
			
			try {
				Collection ct = MonitorSnmpValueUtil.getLocalHome().findByMonitorId(m_monitorId);
				if(ct != null && ct.size() > 0){
			        Object[] infoArray = ct.toArray();
			        for(int j = 0; j < infoArray.length; j++){
			        	MonitorSnmpValueLocal value = (MonitorSnmpValueLocal)infoArray[j];
			            if(value != null){
			            	value.remove();
			            }
			        }
			    }
			} catch (RemoveException e) {
				 AplLogger apllog = new AplLogger("SNMP", "snmp");
				 String[] args = { m_monitorId };
				 apllog.put("SYS", "001", args);
		         m_log.debug("modifyCheckInfo():" + e.getMessage());

				return false;
			}
		}
		return true;
	}

	/**
	 * SNMP監視情報を削除
	 * 
	 * @return 成功した場合、true
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws NamingException
	 * @version 2.1.0
	 * @since 2.1.0
	 */
	public boolean delete() throws FinderException, RemoveException, NamingException {
		
		// SNMP監視情報を取得
		MonitorSnmpInfoPK pk = new MonitorSnmpInfoPK(m_monitorId, m_monitorTypeId);
		MonitorSnmpInfoLocal snmpLocal = MonitorSnmpInfoUtil.getLocalHome().findByPrimaryKey(pk);
		
		// SNMP監視情報を削除
		snmpLocal.remove();
		
		// SNMP前回値情報を削除
		try {
			Collection ct = MonitorSnmpValueUtil.getLocalHome().findByMonitorId(m_monitorId);
			if(ct != null && ct.size() > 0){
		        Object[] infoArray = ct.toArray();
		        for(int j = 0; j < infoArray.length; j++){
		        	MonitorSnmpValueLocal value = (MonitorSnmpValueLocal)infoArray[j];
		            if(value != null){
		            	value.remove();
		            }
		        }
		    }
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("SNMP", "snmp");
			String[] args = { m_monitorId };
			apllog.put("SYS", "001", args);
	        m_log.debug("deleteCheckInfo():" + e.getMessage());

			return false;
		}
		return true;
	}
}
