/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.snmptrap.jmx;

import javax.naming.CompositeName;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.Name;
import javax.naming.NamingException;

import org.jboss.system.ServiceMBeanSupport;
import org.jboss.util.naming.NonSerializableFactory;

import com.clustercontrol.snmptrap.factory.TrapSnmpManager;

/**
 * SNMP トラップを受信するmBean <BR>
 * 
 * @jmx.mbean
 *     name="user:service=SnmpTrapService,name=SnmpTrapService"
 *     description="SnmpTrapService MBean"
 *     extends="org.jboss.system.ServiceMBean"
 * 
 * @jboss.service servicefile="SnmpTrap"
 */
public class SnmpTrapService extends ServiceMBeanSupport implements SnmpTrapServiceMBean{
	private String jndiName = "SnmpTrapService";  // JNDIに登録する際の名前
	
	private TrapSnmpManager m_manager;
	
	private String m_host;
	
	private int m_port;
	
	public SnmpTrapService(){
		
	}
	
	/**
	 * @jmx.managed-attribute
	 * 
	 * @see com.clustercontrol.troubledetection.jmx.SnmpTrapServiceMBean#getJndiName()
	 */
	public String getJndiName() {
		return jndiName;
	}
	
	/**
	 * @jmx.managed-attribute
	 * 
	 * @see com.clustercontrol.troubledetection.jmx.SnmpTrapServiceMBean#setJndiName(java.lang.String)
	 * 
	 * @param jndiName
	 * @throws NamingException 
	 */
	public void setJndiName(String jndiName) throws NamingException {
		log.info("setJndiName() : jndiName = " + jndiName);
		
		String oldName = this.jndiName;
		this.jndiName = jndiName;
		
		if (super.getState() == STARTED) {
			try {
				unbind(oldName);
			} catch (NamingException e){
				NamingException ne = new NamingException("Failed to unbind");
				ne.setRootCause(e);
				log.error(ne);
			}
			
			try{
				rebind();
			} catch (NamingException e){
				NamingException ne = new NamingException("Failed to rebind");
				ne.setRootCause(e);
				log.error(ne);
			}
		}
	}
	
	/**
	 * @jmx.managed-attribute
	 * 
	 * @see org.jboss.system.ServiceMBean#getName()
	 */
	@Override
	public String getName() {
		return "SnmpTrapService(" + jndiName + ")";
	}
	
	/* (non-Javadoc)
	 * @see org.jboss.system.ServiceMBeanSupport#createService()
	 */
	@Override
	public void createService() {
		log.info("Create SnmpTrapService(" + jndiName + ")");
		
		m_manager = new TrapSnmpManager(m_host, m_port);
		
		log.info("Created SnmpTrapService(" + jndiName + ")");
	}
	
	/* (non-Javadoc)
	 * @see org.jboss.system.ServiceMBeanSupport#startService()
	 */
	@Override
	public void startService() throws NamingException {
		log.info("Start SnmpTrapService(" + jndiName + ")");
		
		// 監視開始
		m_manager.exec();
		
		rebind();
		
		log.info("Started SnmpTrapService(" + jndiName + ")");
	}
	
	/* (non-Javadoc)
	 * @see org.jboss.system.ServiceMBeanSupport#stopService()
	 */
	@Override
	public void stopService() throws NamingException {
		log.info("Stop SnmpTrapService(" + jndiName + ")");
		
		// 監視終了
		m_manager.terminate();
		
		unbind(jndiName);
		
		log.info("Stoped SnmpTrapService(" + jndiName + ")");
	}
	
	/* (non-Javadoc)
	 * @see org.jboss.system.ServiceMBeanSupport#destroyService()
	 */
	@Override
	public void destroyService() throws Exception {
		log.info("Destroy SnmpTrapService(" + jndiName + ")");
		
		//監視終了
		if(m_manager != null)
			m_manager.terminate();
		
		m_manager = null;
		
		log.info("Destroyed SnmpTrapService(" + jndiName + ")");
	}
	
	private static Context createContext(Context rootCtx, Name name) throws NamingException {
		Context subctx = rootCtx;
		
		for (int n = 0; n < name.size(); n++) {
			String atom = name.get(n);
			
			try {
				Object obj = subctx.lookup(atom);
				subctx = (Context) obj;
			} catch (NamingException e) {
				// 存在しない場合は、サブコンテキストを生成
				subctx = subctx.createSubcontext(atom);
			}
		}
		
		return subctx;
	}
	
	private void rebind() throws NamingException {
		InitialContext rootCtx = new InitialContext();
		
		Name fullName = rootCtx.getNameParser("").parse(jndiName);
		Name parentName = fullName;
		if(fullName.size() > 1){
			parentName = fullName.getPrefix(fullName.size()-1);
		} else {
			parentName = new CompositeName();
		}
		
		Context parentCtx = createContext(rootCtx, parentName);
		Name atomName = fullName.getSuffix(fullName.size()-1);
		String atomStirng = atomName.get(0);
		
		NonSerializableFactory.rebind(parentCtx, atomStirng, m_manager);
	}
	
	private void unbind(String jndiName) throws NamingException {
		InitialContext rootCtx = null;
		
		try {
			rootCtx = new InitialContext();
			
			Name fullName = rootCtx.getNameParser("").parse(jndiName);
			Name atomName = fullName.getSuffix(fullName.size() - 1);
			String atom = atomName.get(0);
			
			rootCtx.unbind(jndiName);
			NonSerializableFactory.unbind(atom);
		} finally {
			if(rootCtx != null) { 
				rootCtx.close(); 
			}
		}
	}

	/**
	 * @jmx.managed-attribute
	 * 待ち受けポート番号を返します。
	 * 
	 * @return 待ち受けポート番号
	 */
	public int getPort() {
		return m_port;
	}
	
	/**
	 * @jmx.managed-attribute
	 *
	 * 待ち受けポート番号を設定します。
	 * @param port 待ち受けポート番号
	 */
	public void setPort(int port) {
		m_port = port;
	}

	/**
	 * @jmx.managed-attribute
	 * ホスト名を返します。
	 * 
	 * @return ホスト名
	 */
	public String getHost() {
		return m_host;
	}

	/**
	 * @jmx.managed-attribute
	 * ホスト名を設定します。
	 * 
	 * @param host ホスト名
	 */
	public void setHost(String host) {
		this.m_host = host;
	}
}
