/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.syslogng.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.MonitorNotFoundException;
import com.clustercontrol.UsedFacilityException;
import com.clustercontrol.bean.PluginConstant;
import com.clustercontrol.commons.ejb.session.CheckFacility;
import com.clustercontrol.syslogng.bean.LogFilterInfo;
import com.clustercontrol.syslogng.ejb.entity.SyslogMonitorRuleUtil;
import com.clustercontrol.syslogng.factory.ModifyMonitorRule;
import com.clustercontrol.syslogng.factory.SelectMonitorRule;

/**
 * syslog-ng監視の管理を行う Session Bean <BR>
 * クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。
 * 
 * @ejb.bean name="MonitorSyslogNGController"	
 *           jndi-name="MonitorSyslogNGController"
 *           type="Stateless" 
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=SyslogMonitorRuleLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 */
public abstract class MonitorSyslogNGControllerBean implements javax.ejb.SessionBean, CheckFacility {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( MonitorSyslogNGControllerBean.class );
	
	/** コンテキスト情報。 */
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean の生成時に行う処理を実装します。
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * フィルタ情報一覧を取得します。<BR>
	 * 
	 * ArrayListにはLogFilterInfoが格納されています。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SyslogNgMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @return フィルタ情報一覧 （{@link com.clustercontrol.syslogng.bean.LogFilterInfo}のリスト）
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 * 
	 * @see com.clustercontrol.syslogng.factory.SelectMonitorRule#getFilterInfoList()
	 */
	public ArrayList<LogFilterInfo> getFilterInfoList() throws MonitorNotFoundException, HinemosUnknownException{
		
	    SelectMonitorRule select = new SelectMonitorRule();
	    ArrayList<LogFilterInfo> list = null;
	    try {
			list = select.getFilterInfoList();
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getFilterInfoList " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
	    return list;
	}

	/**
	 * フィルタ情報をマネージャに登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SyslogNgMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param filterInfoList フィルタ情報一覧 （{@link com.clustercontrol.syslogng.bean.LogFilterInfo}のリスト）
	 * @return 登録に成功した場合、<code> true </code>
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 * 
	 * @see com.clustercontrol.syslogng.factory.ModifyMonitorRule#deleteMonitorRule()
	 * @see com.clustercontrol.syslogng.factory.ModifyMonitorRule#createMonitorRule(ArrayList, String)
	 */
	public boolean createMonitorRuleList(ArrayList<LogFilterInfo> filterInfoList) throws MonitorNotFoundException, HinemosUnknownException{
		
		Principal pri = m_context.getCallerPrincipal();

		ModifyMonitorRule modify = new ModifyMonitorRule();
		
		// 監視条件削除 
		try {
			modify.deleteMonitorRule();
		} catch (MonitorNotFoundException e) {
			m_log.error("createMonitorRuleList() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("createMonitorRuleList() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		// 監視条件登録
		boolean flag;
		try {
			flag = modify.createMonitorRule(filterInfoList, pri.getName());
		} catch (Exception e) {
			m_log.error("createMonitorRuleList() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * ファシリティIDが使用されているかチェックします。<BR>
	 * <P>
	 * 使用されていたら、UsedFacilityException がスローされる。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="SyslogNgMonitorRead"
	 *     method-intf="Remote"
	 *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param facilityId ファシリティ
	 * @throws UsedFacilityException
	 * 
	 * @see com.clustercontrol.commons.ejb.session.CheckFacility#isUseFacilityId(java.lang.String)
	 * @see com.clustercontrol.bean.PluginConstant;
	 */
	public void isUseFacilityId(String facilityId) throws UsedFacilityException {
		try {
			// ファシリティIDが監視対象になっている監視情報を取得する。
			Collection ct = SyslogMonitorRuleUtil.getLocalHome().findByFacilityId(facilityId);
			if(ct != null && ct.size() > 0) {
				throw new UsedFacilityException(PluginConstant.TYPE_SYSLOGNG_MONITOR);
			}
		} catch (FinderException e) {
			m_log.debug("isUseFacilityId() : " + e.getMessage());
		} catch (NamingException e) {
			m_log.debug("isUseFacilityId() : " + e.getMessage());
		}
	}
}
