/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.logagent;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.URL;
import java.net.URLConnection;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.ejb.CreateException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.log4j.PropertyConfigurator;
import org.apache.log4j.helpers.LogLog;
import org.jboss.util.stream.Streams;

import com.clustercontrol.logagent.util.AgentProperties;
import com.clustercontrol.logagent.util.EjbConnectionManager;
import com.clustercontrol.repository.ejb.session.RepositoryController;
import com.clustercontrol.util.Dummy;

/**
 * ログ転送エージェントメインクラス<BR>
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
public class Agent {
	
	/** プラグインID */
	static public final String PLUGIN_ID = "LOGAGENT";
	
	private EjbConnectionManager m_ejbConnectionManager;
	
	private ReceiveTopic m_receiveTopic;
	
	private SendQueue m_sendQueue;

	private UpdateRepositoryInfoReceiveTopic m_updateRepository;

	/** Agentクラス用スケジューラ * */
	private ScheduledExecutorService m_scheduler = Executors.newScheduledThreadPool(10);
	
	private TransferLogManager m_transferManager;
	
	/** マネージャへの疎通(Dummy)ポーリング周期（秒） */
	private int m_managerPollingInterval = 240;
	
	/** マネージャへの疎通ポーリングのタイムアウト(秒) */
	private int m_managerPollingTimeout = 60;
	
	/** ファシリティID更新周期（秒） */
	private int m_facilityUpdateInterval = 300;
	
	/** ログ転送エージェントで保持する最新ファシリティIDリスト */
	private Collection<String> m_facilityIdList = null;
	
	//ロガー
	static private Log log = LogFactory.getLog(Agent.class);
	
	/** log4j設定 */
	public String m_log4jFileName = null;
	/** log4j設定ファイル再読み込み間隔 */
	private long m_reconfigLog4jInterval = 600;
	
	/**
	 * メイン処理<BR>
	 * 
	 * ログ転送エージェントを起動するとこのメソッドがコールされる。
	 * 
	 * @param args プロパティファイル名
	 */
	public static void main(String[] args) throws Exception{
		
		// 引数チェック
		if(args.length != 2){
			System.out.println("Usage : java Agent [Agent.properties File Path] [log4j.properties File Path]");
			System.exit(1);
		}
		
		// Systemプロパティ
		log.info("starting Hinemos log agent...");
		log.info("java.vm.version = " + System.getProperty("java.vm.version"));
		log.info("java.vm.vendor = " + System.getProperty("java.vm.vendor"));
		log.info("java.home = " + System.getProperty("java.home"));
		log.info("os.name = " + System.getProperty("os.name"));
		log.info("os.arch = " + System.getProperty("os.arch"));
		log.info("os.version = " + System.getProperty("os.version"));
		log.info("user.name = " + System.getProperty("user.name"));
		log.info("user.dir = " + System.getProperty("user.dir"));
		log.info("user.country = " + System.getProperty("user.country"));
		log.info("user.language = " + System.getProperty("user.language"));
		log.info("file.encoding = " + System.getProperty("file.encoding"));
		
		// OS情報(識別したホスト名、IPアドレス)
		log.info("nodename = " + InetAddress.getLocalHost().getHostName());
		Enumeration networkInterfaces = NetworkInterface.getNetworkInterfaces();
		if (null != networkInterfaces) {
			while (networkInterfaces.hasMoreElements()) {
				NetworkInterface ni = (NetworkInterface) networkInterfaces.nextElement();
				Enumeration inetAddresses = ni.getInetAddresses();
				while (inetAddresses.hasMoreElements()) {
					InetAddress in = (InetAddress) inetAddresses.nextElement();
					String hostAddress = in.getHostAddress();
					log.info("ipaddress = " + hostAddress);
				}
			}
		}
		
		// Agent設定の初期化
		log.info("Agent.properties = " + args[0]);
		// ログ再読み込みの機構初期化
		log.info("log4j.properties = " + args[1]);
		
		// Agentインスタンス作成
		Agent agent = new Agent(args[0], args[1]);
		
		//エージェント処理開始
		agent.exec();
		
		log.info("Hinemos log agent started");
		
		//終了待ち
		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
		while (true) {
			// shutdown hook待ち
			synchronized (agent) {
				agent.wait();
			} 
		}
	}
	
	/**
	 * コンストラクタ
	 */
	public Agent(String propFileName, String log4jFileName) throws Exception{
		
		//------------
		//-- 初期処理
		//------------
		
		//プロパティファイル読み込み初期化
		AgentProperties.init(propFileName);
		
		String interval = null;
		// マネージャへのポーリング周期（秒）
		interval = AgentProperties.getProperty("manager.polling.interval");
		if (interval != null) {
			try {
				m_managerPollingInterval = Integer.parseInt(interval);
				log.info("manager.polling.interval = " + m_managerPollingInterval + " sec");
			} catch (NumberFormatException e) {
				log.error("manager.polling.interval",e);
			}
		}
		// ポーリングのタイムアウト(秒)
		String timeout = AgentProperties.getProperty("manager.polling.timeout");
		if (timeout != null) {
			try {
				m_managerPollingTimeout = Integer.parseInt(timeout);
				log.info("manager.polling.timeout = " + m_managerPollingTimeout + " sec");
			} catch (NumberFormatException e) {
				log.error("manager.polling.timeout", e);
			}
		}
		// ファシリティ更新周期（秒）
		interval = AgentProperties.getProperty("facility.update.interval");
		if (interval != null) {
			try {
				m_facilityUpdateInterval = Integer.parseInt(interval);
				log.info("facility.update.interval = " + m_facilityUpdateInterval + " sec");
			} catch (NumberFormatException e) {
				log.error("facility.update.interval",e);
			}
		}
		// log4j設定ファイル再読み込み設定
		if(log4jFileName == null){
			log.error("log4j.properties does not specify!");
		}
		m_log4jFileName = log4jFileName;
		
		m_ejbConnectionManager = new EjbConnectionManager();
		m_sendQueue = new SendQueue(this);
		m_transferManager = new TransferLogManager(m_ejbConnectionManager, m_sendQueue);
		
		//終了呼び出し設定
		Runtime.getRuntime().addShutdownHook(new Thread() {
			@Override
			public void run() { 
				terminate();
				log.info("Hinemos log agent stopped");
			}
		});
	}
	
	
	/**
	 * ログ転送エージェント処理を実行します。<BR>
	 * 
	 * トピックへの接続を行う。 
	 */
	public void exec() {
		//-----------------
		//-- トピック接続
		//-----------------
		if (log.isDebugEnabled())
			log.debug("exec() : create topic");
		
		//受信トピック生成
		try {
			refreshFacilityIdList();
			m_receiveTopic = new ReceiveTopic(this, getFacilityIdList(), m_transferManager, m_sendQueue);
		} catch (RemoteException e) {
			// マネージャに接続できなかった場合
			HashMap<String, String> map = new HashMap<String, String>();
			m_receiveTopic = new ReceiveTopic(this, map.values(), m_transferManager, m_sendQueue);
			
			log.error("exec(): Unable to connect to Hinemos manager. ",e);
		} catch (NamingException e) {
			// マネージャに接続できなかった場合
			HashMap<String, String> map = new HashMap<String, String>();
			m_receiveTopic = new ReceiveTopic(this, map.values(), m_transferManager, m_sendQueue);
			
			log.error("exec(): Unable to connect to Hinemos manager. ",e);
		} catch (CreateException e) {
			// マネージャに接続できなかった場合
			HashMap<String, String> map = new HashMap<String, String>();
			m_receiveTopic = new ReceiveTopic(this, map.values(), m_transferManager, m_sendQueue);
			
			log.error("exec(): Unable to connect to Hinemos manager. ",e);
		} catch (Exception e) {
			// 上記以外の例外が発生した場合
			HashMap<String, String> map = new HashMap<String, String>();
			m_receiveTopic = new ReceiveTopic(this, map.values(), m_transferManager, m_sendQueue);
			
			log.error("exec(): Other exceptions were generated. ",e);
		}	
		
		
		m_updateRepository = new UpdateRepositoryInfoReceiveTopic(this);

		// キャッシュ更新タイマー開始
		m_scheduler.scheduleWithFixedDelay(new ReflashFilterTask(), 
				m_facilityUpdateInterval, m_facilityUpdateInterval, TimeUnit.SECONDS);
		
		// Hinemosマネージャ間疎通タイマー開始
		if (m_managerPollingInterval > 0) {
			m_scheduler.scheduleWithFixedDelay(new HeartBeatTask(this, m_managerPollingTimeout), 
					30, m_managerPollingInterval, TimeUnit.SECONDS);
		} else {
			log.info("hinemos heartbeat : disabled");
		}
		
		// log4j再読み込みタスク
		m_scheduler.scheduleWithFixedDelay(new ReloadLog4jTask(), 
				m_reconfigLog4jInterval, m_reconfigLog4jInterval, TimeUnit.SECONDS);
	}
	
	
	/**
	 * 全ログファイルの転送を終了します。<BR>
	 *  
	 */
	public void terminateTransfer() {
		
		// ログ転送停止
		m_transferManager.stopTransfer();
	}
	
	

	/**
	 * ファシリティIDを設定します。<BR>
	 * 
	 * Hinemosでは、ファシリティIDを定期的に取得し
	 * セットします。
	 * 
	 */
	public void setFacility() {
		
		try {
			refreshFacilityIdList();
			m_receiveTopic.setFacilityIdList(getFacilityIdList());
		} catch (RemoteException e) {
			// マネージャに接続できなかった場合			
			log.error("Unable to connect to Hinemos manager. ",e);
		} catch (NamingException e) {
			// マネージャに接続できなかった場合
			log.error("Unable to connect to Hinemos manager. ",e);
		} catch (CreateException e) {
			// マネージャに接続できなかった場合
			log.error("Unable to connect to Hinemos manager. ",e);
		} catch (Exception e) {
			// 上記以外の例外が発生した場合
			log.error("Other exceptions were generated. ",e);
		}
		
	}
	
	/**
	 * ファシリティIDを取得します。<BR>
	 * 
	 * 自局のIPアドレスと、ホスト名でリポジトリからファシリティIDを取得します。
	 * 
	 * @throws Exception 
	 * @since
	 */
	private void refreshFacilityIdList() throws Exception {
		
		//RepositoryControllerを取得
		
		HashMap<String, String> map = new HashMap<String, String>();
		
		RepositoryController repository = null;
		try {
			repository = m_ejbConnectionManager.getRepositoryController();
		} catch (Exception e1) {
			throw e1;
		}
		
		try {
			//ネットワーク情報取得
			Enumeration networkInterfaces = NetworkInterface
			.getNetworkInterfaces();
			if (null != networkInterfaces) {
				
				while (networkInterfaces.hasMoreElements()) {
					
					NetworkInterface ni = (NetworkInterface) networkInterfaces
					.nextElement();
					
					if (log.isDebugEnabled()) {
						try {
							log.debug("NetworkInterface Detected :\t"
									+ new String(ni.getDisplayName().getBytes(
									"iso-8859-1")));
						} catch (UnsupportedEncodingException e) {
							log.debug("NetworkInterface Detected :\t", e);
						}
					}
					
					Enumeration inetAddresses = ni.getInetAddresses();
					while (inetAddresses.hasMoreElements()) {
						
						InetAddress in = (InetAddress) inetAddresses.nextElement();
						
						String hostAddress = in.getHostAddress();
						String hostName = in.getLocalHost().getHostName();
						
						if (log.isDebugEnabled()){
							log.debug("IP/HOST Detected :\t" + hostAddress	+ "/" + hostName);
						}
						
						//ローカルホスト以外の時場合
						if (hostAddress.compareTo("127.0.0.1") != 0) {
							
							//IPとホスト名(ドメイン名なし)で、ファイシリティIDを特定
							ArrayList facilityIdList 
								= repository.getFacilityIdList(hostName, hostAddress);
							
							if (log.isDebugEnabled()) {
								if (facilityIdList.size() == 0) {
									log.debug("facility not found : " + "hostName=" + hostName + ", hostAddress=" + hostAddress);
								}
							}
							
							for (Iterator iter = facilityIdList.iterator(); iter.hasNext();) {
								String facilityId = (String) iter.next();
								
								log.info("My facilityId = " + facilityId + " : hostName = " + hostName + ", hostAddress = " + hostAddress);

								//ファシリティの対象に追加
								if (map.containsKey(facilityId) == false) {
									map.put(facilityId, facilityId);
								}
							}
						}
					}
					
				}
			}
			
			
		} catch (SocketException e) {
			log.debug("There is no NetWorkInterfaces Information");
			log.error(e.getMessage(), e);
			
		} catch (RemoteException e) {
			log.debug("getFacilityIdList() Failed");
			log.error(e.getMessage(), e);
		}
		
		// 1つもfacilityIdを検出できなかった場合に警告ログを出力する
		if(map.size() == 0){
			log.warn("failed to detect local facilityId !");
		}
		
		m_facilityIdList =map.values(); 
	}

	
	/**
	 * ファシリティIDリストの取得
	 * @return
	 */
	public Collection<String> getFacilityIdList() {
		return m_facilityIdList;
	}

	/**
	 * ファシリティIDリストの設定
	 * @param idList
	 */
	public synchronized void setFacilityIdList(Collection<String> idList) {
		m_facilityIdList = idList;
	}

	/**
	 * ログ転送エージェントを終了します。<BR>
	 */
	public void terminate() {
		log.info("terminate() start");
		
		m_receiveTopic.terminate();
		m_updateRepository.terminate();
		m_sendQueue.terminate();
		m_scheduler.shutdownNow();
		
		log.info("terminate() end");
	}
	
	/**
	 * 全てのTopicとQueueの再接続処理を行う
	 * 
	 */
	synchronized public void reConnection() {
		
		log.info("Hinemos log agent reconnect start");
		
		m_receiveTopic.reconnect();
		
		m_updateRepository.reconnect();
		
		m_sendQueue.reconnect();
		
		log.info("Hinemos log agent reconnect end");
	}
	
	/**
	 * ローカルディスクに存在するファイルのURLオブジェクトを作成する
	 * 
	 * @param localFilePath
	 * @return
	 */
	private URL toUrl(String localFilePath){
		URL url = null;
		
		// ファイルの存在チェック
		FileInputStream in = null;
		try{
			in = new FileInputStream(localFilePath);
		}catch (Exception e) {
			log.error(localFilePath + " does not exists!", e);
		}finally{
			if(in != null){
				try {
					in.close();
				} catch (Exception e) {
				}
			}
		}
		
		// URLクラスの作成
		try{
			url = new URL("file", "localhost", localFilePath);
		}catch (Exception e) {
			log.error(localFilePath + " : unknown exception", e);
		}
		
		return url;
	}
	
	/**
	 * ファシリティID再取得タイマータスク<BR>
	 */
	protected class ReflashFilterTask implements Runnable {

		/**
		 * デフォルトコンストラクタ
		 */
		public ReflashFilterTask() {
		}
		
		/**
		 * キャッシュ更新
		 */
		public void run() {
			log.debug("ReflashFilterTask.run() start ");
			try{
				setFacility();
			}catch (Exception e) {
				log.error("ReflashFilterTask.run() catche RuntimeException", e);
			}
		}
	}
	
	/**
	 * マネージャ間疎通用タスク<BR>
	 * 1.Hinemosエージェントが定期的にHinemosマネージャ(Queue)にDummyクラスを送信する。<BR>
	 * 2.HinemosマネージャはDummyクラスを受信するとHinemosエージェントが使用するTOPICにDummyクラスを送信する。
	 * 
	 */
	protected class HeartBeatTask implements Runnable {

		private Agent agent = null;
		private int timeout = 60;
		
		/**
		 * デフォルトコンストラクタ
		 */
		public HeartBeatTask(Agent agent, int timeout) {
			this.agent = agent;
			this.timeout = timeout;
		}
		
		/**
		 * QueueにDummyクラス送信
		 */
		public void run() {
			// Queueはreconnect時に再接続しないため、動作判定に用いない
			// (dummy msgを送信した際に、Queueも再接続される)
			if (m_receiveTopic.isReconnecting() || m_updateRepository.isReconnecting()) {
				log.info("skipped jms validation : now reconnecting... (log_topic = " + m_receiveTopic.isReconnecting()
						+ ", repository_topic = " + m_updateRepository.isReconnecting() + ")");
				return;
			}
			
			Date heartbeatSendDate = null;
			boolean sendStatus = false;
			
			log.debug("HeartBeatTask.run() start");
			try{
				// dummy msgの送信
				heartbeatSendDate = new Date();
				sendStatus = sendDummy();
				
				// msg受信の確認
				if (sendStatus)
					checkMessageReceived(heartbeatSendDate);
			}catch (Exception e) {
				log.error("HeartBeatTask.run() catche RuntimeException", e);
			}
		}
		
		/**
		 * マネージャに対してDummyオブジェクトを送信する。
		 * マネージャ側はこのDummyオブジェクトを受信した後、ログ転送エージェントへの接続確立のためTOPIC送信
		 * 
		 */
		private boolean sendDummy(){
			boolean ret = false;
			
			log.debug("HeartBeatTask.sendDummy() start");
			
			if(m_facilityIdList != null && m_facilityIdList.size() > 0){
				log.debug("HeartBeatTask.sendDummy() facilityIdList is not null & send dummy object");

				// Serializableのリストに変換
				ArrayList<String> list = new ArrayList<String>();
				Iterator<String> itr = m_facilityIdList.iterator();
				while (itr.hasNext()) {
					list.add(itr.next());
				}

				// Dummyオブジェクトの送信
				Dummy dummy = new Dummy(list);
				if(m_facilityIdList.size() > 0)
					ret = m_sendQueue.put(dummy);
			}
			log.debug("HeartBeatTask.sendDummy() finish");
			
			return ret;
		}
		
		/**
		 * マネージャからのDummyオブジェクトの受信状況を確認する。
		 * 受信していないと判定した場合、JMSの再接続を開始する。
		 * 
		 * @param sendDate Dummyオブジェクトを送信した日時
		 */
		private void checkMessageReceived(Date sendDate) {
			log.debug("checking jms (hinemos heartbeat).");
			
			try {
				// wait timeout (whether response is received or not)
				Thread.sleep(timeout * 1000);
			} catch (InterruptedException e) {
				log.warn("skipped jms validation : wait interrupted", e);
				return;
			}
			
			if (ReceiveTopic.getHeartbeatReceiveDate().getTime() < sendDate.getTime() ||
					UpdateRepositoryInfoReceiveTopic.getHeartbeatReceiveDate().getTime() < sendDate.getTime()) {
				log.warn("heartbeat not received. (send : " + sendDate + ", receive(job topic) : " + ReceiveTopic.getHeartbeatReceiveDate() + ")");
				log.warn("heartbeat not received. (send : " + sendDate + ", receive(repository topic) : " + UpdateRepositoryInfoReceiveTopic.getHeartbeatReceiveDate() + ")");
				
				// reconnect queue and topic
				agent.reConnection();
			} else {
				if (log.isDebugEnabled()) {
					log.debug("heartbeat received. (send : " + sendDate + ", receive(job topic) : " + ReceiveTopic.getHeartbeatReceiveDate() + ")");
					log.debug("heartbeat received. (send : " + sendDate + ", receive(repository topic) : " + UpdateRepositoryInfoReceiveTopic.getHeartbeatReceiveDate() + ")");
				}
			}
		}
	}
	
	/**
	 * ログ設定を再読み込みするタスク
	 * @author Hinemos
	 *
	 */
	protected class ReloadLog4jTask implements Runnable{

		/**
		 * デフォルトコンストラクタ
		 */
		public ReloadLog4jTask() {
		}
		
		// 最終更新時刻
		private long lastConfigured = -1;		
		// log4j設定ファイルURL
		URL configURL = null;
		
		/**
		 * ログ設定ファイルリロードタスクの開始
		 */
		public void run() {
			log.debug("ReloadLog4jTask.run() Checking if configuration changed");
			
			// File Load
			configURL = toUrl(m_log4jFileName);
			
			// 対象ファイルが存在する場合に限り実行
			if(configURL != null){
				
				// File Check
				try{
					URLConnection conn = configURL.openConnection();
					long lastModified = conn.getLastModified();
					
					if (lastConfigured < lastModified)
					{
						log.debug("ReloadLog4jTask.run() do re-configuration");
						reconfigure(conn);
					}
					
				} catch (IOException e) {
					log.warn("ReloadLog4jTask.run() Failed to check URL: " + configURL, e);
				}
			}
			log.debug("ReloadLog4jTask.run() finish");
		}
		
		/**
		 * ログ設定のリロード
		 * @param conn
		 */
		private void reconfigure(final URLConnection conn) {
			log.info("Configuring from URL: " + configURL);
			
			if(log.isDebugEnabled()){
				try
				{
					System.out.println((new Date()).toString() + " : Configuring URL Data Dump");
					java.io.InputStream is = conn.getInputStream();
					Streams.copy(is, System.out);
				}
				catch (Exception e)
				{
					log.error("Failed to dump config", e);
				}
			}
			
			// reconfigure
			try{
				PropertyConfigurator.configure(configURL);
				
				/* Set the LogLog.QuietMode. As of log4j1.2.8 this needs to be set to
					avoid deadlock on exception at the appender level. See bug#696819.
				 */
				LogLog.setQuietMode(true);
				
				lastConfigured = System.currentTimeMillis();
				
				log.debug("ReloadLog4jTask.reconfigure() lastConfigured = " + lastConfigured);
				
			} catch (Exception e) {
				log.error("Failed to configure from URL: " + configURL, e);
			}
		}
	}
}
