/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.logagent;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.ObjectMessage;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicSession;
import javax.jms.TopicSubscriber;
import javax.naming.InitialContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.logagent.util.AgentProperties;
import com.clustercontrol.logagent.util.InternalLogger;
import com.clustercontrol.logtransfer.bean.LogTransferFileInfo;
import com.clustercontrol.logtransfer.bean.TopicConstant;
import com.clustercontrol.util.Dummy;




/**
 * マネージャからのJMSメッセージ(Topic)を受信するクラス<BR>
 * 
 * マネージャから監視するログファイルの指示がJMSを用いて送信されるので、
 * JMS(Topic)への接続と、メッセージの受信を行う。
 * 
 * @version 2.0.0
 * @since 2.0.0
 * 
 */
public class ReceiveTopic implements MessageListener, ExceptionListener {

	private static final String TOPIC_CON_FACTORY = "ConnectionFactory";
	private static final String TOPIC_USER_NAME = "topic.user.name";
	private static final String TOPIC_USER_PASSWORD = "topic.user.password";

	protected Agent m_agent;
	
	protected ArrayList m_facilityIdList;
    protected SendQueue m_sendQueue;

	private TopicConnectionFactory m_factory;

    protected TopicConnection m_con;
    protected Topic m_topic;
    protected TopicSession m_session;
    protected TopicSubscriber m_subscriber;
    
    protected TransferLogManager m_logManager;
	
    private long m_logTopicReconnectionInterval = 30;
    
	private volatile boolean m_errFlg = false;

	// heartbeat送受完了日時(jms妥当性判定用)
	private volatile static Date heartbeatReceiveDate = new Date();
	public static final Object _instanceHeartbeatDateLock = new Object();
	
	/** ReceiveTopicクラス用スケジューラ * */
	private ScheduledExecutorService m_scheduler = Executors.newScheduledThreadPool(10);
    private ScheduledFuture<?> m_future;

	//ロガー
	static private Log log = LogFactory.getLog(ReceiveTopic.class);

	
	/**
	 * コンストラクタ
	 * @param facilityIdList ファシリティIDのリスト　
	 * @param sendQueue　メッセージ送信用クラス
	 */
	@SuppressWarnings("unchecked")
	public ReceiveTopic(Agent agent, Collection facilityIdList, TransferLogManager logManager, SendQueue sendQueue) {
		super();
		m_agent = agent;
		m_logManager = logManager;
		m_facilityIdList = new ArrayList(facilityIdList);
		m_sendQueue = sendQueue;

		// ReciveTopic再接続処理実行間隔取得
		String interval = AgentProperties.getProperty("log.topic.reconnection.interval");
		if (interval != null) {
			try {
				// プロパティファイルには秒で記述されているため、1000をかける
				m_logTopicReconnectionInterval = Integer.parseInt(interval);
				log.info("log.topic.reconnection.interval = " + m_logTopicReconnectionInterval + " sec");
			} catch (NumberFormatException e) {
				log.error("log.topic.reconnection.interval",e);
			}
		}
		
		//接続処理
        initial();
	}
    
    
    /**
     * ファシリティIDリストを設定します。<BR>
     * 
     * 設定されたファイシリティIDのリストが、前回と違う場合は、トピック受信の再接続を行う
     * @param facilityIdList　
     * @since
     */
	@SuppressWarnings("unchecked")
	synchronized public void setFacilityIdList(Collection facilityIdList) {
	
		//取得した情報が変化しているかどうか確認
		if(facilityIdList.size() == m_facilityIdList.size()){
			if( m_facilityIdList.containsAll(facilityIdList) ){
				return;
			}
		}

//		//追加分リスト
//		ArrayList<String> addFacilityList = new ArrayList<String>(facilityIdList);
//		addFacilityList.removeAll(m_facilityIdList);
		
		//削除分リスト
		ArrayList<String> removeFacilityList = new ArrayList<String>(m_facilityIdList);
		removeFacilityList.removeAll(facilityIdList);
		
		
		//削除分をスレッド停止
		for (Iterator iter = removeFacilityList.iterator(); iter.hasNext();) {
			String facility = (String) iter.next();
			//スレッドに削除依頼
			m_logManager.removeTransferLogInfo(facility);
						
		}
		
		//古いファシリティリストを退避
		ArrayList<String> oldFacilityList = new ArrayList(m_facilityIdList);
		
		//ファシリティリスト更新
		m_facilityIdList = new ArrayList(facilityIdList);

		//Topic再接続
		if( !isErrFlg() ){
		    if( initialTopic() ){

				//取得してきた全てのログ情報を要求送信
				for (Iterator<String> iter = facilityIdList.iterator(); iter.hasNext();) {
					String facility = iter.next();
					if (!m_sendQueue.put(facility)) {
						// 失敗した場合ファシリティリストから削除
						m_facilityIdList.remove(facility);
					}
				}
			} else {
				//Topicの接続に失敗した場合は退避させたファシリティリストに戻す
				m_facilityIdList = new ArrayList(oldFacilityList);
			}
		
		}

		
	}

	
	/**
	 * JMSメッセージ受信した際に実行される処理<BR>
	 * 
	 */
	
	/* 
     * トピック受信処理
     * (non-Javadoc)
     * @see javax.jms.MessageListener#onMessage(javax.jms.Message)
     */
    @SuppressWarnings("unchecked")
	public void onMessage(Message message) {
        
        
        log.debug("onMessage start");
        String facilityId;
        boolean initFlg;
        try {
			facilityId = message.getStringProperty("FacilityId");
			initFlg = message.getBooleanProperty("Init");
	        log.debug("onMessage target facility : " + facilityId);
		} catch (JMSException e1) {
			log.error("message is incorrect", e1);
			return;
		}
        
        if(message instanceof ObjectMessage){
            ObjectMessage objectMessage = (ObjectMessage)message;
            Object obj;
			try {
				obj = objectMessage.getObject();
			} catch (JMSException e) {
				log.error("message is incorrect", e);
				return;
			}

			if(obj instanceof ArrayList){
            	log.debug("onMessage get ArrayList");
            	                
            	ArrayList list = (ArrayList)obj;
            	
				// デバッグ出力
				if (log.isDebugEnabled()) {
					Iterator itr = list.iterator();
					while (itr.hasNext()) {
						LogTransferFileInfo info = (LogTransferFileInfo) itr
								.next();
						log.debug("  " + info.getTransferId() + ", "
								+ info.getFilePath());
					}
				}
            	
				// 接続・再接続時のみログ転送情報リフレッシュ
				if(initFlg){
					log.debug("refresh RunTransferLog");
					m_logManager.refreshRunTransferLog(facilityId, list);		
				}
				// ログ転送
            	m_logManager.setTransferLogInfo(facilityId, list);
                
			}else if(obj instanceof Dummy){
				// TOPIC接続疎通のためのオブジェクトのため何もしない
				log.debug("onMessage() get Dummy");
				
				// 最終受信日時を格納する
				synchronized (_instanceHeartbeatDateLock) {
					heartbeatReceiveDate = new Date();
				}
            }else{
				log.error("message is incorrect" + obj.toString());
				return;
            }
        }else{
			log.error("message is incorrect" + message.getClass());
			return;
        }
        log.debug("onMessage end");
    }
    
    /* 通信エラーハンドラ
     * (non-Javadoc)
     * @see javax.jms.ExceptionListener#onException(javax.jms.JMSException)
     */
    public void onException(JMSException arg0) {

    	log.error(arg0.getMessage(), arg0);
    	
    	// syslogにログ出力
		InternalLogger.error("hinemos_logagent: A problem has been detected with the connection to MGR");

    	m_agent.reConnection();

    }

    /**
	 * @param errFlg
	 *            errFlg を設定します。
	 */
	synchronized private void setErrFlg(boolean errFlg) {
		log.debug("setErrFlg() start : errFlg = " + errFlg);
		if (m_errFlg == false && errFlg == true) {

			log.debug("setErrFlg() set ReSetupTask to timer");
			m_future = m_scheduler.scheduleWithFixedDelay(
					new ReSetupTask(), 
					m_logTopicReconnectionInterval, 
					m_logTopicReconnectionInterval, 
					TimeUnit.SECONDS);

		}
		m_errFlg = errFlg;
		log.debug("setErrFlg() finish");
	}

	/**
	 * errFlg を取得します。<BR>
	 * @return errFlg
	 */
	synchronized private boolean isErrFlg() {
		return m_errFlg;
	}
	/**
	 *  
	 */
	private boolean reInitial() {
		log.info("reInitial() start");
		terminate();

		boolean ret = false;
		synchronized (this) {
			if (initial()) {

				log.info("reInitial() success!");
				ret = true;

				//エラーフラグ解除
				setErrFlg(false);
				
			} else {
				log.warn("reInitial() fail!");
			}
		}
		log.info("reInitial() finish");
		return ret;
	}
	/**
	 * サーバ接続の終了処理を行います。
	 *  
	 */
	public void terminate() {
		log.debug("terminate() start");

		terminateSumscriber();

		
        try {
			if (m_session != null)
				m_session.close();
		} catch (JMSException e) {
			log.debug("terminate() session close error", e);
		}

		try {
			if (m_con != null)
				m_con.close();
		} catch (JMSException e1) {
			log.debug("terminate() connection close error", e1);
		}
		log.debug("terminate() finish");
	}
	/**
	 * トピック受信処理の終了 
	 */
	private void terminateSumscriber() {
		try {
			if (m_subscriber != null)
		        m_subscriber.close();
		} catch (JMSException e) {
			log.debug("terminateSumscriber() subscriber close error", e);
		}
	}



	/**
	 * 初期化処理<BR>
	 * JMSへの接続、トピック受信設定を行う
	 * @return
	 */
	private boolean initial() {

		log.info("initial() start");

		InitialContext con = null;

		try {
			//InitialContextの生成
			con = new InitialContext(AgentProperties.getProperties());

			//コネクションファクトリ生成
			m_factory = (TopicConnectionFactory) con.lookup(TOPIC_CON_FACTORY);

			//コネクション生成
			if (AgentProperties.getProperty(TOPIC_USER_NAME) != null) {
				//ユーザ認証
				m_con = m_factory.createTopicConnection(
						AgentProperties.getProperty(TOPIC_USER_NAME),
						AgentProperties.getProperty(TOPIC_USER_PASSWORD));
			} else {
				//ユーザ認証なし
				m_con = m_factory.createTopicConnection();
			}

			
			//セッション生成
			m_session = m_con.createTopicSession(false,
					Session.AUTO_ACKNOWLEDGE);

			//メッセージTopic取得
			m_topic = (Topic) con.lookup(TopicConstant.TOPIC_NAME_UPDATE);

			
			//エラーハンドらセット
	        m_con.setExceptionListener(this);

	        m_con.start();

	        //トピック接続開始
	        initialTopic();

			//ファシリティIDをマネージャに通知（メッセージ送信）
			for (Iterator iter = m_facilityIdList.iterator(); iter.hasNext();) {
				m_sendQueue.put(iter.next());	
			}


		} catch (Exception e) {
			log.error("initial() fail!", e);
			setErrFlg(true);
			return false;
		} finally {
			try {
				if (con != null)
					con.close();
			} catch (Exception e1) {
				log.warn("initial() close fail!", e1);
			}
		}
		log.info("initial() success!");
		return true;

	}

	/**
	 * トピック受信設定
	 * @return
	 * @since
	 */
	private boolean initialTopic() {


		//現在のTopic受信を終了
		terminateSumscriber();

//		if(isErrFlg()){
//			//エラー中の再接続ならば
//			//再接続できたら一旦ログの読み込みスレッドを停止させる
//			m_logManager.stopTransfer();
//		}
		
		//新しいファシリティIDでトピック受信開始
		try {
	        
	        //ファシリティIDでメッセージセレクターを作成
	        StringBuffer msgSelector = new StringBuffer();
	        for (Iterator iter = m_facilityIdList.iterator(); iter.hasNext();) {
				String facilityId = (String) iter.next();
				msgSelector.append("FacilityId='");
				msgSelector.append(facilityId);
				msgSelector.append("'");
				if( iter.hasNext() ){
					msgSelector.append(" OR ");
				}
			}
	        
	        if(msgSelector.length() != 0){
	        
		        m_subscriber = m_session.createSubscriber(m_topic, msgSelector.toString(), false);
		        
		        //コールバックを登録する
		        m_subscriber.setMessageListener(this);

		        
	        }else{
	        	log.debug("facility is null");
	        }
			

		} catch (Exception e) {
			log.error("TopicInit", e);
			setErrFlg(true);
			return false;
		} finally {

		}
		return true;

	}
	
	/**
	 * 再接続処理を行う
	 */
	synchronized public void reconnect() {
		setErrFlg(true);
	}
	
	/**
	 * 接続状況を返す
	 * @return 接続状況(再接続中 : true, 接続中　: false)
	 */
	synchronized public boolean isReconnecting() {
		return m_errFlg;
	}
	
	/**
	 * Heartbeanメッセージの最終受信日時を取得するメソッド
	 * @return Heartbeatメッセージの最終受信日時
	 */
	public static Date getHeartbeatReceiveDate() {
		synchronized (_instanceHeartbeatDateLock) {
			return heartbeatReceiveDate;
		}
	}
	
	/**
	 * EJB再接続タイマータスク
	 * 通信エラーとなった場合に定周期で呼ばれ再接続を行う 
	 */
	protected class ReSetupTask implements Runnable {

		/**
		 * デフォルトコンストラクタ
		 */
		public ReSetupTask() {
		}
		
		/**
		 * コネクションクローズ
		 */
		public void run() {
			log.debug("ReceiveTopic.ReSetupTask.run() start");
			try {
				if (reInitial()) {
					//このタスクをタイマーから解除
					log.debug("ReceiveTopic.ReSetupTask.run() task cancel");
					if(m_future != null){
						m_future.cancel(true);
					}
				}
			} catch (Exception e) {
				log.error("ReceiveTopic.ReSetupTask.run()", e);
			}
		}

	}
}
