/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.logagent;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import javax.jms.DeliveryMode;
import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.naming.InitialContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.OutputBasicInfo;
import com.clustercontrol.bean.OutputNotifyGroupInfo;
import com.clustercontrol.logagent.util.AgentProperties;
import com.clustercontrol.logagent.util.InternalLogger;
import com.clustercontrol.notify.bean.QueueConstant;
import com.clustercontrol.util.Dummy;

/**
 * マネージャとのやり取りに利用するQueue送信クラス<BR>
 * 
 * ログ転送エージェントはHinemosイベントと転送対象ファイルのリクエストを
 * マネージャに送信する。<BR>
 * 
 * Queueとしては、<BR>
 * com.clustercontrol.logtransfer.bean.QueueConstant.QUEUE_NAME_FILE_MANAGE<BR>
 * と<BR>
 * com.clustercontrol.monitor.bean.QueueConstant.QUEUE_NAME_LOG<BR>
 * に接続する。
 *
 * @version 3.0.0
 * @since 2.1.0
 */
public class SendQueue implements ExceptionListener {

	private static final String QUEUE_CON_FACTORY = "ConnectionFactory";

	private static final String QUEUE_USER_NAME = "queue.user.name";

	private static final String QUEUE_USER_PASSWORD = "queue.user.password";

	private static final long SEND_TIMEOUT = 60 * 1000l;
	private static final long RECONNECT_TIMEOUT = 60 * 1000l;
	
	private Agent m_agent;
	
	private QueueConnection m_con;

	private Queue m_queueLog;

	private Queue m_queueFacility;

	private QueueSession m_session;

	private QueueConnectionFactory m_factory;

	private volatile boolean m_isErr = true;
	
	private long m_sendQueueReconnectionInterval = 10000;

	//ロガー
	static private Log log = LogFactory.getLog(SendQueue.class);

	// reInitialメソッドが複数スレッドから呼ばれないよにするロック用オブジェクト
	private final Object reInitialLock = new Object();

	// Queue送信処理が複数スレッドから呼ばれないよにするロック用オブジェクト
	private final Object sendQueueLock = new Object();  
	
	// Queueコネクションに対する処理が複数スレッドから呼ばれないようにするためのロック用オブジェクト  
	// 再接続処理でコネクションを生成し直すと別オブジェクトとなる
	private volatile Object connectionLock = new Object();  
	
	/**
	 * コンストラクタ
	 */
	public SendQueue(Agent agent) {
		super();
		m_agent = agent;
		
		// ReciveTopic再接続処理実行間隔取得
		String interval = AgentProperties.getProperty("sendqueue.reconnection.interval");
		if (interval != null) {
			try {
				// プロパティファイルには秒で記述されているため、1000をかける
				m_sendQueueReconnectionInterval = Integer.parseInt(interval);
				log.info("sendqueue.reconnection.interval = " + m_sendQueueReconnectionInterval + " sec");
			} catch (NumberFormatException e) {
				log.error("sendqueue.reconnection.interval",e);
			}
		}
	}

	/**
	 *  再接続処理
	 */
	private boolean reInitial() {
		// 単一スレッドのみ以下の処理が実行できるように、クラスインスタンスのロックオブジェクトで、
		// 同期化しシリアルに動作するようにする。
		synchronized(reInitialLock){
			if(m_isErr == false){
				// 別スレッドで接続が復旧しているため、何もせずに処理を抜ける
				return true;
			}
			
			log.debug("reInitial() start");

			ExecutorService es = null;
			Future<Boolean> task = null;
			try {
				// Executorオブジェクトの生成
				ReconnectorThreadFactory threadFactory = 
					new ReconnectorThreadFactory(Thread.currentThread().getName());
				es = Executors.newSingleThreadExecutor(threadFactory);
				task = es.submit(new Reconnector());

				return task.get(RECONNECT_TIMEOUT, TimeUnit.MILLISECONDS);
			} catch (Exception e) {
				log.error(e.getMessage(), e);
				m_isErr = true;
				return false;
			} finally {
				// タスクをキャンセル
				task.cancel(true);
				
				if (es != null) {
					es.shutdown();
				}
				log.debug("reInitial() end");
			}
		}
	}
	/**
	 * サーバ接続の終了処理
	 *  
	 */
	public void terminate() {
		log.debug("terminate() start");

		try {
			if (m_session != null)
				m_session.close();
		} catch (JMSException e) {
			log.debug("terminate() session close error", e);
		}

		try {
			if (m_con != null)
				m_con.close();
		} catch (JMSException e) {
			log.debug("terminate() connection close error", e);
		}
		log.debug("terminate() finish");
	}

	/**
	 * メッセージをマネージャに送信します。<BR>
	 * 
	 * 実行応答メッセージを送信する。<BR>
	 * 処理失敗は1回再試行する。
	 * 
	 * @param msg
	 */
	public boolean put(Object info) {
		// 単一スレッドのみ以下の処理が実行できるように、クラスインスタンスのロックオブジェクト（sendQueueLock）で、
		// 同期化しシリアルに動作するようにする。
		synchronized(sendQueueLock){
			log.debug("put() start : " + info.getClass().getCanonicalName());
			
			String id = "";
			boolean sendQueueStatus = false;
			while (true) {
				log.debug("put() while (true)");

				ExecutorService es = null;
				Future<Boolean> task = null;
				try {
					// Executorオブジェクトの生成
					if (info instanceof OutputNotifyGroupInfo) {
						id = ((OutputNotifyGroupInfo)info).getMonitorId();
						log.debug("put() : OutputNotifyGroupInfo : " + id);
					} else if(info instanceof Dummy){
						id = Dummy.dummyID;
						log.debug("put() : Dummy : " + info.hashCode());
					} else {
						id = info.toString();
						log.debug("put() : String : " + id);
					}
					
					SenderThreadFactory threadFactory = new SenderThreadFactory(id);
					es = Executors.newSingleThreadExecutor(threadFactory);

					// Queue送信タスクを実行する。
					// 別スレッドでQueue送信処理を実行することで、送信処理に無限の時間がかかっても、
					// Future.get()のタイムアウトにより、本スレッドに制御が戻るようにする。
					log.debug("put() submit");
					task = es.submit(new Sender(info));
					sendQueueStatus = task.get(SEND_TIMEOUT, TimeUnit.MILLISECONDS);

					// Queue送信に成功した場合はループを抜ける
					if(sendQueueStatus == true){
						log.debug("put() return true : " + info.getClass().getCanonicalName());
						return true;
					}
				} catch (Exception e) {
					// Queue送信処理で例外が発生した場合、もしくは、Future.get()でタイムアウトが発生した場合に、
					// ここに制御が移る
					
					// ログファイルにログ出力
					log.error("put() : Failed to connect to MGR", e);
					// syslogにログ出力
					InternalLogger.error("hinemos_logagent: Failed to connect to MGR");
					
				} finally {
					// タスクをキャンセル
					task.cancel(true);
					
					if (es != null) {
						es.shutdown();
					}
					
					log.debug("put() end    : " + info.getClass().getCanonicalName());
				}
				
				// 送信が成功していない場合はsleep後に再送を試みる
				if(!sendQueueStatus){
					// 一定時間sleepした後、QueueConnection、QueueSession を再接続する。
					try {
						Thread.sleep(m_sendQueueReconnectionInterval*1000);
					} catch (InterruptedException e1) {
						// 何もしない
						log.error(e1.getMessage(), e1);
					}

					m_isErr = true;
					
					// Dummy Objectの場合はwhileループを抜ける
					if(id.equals(Dummy.dummyID)){
						break; 
					}
				}
			} // End While Loop
			
			log.debug("put() return false : " + info.getClass().getCanonicalName());
			return false;
		}
	}


	/**
	 * 接続初期化します。<BR>
	 * 
	 * @return
	 */
	private boolean initial() {

		log.info("initial() start");

		InitialContext con = null;

		try {
			//InitialContextの生成
			con = new InitialContext(AgentProperties.getProperties());

			//コネクションファクトリ生成
			m_factory = (QueueConnectionFactory) con.lookup(QUEUE_CON_FACTORY);

			//コネクション生成
			if (AgentProperties.getProperty(QUEUE_USER_NAME) != null) {
				//ユーザ認証
				m_con = m_factory.createQueueConnection(
						AgentProperties.getProperty(QUEUE_USER_NAME), 
						AgentProperties.getProperty(QUEUE_USER_PASSWORD));
			} else {
				//ユーザ認証なし
				m_con = m_factory.createQueueConnection();
			}

			
			//セッション生成
			m_session = m_con.createQueueSession(false, Session.AUTO_ACKNOWLEDGE);
			
			//エラーハンドラセット
			m_con.setExceptionListener(this);
			
			//コネクション開始
			m_con.start();

			//メッセージQueue取得
			m_queueLog = (Queue) con.lookup(QueueConstant.QUEUE_NAME_EVENT);
			m_queueFacility = (Queue) con.lookup(com.clustercontrol.logtransfer.bean.QueueConstant.QUEUE_NAME_FILE_MANAGE);

			
			

		} catch (Exception e) {
			log.error("initial() fail!", e);
			m_isErr = true;
			return false;
		} finally {
			try {
				if (con != null)
					con.close();
			} catch (Exception e) {
				log.warn("initial() close fail!", e);
			}
		}
		log.info("initial() success!");
		return true;

	}

	/* 通信エラーハンドラ
	 * (non-Javadoc)
	 * @see javax.jms.ExceptionListener#onException(javax.jms.JMSException)
	 */
	public void onException(JMSException arg0) {

		log.error(arg0.getMessage(), arg0);

		m_agent.reConnection();

	}

	/**
	 * 再接続処理を行う
	 */
	public void reconnect() {
		m_isErr = true;
	}
	
	/**
	 * 接続状況を返す
	 * @return 接続状況(再接続中 : true, 接続中　: false)
	 */
	synchronized public boolean isReconnecting() {
		return m_isErr;
	}
	
	/**
	 * Queueメッセージ送信処理を実行するタスク用クラス
	 */
	private class Sender implements Callable<Boolean> {
		private Object m_info;

		public Sender(Object info){
			m_info = info;
		}

		public Boolean call() throws Exception {
			if (m_isErr) {
				if(reInitial() == false){
					if(m_info instanceof Dummy){
						// Dummyの場合は一度のreInitialであきらめる
						return true;
					}else{
						// 再接続に失敗した場合
						return false;
					}
				}
			}

			if(m_session == null){
				return false;
			}

			QueueSender sender = null;
			QueueSender senderEvent = null;
			
			try {
				// 複数スレッドがQueueConnection, QueueSessionのインスタンスにアクセスしないように、
				// 送信処理を同期化する。
				synchronized(connectionLock){
					Message mess = null;
					if (m_info instanceof OutputBasicInfo) {
						//監視機能へ送信
						//Sender作成
						sender = m_session.createSender(m_queueLog);
						//メッセージ作成
						mess = m_session.createObjectMessage((OutputBasicInfo)m_info);

						if (log.isDebugEnabled()) {
							OutputBasicInfo loginfo = (OutputBasicInfo)m_info;
							log.debug("Sender Send OutputBasicInfo : "
									+ "Facility ID : " + loginfo.getFacilityId() + ", "
									+ "Monitor ID : " + loginfo.getMonitorId() + ", "
									+ "Priority : " + loginfo.getPriority() + ", "
									+ "Message ID : " + loginfo.getMessageId() + ", "
									+ "Message : " + loginfo.getMessage());
						}
						
						//送信
						sender.send(mess);

					}else if (m_info instanceof String) {
						//ログ転送へファシリティID送信
						//Sender作成
						sender = m_session.createSender(m_queueFacility);
						//メッセージ作成
						mess = m_session.createTextMessage((String)m_info);

						log.debug("Sender Send String : notify Fasility ID : " + m_info);
						
						//送信
						sender.send(mess);

					}else if (m_info instanceof Dummy) {
						log.debug("Sender Send Dummy Object!");
						//ログ転送へDummyオブジェクトを送信(ログ転送キュー)
						//Sender作成
						sender = m_session.createSender(m_queueFacility);
						sender.setDeliveryMode(DeliveryMode.NON_PERSISTENT);
						//メッセージ作成
						mess = m_session.createObjectMessage((Dummy)m_info);

						//送信
						sender.send(mess);

						//ログ転送へDummyオブジェクトを送信(イベントキュー)
						//Sender作成
						senderEvent = m_session.createSender(m_queueLog);
						senderEvent.setDeliveryMode(DeliveryMode.NON_PERSISTENT);

						//メッセージ作成
						mess = m_session.createObjectMessage((Dummy)m_info);

						//送信
						senderEvent.send(mess);

					} else {
						log.error("Sender Send Object is not unknown = " + m_info.getClass().getCanonicalName());
						return false;
					}

				}

				return true;
			} catch (Exception e) {
				throw e;
			} finally {
				try {
					if (sender != null)
						sender.close();
				} catch (Exception e) {
					log.error(e.getMessage(), e);
				}
				try{
					if (senderEvent != null)
						senderEvent.close();
				} catch (Exception e) {
					log.error(e.getMessage(), e);
				}
			}
		}
	}

	/**
	 * Queueメッセージ送信処理を実行するタスク用のThreadFactory
	 */
	private class SenderThreadFactory implements ThreadFactory {
		private final String m_threadName;
		
		public SenderThreadFactory(String threadName){
			m_threadName = threadName;
		}
		
		public Thread newThread(Runnable r) {
			return new Thread(r, "Sender-" + m_threadName);
		}
	}
	
	/**
	 * 再接続処理を実行するタスク用クラス
	 */
	private class Reconnector implements Callable<Boolean> {
		public Boolean call() throws Exception {
			boolean ret = false;

			log.info("SendQueue.Reconnector start");

			try{
				terminate();

				if (initial()) {

					ret = true;

					log.info("SendQueue.Reconnector success!");

					//エラーフラグ解除
					m_isErr = false;

				} else {
					log.warn("SendQueue.Reconnector fail!");
				}
			} catch (Exception e) {
				log.error("SendQueue.Reconnector error", e);
			}
			
			return ret;
		}
	}

	/**
	 * 再接続処理を実行するタスク用のThreadFactory
	 */
	private class ReconnectorThreadFactory implements ThreadFactory {
		private final String m_threadName;
		
		public ReconnectorThreadFactory(String threadName){
			m_threadName = threadName;
		}
		
		public Thread newThread(Runnable r) {
			return new Thread(r, "Reconnector-" + m_threadName);
		}
	}
}
