/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.composite;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Tree;

import com.clustercontrol.jobmanagement.util.JobEndpointWrapper;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.jobmanagement.InvalidRole_Exception;
import com.clustercontrol.ws.jobmanagement.JobInfo;
import com.clustercontrol.ws.jobmanagement.JobTreeItem;

/**
 * ジョブツリー用のコンポジットクラスです。
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class JobTreeComposite extends Composite {

	// ログ
	private static Log m_log = LogFactory.getLog( JobTreeComposite.class );

	/** ツリービューア */
	private TreeViewer m_viewer = null;
	/** 選択ジョブツリーアイテム */
	private JobTreeItem m_selectItem = null;
	/** ツリーのみ */
	private boolean m_treeOnly = false;
	private String m_jobId = null;

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親コンポジット
	 * @param style スタイル
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public JobTreeComposite(Composite parent, int style) {
		super(parent, style);

		m_treeOnly = false;

		initialize();
	}

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親コンポジット
	 * @param style スタイル
	 * @param treeOnly true：ツリーのみ、false：ジョブ情報を含む
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public JobTreeComposite(Composite parent, int style, boolean treeOnly) {
		super(parent, style);

		m_treeOnly = treeOnly;

		initialize();
	}

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親コンポジット
	 * @param style スタイル
	 * @param parentJobId 親ジョブID
	 * @param jobId ジョブID
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public JobTreeComposite(Composite parent, int style, JobTreeItem selectItem) {
		super(parent, style);

		m_treeOnly = true;
		m_selectItem = selectItem;
		m_jobId = selectItem.getData().getId();

		initialize();
	}

	/**
	 * コンポジットを構築します。
	 */
	private void initialize() {
		GridLayout layout = new GridLayout(1, true);
		this.setLayout(layout);
		layout.marginHeight = 0;
		layout.marginWidth = 0;

		Tree tree = new Tree(this, SWT.SINGLE | SWT.BORDER);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		tree.setLayoutData(gridData);

		m_viewer = new TreeViewer(tree);
		m_viewer.setContentProvider(new JobTreeContentProvider());
		m_viewer.setLabelProvider(new JobTreeLabelProvider());

		// 選択アイテム取得イベント定義
		m_viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			@Override
			public void selectionChanged(SelectionChangedEvent event) {
				StructuredSelection selection = (StructuredSelection) event
				.getSelection();

				m_selectItem = (JobTreeItem) selection.getFirstElement();
			}
		});

		update();
	}

	/**
	 * このコンポジットが利用するツリービューアを返します。
	 * 
	 * @return ツリービューア
	 */
	public TreeViewer getTreeViewer() {
		return m_viewer;
	}

	/**
	 * このコンポジットが利用するツリーを返します。
	 * 
	 * @return ツリー
	 */
	public Tree getTree() {
		return m_viewer.getTree();
	}

	/**
	 * ツリービューアーを更新します。<BR>
	 * ジョブツリー情報を取得し、ツリービューアーにセットします。
	 * <p>
	 * <ol>
	 * <li>ジョブツリー情報を取得します。</li>
	 * <li>ツリービューアーにジョブツリー情報をセットします。</li>
	 * </ol>
	 * 
	 * @see com.clustercontrol.jobmanagement.action.GetJobTree#getJobTree(boolean)
	 */
	@Override
	public void update() {
		JobTreeItem jobTree = null;

		//　ジョブ一覧情報取得
		if (m_jobId == null) {
			try {
				jobTree = JobEndpointWrapper.getJobTree(m_treeOnly);
			} catch (InvalidRole_Exception e) {
				// アクセス権なしの場合、エラーダイアログを表示する
				MessageDialog.openInformation(
						null,
						Messages.getString("message"),
						Messages.getString("message.accesscontrol.16"));
			} catch (Exception e) {
				m_log.warn("update() getJobTree, " + e.getMessage(), e);
				MessageDialog.openError(
						null,
						Messages.getString("failed"),
						Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
			}
		} else {
			jobTree = getJobTreeOneLevel(m_selectItem);
		}
		m_selectItem = null;

		m_viewer.setInput(jobTree);

		//ジョブユニットのレベルまで展開
		m_viewer.expandToLevel(2);
	}

	/**
	 * 選択ジョブツリーアイテムを返します。
	 * 
	 * @return ジョブツリーアイテム
	 */
	public JobTreeItem getSelectItem() {
		return m_selectItem;
	}

	/**
	 * 選択ジョブツリーアイテムを設定
	 * 
	 * @param item ジョブツリーアイテム
	 */
	public void setSelectItem(JobTreeItem item) {
		m_selectItem = item;
	}
	/**
	 * ジョブ[一覧]ビューのジョブツリー情報から、<BR>
	 * 引数で渡された親ジョブIDの直下のジョブツリーアイテムを取得する。<BR><BR>
	 * 取得したジョブツリーアイテムから、<BR>
	 * 引数で渡されたジョブIDと一致するジョブツリーアイテムを除いたジョブツリーアイテムを返す。
	 * 
	 * @return ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 */
	public JobTreeItem getJobTreeOneLevel(JobTreeItem self) {
		JobTreeItem parentOrg = self.getParent();

		// selfの親
		JobTreeItem parentClone = new JobTreeItem();
		parentClone.setData(copyJobInfo(parentOrg.getData()));

		// selfの兄弟
		String jobId = self.getData().getId();
		for (JobTreeItem brotherOrg : self.getParent().getChildren()) {
			if (!jobId.equals(brotherOrg.getData().getId())) {
				JobTreeItem brotherClone = new JobTreeItem();
				brotherClone.setParent(parentClone);
				parentClone.getChildren().add(brotherClone);
				brotherClone.setData(copyJobInfo(brotherOrg.getData()));
			}
		}
		return parentClone;
	}

	/**
	 * 引数で渡されたジョブ情報のコピーインスタンスを作成する。
	 * 
	 * @param orgInfo コピー元ジョブ情報
	 * @return ジョブ情報
	 */
	private JobInfo copyJobInfo(JobInfo orgInfo) {

		JobInfo info = new JobInfo();
		info.setJobunitId(orgInfo.getJobunitId());
		info.setId(orgInfo.getId());
		info.setName(orgInfo.getName());
		info.setType(orgInfo.getType());

		return info;
	}
}