/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.view.action;

import java.util.Date;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.ui.IViewActionDelegate;
import org.eclipse.ui.IViewPart;

import com.clustercontrol.jobmanagement.composite.JobTreeComposite;
import com.clustercontrol.jobmanagement.util.JobEndpointWrapper;
import com.clustercontrol.jobmanagement.view.JobListView;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.jobmanagement.InvalidRole_Exception;
import com.clustercontrol.ws.jobmanagement.JobInvalid_Exception;
import com.clustercontrol.ws.jobmanagement.JobTreeItem;

/**
 * ジョブ[一覧]ビューの「登録」のクライアント側アクションクラス<BR>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class RegisterJobAction implements IViewActionDelegate {

	// ログ
	private static Log m_log = LogFactory.getLog( RegisterJobAction.class );

	/** ビュー */
	protected IViewPart m_view;

	/**
	 * ビューを保持します。ワークベンチにロードされた際に呼ばれます。
	 * 
	 * @param view ビューのインスタンス
	 * 
	 * @see org.eclipse.ui.IViewActionDelegate#init(org.eclipse.ui.IViewPart)
	 */
	@Override
	public void init(IViewPart view) {
		m_view = view;
	}

	/**
	 * ジョブ[一覧]ビューの「登録」が押された場合に、ジョブを登録します。
	 * <p>
	 * <ol>
	 * <li>登録の確認ダイアログを表示します。</li>
	 * <li>ジョブ[一覧]ビューからジョブツリーアイテムを取得します。</li>
	 * <li>ジョブツリーアイテムを登録します。</li>
	 * </ol>
	 * 
	 * @see org.eclipse.ui.IActionDelegate#run(org.eclipse.jface.action.IAction)
	 * @see com.clustercontrol.jobmanagement.view.JobListView
	 * @see com.clustercontrol.jobmanagement.composite.JobTreeComposite
	 * @see com.clustercontrol.jobmanagement.action.RegisterJob#registerJob(JobTreeItem)
	 */
	@Override
	public void run(IAction action) {
		if (m_view instanceof JobListView) {
			// 確認ダイアログを生成
			if (MessageDialog.openQuestion(
					null,
					Messages.getString("confirmed"),
					Messages.getString("message.job.31"))) {

				JobListView view = (JobListView) m_view
				.getAdapter(JobListView.class);
				JobTreeComposite tree = view.getJobTreeComposite();
				JobTreeItem item = (JobTreeItem) tree.getTreeViewer()
				.getInput();

				// ジョブ登録
				m_log.debug("registerJob start " + new Date());

				try {
					// ジョブ登録(ID重複結果、待ち条件チェックにERRORがある場合は例外発生)
					JobEndpointWrapper.registerJob(item);
					m_log.debug("registerJob end   " + new Date());
					MessageDialog.openInformation(null, Messages.getString("message"),
							Messages.getString("regist.success"));
					// 登録に成功した場合は、編集フラグをfalseに戻して、実行可能な状態とする。
					view.setUpdate(false);
				} catch (JobInvalid_Exception e) {
					MessageDialog.openWarning(
							null,
							Messages.getString("message.hinemos.1"),
							e.getMessage());
				} catch (InvalidRole_Exception e) {
					MessageDialog.openInformation(null, Messages.getString("message"),
							Messages.getString("message.accesscontrol.16"));
				} catch (Exception e) {
					m_log.warn("run(), " + e.getMessage(), e);
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
				}

				// 再表示のため親の参照を戻す
				JobEndpointWrapper.setTreeParent(item);
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IActionDelegate#selectionChanged(org.eclipse.jface.action.IAction, org.eclipse.jface.viewers.ISelection)
	 */
	@Override
	public void selectionChanged(IAction action, ISelection selection) {

	}
}