/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.view.action;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.ui.IViewActionDelegate;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.PlatformUI;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.PropertyConstant;
import com.clustercontrol.jobmanagement.action.OperationJob;
import com.clustercontrol.jobmanagement.bean.JobOperationConstant;
import com.clustercontrol.jobmanagement.bean.OperationConstant;
import com.clustercontrol.jobmanagement.composite.DetailComposite;
import com.clustercontrol.jobmanagement.composite.HistoryComposite;
import com.clustercontrol.jobmanagement.composite.NodeDetailComposite;
import com.clustercontrol.jobmanagement.dialog.JobOperationDialog;
import com.clustercontrol.jobmanagement.util.JobEndpointWrapper;
import com.clustercontrol.jobmanagement.view.JobDetailView;
import com.clustercontrol.jobmanagement.view.JobHistoryView;
import com.clustercontrol.jobmanagement.view.JobNodeDetailView;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.jobmanagement.InvalidRole_Exception;

/**
 * ジョブ[履歴]・ジョブ[ジョブ詳細]・ジョブ[ノード詳細]ビューの「停止」のクライアント側アクションクラス<BR>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class StopJobAction implements IViewActionDelegate {

	// ログ
	private static Log m_log = LogFactory.getLog( StartJobAction.class );

	/** アクションID */
	public static final String ID = "com.clustercontrol.jobmanagement.view.action.StopJobAction";
	/** ビュー */
	protected IViewPart m_view;

	/**
	 * ビューを保持します。ワークベンチにロードされた際に呼ばれます。
	 * 
	 * @param view ビューのインスタンス
	 * 
	 * @see org.eclipse.ui.IViewActionDelegate#init(org.eclipse.ui.IViewPart)
	 */
	@Override
	public void init(IViewPart view) {
		m_view = view;
	}

	/**
	 * ジョブ[履歴]・ジョブ[ジョブ詳細]・ジョブ[ノード詳細]ビューの「停止」が押された場合に、ジョブの停止操作を行います。
	 * <p>
	 * <ol>
	 * <li>ジョブ[履歴]・ジョブ[ジョブ詳細]ビューの場合、ビューからセッションID・ジョブIDを取得します。</li>
	 * <li>ジョブ[ノード詳細]ビューの場合、ビューから、セッションID・ジョブID・ファシリティIDを取得します。</li>
	 * <li>ジョブ停止操作用プロパティを取得します。</li>
	 * <li>ジョブ[停止]ダイアログを表示します。</li>
	 * <li>ジョブ[停止]ダイアログからジョブ停止操作用プロパティを取得します。</li>
	 * <li>ジョブ停止操作用プロパティを元にジョブ停止操作を行います。</li>
	 * </ol>
	 * 
	 * @see org.eclipse.ui.IActionDelegate#run(org.eclipse.jface.action.IAction)
	 * @see com.clustercontrol.jobmanagement.view.JobHistoryView
	 * @see com.clustercontrol.jobmanagement.view.JobDetailView
	 * @see com.clustercontrol.jobmanagement.view.JobNodeDetailView
	 * @see com.clustercontrol.jobmanagement.dialog.JobOperationDialog
	 * @see com.clustercontrol.jobmanagement.action.OperationJob#operationJob(Property)
	 */
	@Override
	public void run(IAction action) {
		String sessionId = null;
		String jobunitId = null;
		String jobId = null;
		String facilityId = null;

		if (m_view instanceof JobHistoryView) { // ボタンが押された場合
			JobHistoryView view = (JobHistoryView) m_view
			.getAdapter(JobHistoryView.class);
			HistoryComposite historyComposite = view.getComposite();
			//セッションID取得
			sessionId = historyComposite.getSessionId();
			//ジョブユニットID取得
			jobunitId = historyComposite.getJobunitId();
			//ジョブID取得
			jobId = historyComposite.getJobId();
		} else if (m_view instanceof JobDetailView) { // ボタンが押された場合
			JobDetailView view = (JobDetailView) m_view
			.getAdapter(JobDetailView.class);
			DetailComposite detailComposite = view.getComposite();
			//セッションID取得
			sessionId = detailComposite.getSessionId();
			//ジョブユニットID取得
			jobunitId = detailComposite.getJobunitId();
			//ジョブID取得
			jobId = detailComposite.getJobId();
		} else if (m_view instanceof JobNodeDetailView) { // ボタンが押された場合
			JobNodeDetailView view = (JobNodeDetailView) m_view
			.getAdapter(JobNodeDetailView.class);
			NodeDetailComposite nodeDetailComposite = view.getComposite();
			//セッションID取得
			sessionId = nodeDetailComposite.getSessionId();
			//ジョブユニットID取得
			jobunitId = nodeDetailComposite.getJobunitId();
			//ジョブID取得
			jobId = nodeDetailComposite.getJobId();
			//ファシリティID取得
			facilityId = nodeDetailComposite.getFacilityId();
			if (facilityId == null) {
				sessionId = null;
				jobunitId = null;
				jobId = null;
			}
		}

		if (sessionId != null && sessionId.length() > 0 && jobunitId != null
				&& jobunitId.length() > 0 && jobId != null
				&& jobId.length() > 0) {

			JobOperationDialog dialog = new JobOperationDialog(PlatformUI
					.getWorkbench().getActiveWorkbenchWindow().getShell());

			//プロパティ設定
			dialog.setProperty(getStopProperty(sessionId, jobunitId, jobId, facilityId));
			dialog.setTitleText(Messages.getString("job") + "["
					+ Messages.getString("stop") + "]");

			//ダイアログ表示
			if (dialog.open() == IDialogConstants.OK_ID) {
				//ジョブ停止
				OperationJob operation = new OperationJob();
				operation.operationJob(dialog.getProperty());
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IActionDelegate#selectionChanged(org.eclipse.jface.action.IAction, org.eclipse.jface.viewers.ISelection)
	 */
	@Override
	public void selectionChanged(IAction action, ISelection selection) {

	}


	/**
	 * 停止用プロパティ取得
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param facilityId ファシリティID
	 * @return ジョブ停止操作用プロパティ
	 * 
	 */
	private Property getStopProperty(String sessionId, String jobunitId, String jobId, String facilityId) {
		Locale locale = Locale.getDefault();

		//セッションID
		Property session =
			new Property(JobOperationConstant.SESSION, Messages.getString("session.id", locale), PropertyConstant.EDITOR_TEXT);
		//ジョブユニットID
		Property jobUnit =
			new Property(JobOperationConstant.JOB_UNIT, Messages.getString("jobunit.id", locale), PropertyConstant.EDITOR_TEXT);
		//ジョブID
		Property job =
			new Property(JobOperationConstant.JOB, Messages.getString("job.id", locale), PropertyConstant.EDITOR_TEXT);
		//ファシリティID
		Property facility =
			new Property(JobOperationConstant.FACILITY, Messages.getString("facility.id", locale), PropertyConstant.EDITOR_TEXT);
		//制御
		Property control =
			new Property(JobOperationConstant.CONTROL, Messages.getString("control", locale), PropertyConstant.EDITOR_SELECT);
		//終了値
		Property endValue =
			new Property(JobOperationConstant.END_VALUE, Messages.getString("end.value", locale), PropertyConstant.EDITOR_NUM,
					DataRangeConstant.SMALLINT_HIGH, DataRangeConstant.SMALLINT_LOW);

		ArrayList<Property> endList = new ArrayList<Property>();
		endList.add(endValue);

		HashMap<String, Object> skipEndMap = new HashMap<String, Object>();
		skipEndMap.put("value", OperationConstant.STRING_STOP_SKIP);
		skipEndMap.put("property", endList);

		HashMap<String, Object> mainteEndMap = new HashMap<String, Object>();
		mainteEndMap.put("value", OperationConstant.STRING_STOP_MAINTENANCE);
		mainteEndMap.put("property", endList);

		HashMap<String, Object> forceEndMap = new HashMap<String, Object>();
		forceEndMap.put("value", OperationConstant.STRING_STOP_FORCE);
		forceEndMap.put("property", endList);

		List<String> values1 = null;
		try {
			values1 = JobEndpointWrapper.getAvailableStopOperation(sessionId, jobunitId, jobId, facilityId);
		} catch (InvalidRole_Exception e) {
			MessageDialog.openInformation(null, Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
		} catch (Exception e) {
			m_log.warn("getStopProperty(), " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}

		ArrayList<Object> values2 = new ArrayList<Object>();
		if(values1.contains(OperationConstant.STRING_STOP_AT_ONCE)) {
			values2.add(OperationConstant.STRING_STOP_AT_ONCE);
		}
		if(values1.contains(OperationConstant.STRING_STOP_SUSPEND)){
			values2.add(OperationConstant.STRING_STOP_SUSPEND);
		}
		if(values1.contains(OperationConstant.STRING_STOP_WAIT)){
			values2.add(OperationConstant.STRING_STOP_WAIT);
		}
		if(values1.contains(OperationConstant.STRING_STOP_SKIP)) {
			values2.add(skipEndMap);
		}
		if(values1.contains(OperationConstant.STRING_STOP_MAINTENANCE)) {
			values2.add(mainteEndMap);
		}
		if(values1.contains(OperationConstant.STRING_STOP_FORCE)) {
			values2.add(forceEndMap);
		}


		//値を初期化
		Object controlValues[][] = {values1.toArray(), values2.toArray()};

		control.setSelectValues(controlValues);
		if(values1.size() >= 1){
			control.setValue(values1.get(0));
		}
		else{
			control.setValue("");
		}

		endValue.setValue("");

		session.setValue(sessionId);
		jobUnit.setValue(jobunitId);
		job.setValue(jobId);
		if(facilityId != null && facilityId.length() > 0){
			facility.setValue(facilityId);
		}
		else{
			facility.setValue("");
		}

		//変更の可/不可を設定
		session.setModify(PropertyConstant.MODIFY_NG);
		jobUnit.setModify(PropertyConstant.MODIFY_NG);
		job.setModify(PropertyConstant.MODIFY_NG);
		facility.setModify(PropertyConstant.MODIFY_NG);
		control.setModify(PropertyConstant.MODIFY_OK);
		endValue.setModify(PropertyConstant.MODIFY_OK);

		Property property = new Property(null, null, "");

		// 初期表示ツリーを構成。
		property.removeChildren();
		property.addChildren(session);
		property.addChildren(jobUnit);
		property.addChildren(job);
		if(facilityId != null && facilityId.length() > 0){
			property.addChildren(facility);
		}
		property.addChildren(control);

		if(OperationConstant.STRING_STOP_SKIP.compareTo((String)control.getValue()) == 0 ||
				OperationConstant.STRING_STOP_MAINTENANCE.compareTo((String)control.getValue()) == 0 ||
				OperationConstant.STRING_STOP_FORCE.compareTo((String)control.getValue()) == 0 ){
			control.addChildren(endValue);
		}

		return property;
	}
}