/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.dialog;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;

import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.notify.action.DeleteNotify;
import com.clustercontrol.notify.action.GetNotify;
import com.clustercontrol.notify.action.ModifyNotify;
import com.clustercontrol.notify.action.GetNotifyTableDefine;
import com.clustercontrol.notify.composite.NotifyListComposite;
import com.clustercontrol.notify.view.action.NotifyModifyAction;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.notify.NotifyInfo;
import com.clustercontrol.ws.notify.NotifyRelationInfo;

/**
 * 通知[一覧]ダイアログクラス<BR>
 * 
 * @version 3.0.0
 * @since 2.0.0
 */
public class NotifyListDialog extends CommonDialog {

	// ログ
	private static Log m_log = LogFactory.getLog( NotifyListDialog.class );

	// 後でpackするためsizeXはダミーの値。
	private final int sizeX = 800;
	private final int sizeY = 400;

	/** 通知一覧 コンポジット。 */
	private NotifyListComposite notifyList = null;

	/** 追加 ボタン。 */
	private Button buttonAdd = null;

	/** 変更 ボタン。 */
	private Button buttonModify = null;

	/** 削除 ボタン。 */
	private Button buttonDelete = null;

	/** 有効ボタン */
	private Button buttonValid = null;

	/** 無効ボタン */
	private Button buttonInvalid = null;

	/*通知を選択するかのフラグ*/
	boolean isSelect = false;

	/***/
	List<NotifyRelationInfo> notify;

	/**
	 * ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 */
	public NotifyListDialog(Shell parent) {
		super(parent);
		setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);
	}


	/**
	 * ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 */
	public NotifyListDialog(Shell parent, boolean isSelect) {

		super(parent);
		setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);

		/**監視などから呼ばれる場合にはtrue*/
		this.isSelect = isSelect;
	}

	public NotifyListDialog(Shell parent, boolean isSelect, int notifyIdType) {

		super(parent);
		setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);

		/**監視などから呼ばれる場合にはtrue*/
		this.isSelect = isSelect;
	}

	/**
	 * ダイアログの初期サイズを返します。
	 * 
	 * @return 初期サイズ
	 */
	@Override
	protected Point getInitialSize() {
		return new Point(sizeX, sizeY);
	}

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.notify.composite.NotifyListComposite
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		Shell shell = this.getShell();

		// タイトル
		shell.setText(Messages.getString("dialog.notify.list"));

		// レイアウト
		GridLayout layout = new GridLayout(8, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.numColumns = 8;
		parent.setLayout(layout);

		/*
		 * ログ一覧
		 */

		this.notifyList = new NotifyListComposite(parent, SWT.BORDER, isSelect);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 7;
		this.notifyList.setLayoutData(gridData);

		if(this.notify != null){
			this.notifyList.setSelectNotify(this.notify);
			this.notifyList.update();
		}


		/*
		 * 操作ボタン
		 */

		Composite composite = new Composite(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.numColumns = 1;
		composite.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 1;
		composite.setLayoutData(gridData);

		// 追加ボタン
		this.buttonAdd = this
		.createButton(composite, Messages.getString("add"));
		this.buttonAdd.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				NotifyTypeDialog dialog = new NotifyTypeDialog(getParentShell(), notifyList);
				dialog.open();
			}
		});

		// 変更ボタン
		@SuppressWarnings("unused") Label dummy = new Label(composite, SWT.NONE);
		this.buttonModify = this.createButton(composite, Messages.getString("modify"));
		this.buttonModify.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {

				String notifyId = null;
				Integer notifyType = null;

				Table table = notifyList.getTableViewer().getTable();
				TableItem[] item = table.getSelection();

				if(item != null && item.length>0){
					try{
						notifyId = (String)((ArrayList)item[0].getData()).get(GetNotifyTableDefine.NOTIFY_ID);
						notifyType = (Integer)((ArrayList)item[0].getData()).get(GetNotifyTableDefine.NOTIFY_TYPE);
					}
					catch(Exception ex){
					}
				}

				if (notifyId != null && notifyType != null) {
					NotifyModifyAction action = new NotifyModifyAction();
					if (action.openDialog(getParentShell(), notifyId, notifyType) ==
						IDialogConstants.OK_ID) {
						int selectIndex = table.getSelectionIndex();
						notifyList.update();
						table.setSelection(selectIndex);
					}
				} else{
					MessageDialog.openWarning(
							null,
							Messages.getString("warning"),
							Messages.getString("message.notify.8"));
				}
			}
		});

		// 削除ボタン
		dummy = new Label(composite, SWT.NONE);
		this.buttonDelete = this.createButton(composite, Messages
				.getString("delete"));
		this.buttonDelete.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {

				String notifyId = null;

				Table table = notifyList.getTableViewer().getTable();
				TableItem[] item = table.getSelection();

				if(item != null && item.length>0){
					try{
						notifyId = (String)((ArrayList)item[0].getData()).get(GetNotifyTableDefine.NOTIFY_ID);
					}
					catch(Exception ex){
					}
				}

				if (notifyId != null) {

					DeleteNotify deleteNotify = new DeleteNotify();

					if(deleteNotify.useCheck(notifyId) == Window.OK){ // 対象の通知IDがどの監視で使用されているかを確認

						String[] args = { notifyId };
						if (MessageDialog.openConfirm(
								null,
								Messages.getString("confirmed"),
								Messages.getString("message.notify.7", args))) {

							boolean result = deleteNotify.delete(notifyId);
							if(result){
								notifyList.update();
							}
							//            			else{
							//                        	MessageDialog.openError(
							//                        			null,
							//                        			Messages.getString("failed"),
							//                        			Messages.getString("message.notify.6", args));
							//            			}
						}
					}
				}
				else{
					MessageDialog.openWarning(
							null,
							Messages.getString("warning"),
							Messages.getString("message.notify.9"));
				}
			}
		});

		// 有効ボタン
		dummy = new Label(composite, SWT.NONE);
		dummy = new Label(composite, SWT.NONE);
		this.buttonValid = this.createButton(composite, Messages
				.getString("valid"));
		this.buttonValid.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				//一括で有効に変更
				setValid(YesNoConstant.TYPE_YES);
			}
		});

		// 無効ボタン
		this.buttonInvalid = this.createButton(composite, Messages
				.getString("invalid"));
		this.buttonInvalid.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				//一括で無効に変更
				setValid(YesNoConstant.TYPE_NO);
			}
		});

		// ラインを引く
		Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 8;
		line.setLayoutData(gridData);

		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);

		//ダイアログのサイズ調整（pack:resize to be its preferred size）
		shell.pack();
		shell.setSize(new Point(sizeX, sizeY ));

	}

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
	@Override
	protected ValidateResult validate() {
		return super.validate();
	}

	/**
	 * ボタンを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param label ラベル文字列
	 * @return 生成されたボタン
	 */
	private Button createButton(Composite parent, String label) {
		Button button = new Button(parent, SWT.NONE);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);

		button.setText(label);

		return button;
	}

	/**
	 * ボタンを生成します。<BR>
	 * 閉じるボタンを生成します。
	 * 
	 * @param parent ボタンバーコンポジット
	 */
	@Override
	protected void createButtonsForButtonBar(Composite parent) {

		if(!isSelect){
			// 閉じるボタン
			this.createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CLOSE_LABEL, false);

		}else{
			// 閉じるボタン
			this.createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.OK_LABEL, false);

		}
	}




	/**
	 * 監視にもともと設定されていた通知を反映します。
	 * 
	 * @param notify
	 */
	public void setSelectNotify(List<NotifyRelationInfo> notify){

		this.notify = notify;

	}



	@SuppressWarnings("unchecked")
	/**
	 * 
	 */
	public List<NotifyRelationInfo> getSelectNotify(){
		return this.notifyList.getSelectNotify();
	}

	/**
	 * 閉じる、キャンセルボタンが押された場合に呼ばれるメソッドで、
	 * 監視から呼ばれた場合にのみ動作させます。
	 * <p>
	 * 
	 * エラーの場合、ダイアログを閉じずにエラー内容を通知します。
	 */
	@Override
	protected void cancelPressed() {
		//監視から呼ばれた場合のみ動作
		if(isSelect){

			if(!notifyList.makeNotifyData()){
				MessageDialog.openWarning(
						null,
						Messages.getString("warning"),
						Messages.getString("message.notify.25"));
				return;
			}

		}

		//上位のcancelPressで
		super.cancelPressed();
	}

	/**
	 * 有効・無効変更処理
	 * 
	 * @param valid
	 */
	public void setValid(int valid) {

		//選択された監視項目IDを取得
		ArrayList<String> list = notifyList.getSelectionData();
		if (list != null && list.size() > 0) {

			StringBuilder notifyIds = new StringBuilder();
			for(int i = 0; i < list.size(); i++){
				if(i > 0){
					notifyIds.append(", ");
				}
				notifyIds.append(list.get(i));
			}

			String[] confirmArgs = { notifyIds.toString() };
			String message;
			if(valid == YesNoConstant.TYPE_YES)
				message = Messages.getString("message.monitor.47",confirmArgs);
			else
				message = Messages.getString("message.monitor.48",confirmArgs);
			if (!MessageDialog.openConfirm(
					null,
					Messages.getString("confirmed"),
					message)) {
				return;
			}

			for(int i = 0; i < list.size(); i++){
				String notifyId = list.get(i);

				if(notifyId != null && !notifyId.equals("")){
					//監視情報を取得
					NotifyInfo info = new GetNotify().getNotify(notifyId);

					//有効・無効を設定
					info.setValidFlg(valid);

					//監視情報を更新
					boolean result = new ModifyNotify().modify(info);
					if(!result){
						String[] args = { notifyId };
						MessageDialog.openError(
								null,
								Messages.getString("failed"),
								Messages.getString("message.monitor.36", args));
					}
				}
			}

			int selectIndex = notifyList.getTableViewer().getTable().getSelectionIndex();
			notifyList.update();
			notifyList.getTableViewer().getTable().setSelection(selectIndex);
		}
		else{
			MessageDialog.openWarning(
					null,
					Messages.getString("warning"),
					Messages.getString("message.monitor.1"));
		}
	}


}