/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.repository.dialog;

import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.action.FacilityTree;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.PropertyConstant;
import com.clustercontrol.bean.RequiredFieldColorConstant;
import com.clustercontrol.bean.SizeConstant;
import com.clustercontrol.bean.SnmpVersionConstant;
import com.clustercontrol.composite.PropertySheet;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ScopeTreeDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.repository.action.GetNodeProperty;
import com.clustercontrol.repository.bean.FacilityConstant;
import com.clustercontrol.repository.bean.NodeConstant;
import com.clustercontrol.repository.util.NodePropertyUtil;
import com.clustercontrol.repository.util.RepositoryEndpointWrapper;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.PropertyUtil;
import com.clustercontrol.ws.repository.FacilityDuplicate_Exception;
import com.clustercontrol.ws.repository.FacilityTreeItem;
import com.clustercontrol.ws.repository.InvalidRole_Exception;
import com.clustercontrol.ws.repository.NodeInfo;
import com.clustercontrol.repository.FacilityPath;

/**
 * ノードの作成・変更ダイアログクラス<BR>
 * 
 * @version 4.0.0
 * @since 1.0.0
 */
public class NodeCreateDialog extends CommonDialog {

	// ログ
	private static Log m_log = LogFactory.getLog( NodeCreateDialog.class );


	// 後でpackするためsizeXはダミーの値。
	private final int sizeX = 500;
	private final int sizeY = 800;

	// ----- instance フィールド ----- //

	/** 初期表示ノードのファシリティID */
	private String facilityId = "";

	/** 一括変更対象スコープのファシリティID */
	private String scopeId = "";

	/** 変更前のプロパティ */
	private Property propertyOld = null;

	/** 変更用ダイアログ判別フラグ */
	private boolean isModifyDialog = false;

	/** 「登録方法の選択」でのノード指定フラグ */
	private boolean isNodeSelected = true;

	/** ノード単位選択ラジオボタン */
	private Button radioCreateTypeNode = null;

	/** スコープ単位選択ラジオボタン */
	private Button radioCreateTypeScope = null;

	/** スコープ文字列表示テキスト */
	private Text textScope = null;

	/** スコープ指定ダイアログ表示ボタン */
	private Button buttonScope = null;

	/** ノード属性プロパティシート */
	protected PropertySheet propertySheet = null;

	/** クリア実行フラグ */
	private boolean isClearDone = false;

	/** クリアボタン */
	private Button clearButtonScope = null;

	//SNMPで検出する機能のための追加コンポーネント
	private Button buttonAuto = null;

	private Label    ipAddressText = null; //"IP Address";
	private Text     ipAddressBox  = null;
	private Label    communityText = null; //"community";
	private Text     communityBox  = null;
	private Label    portText      = null; //"port";
	private Text     portBox       = null;
	private Label    versionText      = null; //"version";
	private Combo    versionBox       = null;
	private TableTree table = null;

	private NodeInfo nodeInfo = null;
	
	// ----- コンストラクタ ----- //

	/**
	 * 指定した形式のダイアログのインスタンスを返します。
	 * 
	 * @param parent
	 *            親のシェルオブジェクト
	 * @param facilityId
	 *            初期表示するノードのファシリティID
	 * @param isModifyDialog
	 *            変更用ダイアログとして利用する場合は、true
	 */
	public NodeCreateDialog(Shell parent, String facilityId,
			boolean isModifyDialog) {
		super(parent);

		this.facilityId = facilityId;
		this.isModifyDialog = isModifyDialog;
	}

	// ----- instance メソッド ----- //

	/**
	 * ダイアログの初期サイズを返します。
	 * 
	 * @return 初期サイズ
	 */
	@Override
	protected Point getInitialSize() {
		return new Point(500, 600);
	}

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent
	 *            親のインスタンス
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		Shell shell = this.getShell();
		// タイトル
		shell.setText(Messages
				.getString("dialog.repository.node.create.modify"));

		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		parent.setLayout(layout);

		// 登録方法の選択(グループ)
		Group group = new Group(parent, SWT.SHADOW_NONE);
		layout = new GridLayout(1, true);
		group.setLayout(layout);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		group.setLayoutData(gridData);
		group.setText(Messages.getString("registration.method"));

		// ノード単位
		this.radioCreateTypeNode = new Button(group, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		this.radioCreateTypeNode.setLayoutData(gridData);
		this.radioCreateTypeNode.setText(Messages
				.getString("registration.method.node.base"));


		//SNMPで検索するためのコンポジット
		Composite compositeSNMP = new Composite(group, SWT.NONE);
		RowLayout rowLayoutAuto = new RowLayout();
		rowLayoutAuto.marginLeft=20;
		rowLayoutAuto.type = SWT.HORIZONTAL;
		compositeSNMP.setLayout(rowLayoutAuto);

		//SNMPによる検出のグループ(SNMP)
		Group groupAuto = new Group(compositeSNMP, SWT.RIGHT);
		layout = new GridLayout(5, false);
		groupAuto.setLayout(layout);
		if (isModifyDialog) {
			groupAuto.setText("Find By SNMP " +
					Messages.getString("repository.find.by.snmp.modify"));
		} else {
			groupAuto.setText("Find By SNMP");
		}

		//IPアドレス
		this.ipAddressText = new Label(groupAuto, SWT.NONE);
		this.ipAddressText.setText(Messages.getString("ip.address") + " : ");

		this.ipAddressBox = new Text(groupAuto, SWT.BORDER | SWT.SINGLE);
		GridData grid = new GridData(GridData.FILL_HORIZONTAL);
		grid.minimumWidth = 80;
		this.ipAddressBox.setLayoutData(grid);

		// プロパティ取得
		if (this.isModifyDialog) {
			propertyOld = new GetNodeProperty().getProperty(this.facilityId,
					PropertyConstant.MODE_MODIFY);
		} else {
			propertyOld = new GetNodeProperty().getProperty(this.facilityId,
					PropertyConstant.MODE_ADD);
		}
		Property propertyChild = null;
		propertyChild = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.IP_ADDRESS_VERSION).get(0);
		String ipAddressVersion = propertyChild.getValue().toString();
		String ipAddress = null;
		if ("6".equals(ipAddressVersion)) {
			propertyChild = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.IP_ADDRESS_V6).get(0);
			ipAddress = propertyChild.getValue().toString();
		} else {
			propertyChild = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.IP_ADDRESS_V4).get(0);
			ipAddress = propertyChild.getValue().toString();
		}
		if ("".equals(ipAddress)) {
			this.ipAddressBox.setText("192.168.0.1");
		} else {
			this.ipAddressBox.setText(ipAddress);
		}

		//ポート名
		this.portText = new Label(groupAuto, SWT.NONE);
		this.portText.setText(Messages.getString("port.number") + " : ");

		this.portBox = new Text(groupAuto, SWT.BORDER | SWT.SINGLE);
		grid = new GridData(GridData.FILL_HORIZONTAL);
		grid.minimumWidth = 50;
		this.portBox.setLayoutData(grid);
		propertyChild = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.SNMP_PORT).get(0);
		String snmpPort = propertyChild.getValue().toString();
		if ("".equals(snmpPort)) {
			this.portBox.setText("161");
		} else {
			this.portBox.setText(snmpPort);
		}

		// 改行のため、ダミーのラベルを挿入。
		Label dummy2 = new Label(groupAuto, SWT.NONE);

		//コミュニティ名
		this.communityText = new Label(groupAuto, SWT.NONE);
		this.communityText.setText(Messages.getString("community.name") + " : ");

		this.communityBox = new Text(groupAuto, SWT.BORDER | SWT.SINGLE);
		this.communityBox.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		propertyChild = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.SNMP_COMMUNITY).get(0);
		String snmpCommunity = propertyChild.getValue().toString();
		if ("".equals(snmpCommunity)) {
			this.communityBox.setText("public");
		} else {
			this.communityBox.setText(snmpCommunity);
		}

		//バージョン
		this.versionText = new Label(groupAuto, SWT.NONE);
		this.versionText.setText(Messages.getString("snmp.version") + " : ");

		this.versionBox = new Combo(groupAuto, SWT.DROP_DOWN | SWT.READ_ONLY);
		this.versionBox.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		this.versionBox.add("1",0);
		this.versionBox.add("2c",1);
		// デフォルトをv2cとする
		propertyChild = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.SNMP_VERSION).get(0);
		String snmpVersion = propertyChild.getValue().toString();
		if ("1".equals(snmpVersion)) {
			this.versionBox.select(0);
		} else {
			this.versionBox.select(1);
		}

		this.buttonAuto = new Button(groupAuto, SWT.PUSH | SWT.RIGHT);
		this.buttonAuto.setText(" Find ");
		gridData = new GridData();
		gridData.horizontalIndent= 30;
		this.buttonAuto.setLayoutData(gridData);
		this.buttonAuto.addSelectionListener(new SelectionAdapter() {
			@Override
			@SuppressWarnings("unchecked")
			public void widgetSelected(SelectionEvent e) {
				try{
					String ipAddressString = ipAddressBox.getText();

					// IPアドレスチェック
					InetAddress address = InetAddress.getByName(ipAddressString);

					if (address instanceof Inet4Address){

						//IPv4の場合はさらにStringをチェック
						if (!ipAddressString.matches(".{1,3}?\\..{1,3}?\\..{1,3}?\\..{1,3}?")){
							MessageDialog.openWarning(getShell(), "Warning",  Messages.getString("message.repository.37"));
							return;
						}

					} else if (address instanceof Inet6Address){
						//IPv6の場合は特にStringチェックは無し
					} else {
						MessageDialog.openWarning(getShell(), "Warning",  Messages.getString("message.repository.37"));
						return;
					}

					String ipAddress = ipAddressBox.getText();
					Integer port = Integer.parseInt(portBox.getText());
					String community = communityBox.getText();
					String version = SnmpVersionConstant.typeToString(new Integer(versionBox.getSelectionIndex()));


					Property propertySNMP = null;

					if (isModifyDialog) {
						propertySNMP = GetNodePropertyBySNMP(ipAddress, port, community, version,
								PropertyConstant.MODE_MODIFY);
						/*
						 * ノード変更時のfind by snmpはデバイスのみ修正する。
						 */
						// 古い値を消して(device)
						for(Object propertyObj : PropertyUtil.getProperty(propertyOld, NodeConstant.DEVICE)){
							Property p = (Property)propertyObj;
							p.getParent().removeChildren(p);
						}
						// 新しい値を入れる(device)
						for(Object propertyObj : PropertyUtil.getProperty(propertySNMP, NodeConstant.DEVICE)){
							Property p = (Property)propertyObj;
							propertyOld.addChildren(p);
						}

						/*
						 * 上記の処理だけでは、順番がおかしくなってしまうため、
						 * ユーザ等を再登録して、後ろにまわす。
						 */
						{
							Property p = null;
							p = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.SERVER_VIRTUALIZATION).get(0);
							propertyOld.removeChildren(p);
							propertyOld.addChildren(p);
							p = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.NETWORK_VIRTUALIZATION).get(0);
							propertyOld.removeChildren(p);
							propertyOld.addChildren(p);
							p = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.NODE_VARIABLE).get(0);
							propertyOld.removeChildren(p);
							propertyOld.addChildren(p);
							p = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.MAINTENANCE).get(0);
							propertyOld.removeChildren(p);
							propertyOld.addChildren(p);
							p = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.CREATE_TIME).get(0);
							propertyOld.removeChildren(p);
							propertyOld.addChildren(p);
							p = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.CREATOR_NAME).get(0);
							propertyOld.removeChildren(p);
							propertyOld.addChildren(p);
							p = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.MODIFY_TIME).get(0);
							propertyOld.removeChildren(p);
							propertyOld.addChildren(p);
							p = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.MODIFIER_NAME).get(0);
							propertyOld.removeChildren(p);
							propertyOld.addChildren(p);

							// 備考は複数含まれることがある。
							ArrayList<Object> oArray = PropertyUtil.getProperty(propertyOld, NodeConstant.NOTE);
							for (Object o : oArray) {
								propertyOld.removeChildren((Property)o);
								propertyOld.addChildren((Property)o);
							}
						}
						propertySheet.setInput(propertyOld);
					} else {
						propertySNMP = GetNodePropertyBySNMP(ipAddress, port, community, version,
								PropertyConstant.MODE_ADD);
						propertySheet.setInput(propertySNMP);
					}

					// シートの展開と色表示
					update();
				} catch (NumberFormatException e1){
					// port番号が不正な場合
					MessageDialog.openWarning(getShell(), "Warning", Messages.getString("message.repository.38"));

				} catch (UnknownHostException e2) {
					// IPアドレスが不正な場合
					MessageDialog.openWarning(getShell(), "Warning",  Messages.getString("message.repository.37"));
				}
			}
		});


		// スコープ単位
		this.radioCreateTypeScope = new Button(group, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		this.radioCreateTypeScope.setLayoutData(gridData);
		this.radioCreateTypeScope.setText(Messages
				.getString("registration.method.scope.base"));

		// スコープ指定
		Composite composite = new Composite(group, SWT.NONE);
		RowLayout rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		composite.setLayout(rowLayout);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		composite.setLayoutData(gridData);
		// ダミーラベル
		Label dummy = new Label(composite, SWT.NONE);
		dummy.setLayoutData(new RowData(12, SizeConstant.SIZE_LABEL_HEIGHT));
		// スコープ文字列テキスト
		this.textScope = new Text(composite, SWT.BORDER | SWT.LEFT
				| SWT.READ_ONLY);
		this.textScope.setText("");
		this.textScope.setLayoutData(new RowData(300,
				SizeConstant.SIZE_TEXT_HEIGHT));
		this.buttonScope = new Button(composite, SWT.PUSH);
		this.buttonScope.setText(Messages.getString("refer"));
		this.buttonScope.pack();
		if (!this.isModifyDialog) {
			this.buttonScope.setEnabled(false);
		}
		this.buttonScope.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				ScopeTreeDialog dialog = new ScopeTreeDialog(getParentShell(),
						true);
				if (dialog.open() == IDialogConstants.OK_ID) {
					FacilityTreeItem selectItem = dialog.getSelectItem();
					if (selectItem.getData().getFacilityType() == FacilityConstant.TYPE_COMPOSITE) {
						textScope.setText("");
						scopeId = "";
					} else {
						FacilityPath facilityPath = new FacilityPath(
								ClusterControlPlugin.getDefault()
								.getSeparator());
						textScope.setText(facilityPath.getPath(selectItem));
						scopeId = selectItem.getData().getFacilityId();
					}
				}
			}
		}

		);

		// ディフォルトではノード単位を選択
		this.radioCreateTypeNode.setSelection(true);
		this.textScope.setEnabled(false);
		this.buttonScope.setEnabled(false);

		// ラジオボタンのイベント
		this.radioCreateTypeNode.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				textScope.setEnabled(false);
				buttonScope.setEnabled(false);
				clearButtonScope.setEnabled(false);

				NodePropertyUtil.modifyPropertySetting((Property)propertySheet.getInput(), true);
			}
		});
		this.radioCreateTypeScope.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				textScope.setEnabled(true);
				buttonScope.setEnabled(true);
				clearButtonScope.setEnabled(true);

				NodePropertyUtil.modifyPropertySetting((Property)propertySheet.getInput(), false);
			}
		});

		// 登録処理の場合、スコープ関連は使用不可とする。
		if (!this.isModifyDialog) {
			this.radioCreateTypeScope.setEnabled(false);
			this.clearButtonScope.setEnabled(false);
		}

		/*
		 * 属性プロパティシート
		 */

		// ラベル
		Label label = new Label(parent, SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("attribute") + " : ");

		// プロパティシート
		table = new TableTree(parent, SWT.H_SCROLL | SWT.V_SCROLL
				| SWT.FULL_SELECTION | SWT.MULTI | SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 1;
		table.setLayoutData(gridData);
		table.addSelectionListener(new SelectionAdapter(){
			@Override
			public void widgetSelected(SelectionEvent e) {
				// シートの展開と色表示
				update();
			}
		});

		this.propertySheet = new PropertySheet(table);

		// プロパティ設定
		this.propertySheet.setInput(propertyOld);

		/////////////////////////////////////////////////////////////

		// ラインを引く
		Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		line.setLayoutData(gridData);

		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);

		//ダイアログのサイズ調整（pack:resize to be its preferred size）
		shell.pack();
		shell.setSize(new Point(shell.getSize().x, sizeY ));

		this.update();

	}

	/**
	 * 更新処理
	 * 
	 */
	public void update(){

		/*プロパティシートの展開を指定します。*/

		//レベル1までの展開
		this.propertySheet.expandToLevel(1);
		//サーバ基本情報
		Object element =  this.propertySheet.getElementAt(4);
		this.propertySheet.expandToLevel(element, 2);
		//サービス
		element =  this.propertySheet.getElementAt(5);
		this.propertySheet.expandToLevel(element, 1);
		//デバイス
		element =  this.propertySheet.getElementAt(6);
		this.propertySheet.expandToLevel(element, 1);


		/*必須項目を色別表示します。*/

		// ファシリティIDのインデックス：0
		if("".equals(table.getItem(0).getText(1))){
			table.getItem(0).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			table.getItem(0).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}

		// ファシリティ名のインデックス：1
		if("".equals(table.getItem(1).getText(1))){
			table.getItem(1).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			table.getItem(1).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}

		// プラットフォームのインデックス：4-0-1
		if("".equals(table.getItem(4).getItem(0).getItem(0).getText(1))){
			table.getItem(4).getItem(0).getItem(0).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			table.getItem(4).getItem(0).getItem(0).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}

		// ネットワークのインデックス：4-1
		if("4".equals((table.getItem(4).getItem(1).getItem(0).getText(1)))){
			// -IPv4のアドレスのインデックス：5
			if("".equals(table.getItem(4).getItem(1).getItem(1).getText(1))){
				table.getItem(4).getItem(1).getItem(1).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
				table.getItem(4).getItem(1).getItem(2).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			}else{
				table.getItem(4).getItem(1).getItem(1).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
				table.getItem(4).getItem(1).getItem(2).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			}
		}else if("6".equals((table.getItem(4).getItem(1).getItem(0).getText(1)))){
			// -IPv6のアドレスのインデックス：6
			if("".equals(table.getItem(4).getItem(1).getItem(2).getText(1))){
				table.getItem(4).getItem(1).getItem(2).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
				table.getItem(4).getItem(1).getItem(1).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			}else{
				table.getItem(4).getItem(1).getItem(2).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
				table.getItem(4).getItem(1).getItem(1).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			}
		}

		// OSのインデックス：8
		// -ノード名のインデックス:0
		if("".equals(table.getItem(4).getItem(2).getItem(0).getText(1))){
			table.getItem(4).getItem(2).getItem(0).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			table.getItem(4).getItem(2).getItem(0).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}

		// 仮想化のインデックス:7
		// -ノード種別のインデックス:0
		if("controller".equals(table.getItem(7).getItem(0).getText(1)) || "host".equals(table.getItem(7).getItem(0).getText(1))){
			// ノード種別が"guest"だった場合の色をクリア
			table.getItem(7).getItem(4).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			table.getItem(7).getItem(1).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			table.getItem(7).getItem(3).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			table.getItem(7).getItem(5).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);

			// -仮想化ソリューションのインデックス:4
			if("".equals(table.getItem(7).getItem(4).getText(1))){
				table.getItem(7).getItem(4).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
			}else{
				table.getItem(7).getItem(4).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			}
			// -仮想化ソフト接続ユーザのインデックス:6
			if("".equals(table.getItem(7).getItem(6).getText(1))){
				table.getItem(7).getItem(6).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
			}else{
				table.getItem(7).getItem(6).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			}
			// -仮想化ソフト接続ユーザパスワードのインデックス:7
			if("".equals(table.getItem(7).getItem(7).getText(1))){
				table.getItem(7).getItem(7).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
			}else{
				table.getItem(7).getItem(7).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			}
			// -仮想化ソフト接続プロトコルのインデックス:8
			if("".equals(table.getItem(7).getItem(8).getText(1))){
				table.getItem(7).getItem(8).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
			}else{
				table.getItem(7).getItem(8).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			}
		}else if("guest".equals(table.getItem(7).getItem(0).getText(1))){
			// ノード種別が"host"または"controller"だった場合の色をクリア
			table.getItem(7).getItem(4).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			table.getItem(7).getItem(6).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			table.getItem(7).getItem(7).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			table.getItem(7).getItem(8).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);

			// -仮想化ソリューションのインデックス:4
			if("".equals(table.getItem(7).getItem(4).getText(1))){
				table.getItem(7).getItem(4).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
			}else{
				table.getItem(7).getItem(4).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			}
			// -管理ノードのインデックス:1
			if("".equals(table.getItem(7).getItem(1).getText(1))){
				table.getItem(7).getItem(1).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
			}else{
				table.getItem(7).getItem(1).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			}
			// -仮想マシン名のインデックス:3
			if("".equals(table.getItem(7).getItem(3).getText(1))){
				table.getItem(7).getItem(3).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
			}else{
				table.getItem(7).getItem(3).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			}
			// -仮想マシンIDのインデックス:5
			if("".equals(table.getItem(7).getItem(5).getText(1))){
				table.getItem(7).getItem(5).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
			}else{
				table.getItem(7).getItem(5).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			}
		}else if(table.getItem(7).getItemCount() > 1){
			// 仮想化のツリーを開いており、ノード種別が空欄の場合
			table.getItem(7).getItem(1).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			table.getItem(7).getItem(3).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			table.getItem(7).getItem(4).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			table.getItem(7).getItem(5).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			table.getItem(7).getItem(6).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			table.getItem(7).getItem(7).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			table.getItem(7).getItem(8).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}	}

	/**
	 * 既存のボタンに加え、クリアボタンを追加します。
	 * 
	 * @param parent
	 *            ボタンバーコンポジット
	 */
	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		// クリアボタン
		clearButtonScope =
			this.createButton(parent, IDialogConstants.OPEN_ID, Messages
					.getString("clear"), false);
		this.getButton(IDialogConstants.OPEN_ID).addSelectionListener(
				new SelectionAdapter() {
					@Override
					public void widgetSelected(SelectionEvent e) {
						// プロパティ取得及び設定
						Property property = null;
						if (isModifyDialog()) {
							property = new GetNodeProperty().getProperty(null,
									PropertyConstant.MODE_MODIFY);

							//ファシリティID設定
							ArrayList list = PropertyUtil.getProperty(property,
									NodeConstant.FACILITY_ID);
							((Property) list.get(0)).setValue(getFacilityId());
						} else {
							property = new GetNodeProperty().getProperty(null,
									PropertyConstant.MODE_ADD);
						}

						//Booleanエディタのプロパティにnullを設定する
						PropertyUtil.setPropertyValue(property,
								PropertyConstant.EDITOR_BOOL, null);

						setInputData(property);

						// クリア実行フラグ
						isClearDone = true;
					}
				});

		super.createButtonsForButtonBar(parent);
	}

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
	@Override
	protected ValidateResult validate() {
		ValidateResult result = null;

		return result;
	}

	/**
	 * 入力値をマネージャに登録します。
	 * 
	 * @return true：正常、false：異常
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#action()
	 */
	@Override
	protected boolean action() {
		boolean result = false;

		Property property = this.getInputData();
		if(property != null){
			String errMessage = "";
			if(!this.isModifyDialog()){
				// 作成の場合
				Property copy = PropertyUtil.copy(property);
				PropertyUtil.deletePropertyDefine(copy);
				try {
					nodeInfo = NodePropertyUtil.property2node(copy);
					RepositoryEndpointWrapper.addNode(nodeInfo);

					// リポジトリキャッシュの更新
					FacilityTree.reflesh();

					MessageDialog.openInformation(
							null,
							Messages.getString("successful"),
							Messages.getString("message.repository.4"));

					result = true;

				} catch (FacilityDuplicate_Exception e) {
					// ファシリティIDが重複している場合、エラーダイアログを表示する
					//ファシリティID取得
					ArrayList values = PropertyUtil.getPropertyValue(copy, NodeConstant.FACILITY_ID);
					String args[] = { (String)values.get(0) };

					MessageDialog.openInformation(
							null,
							Messages.getString("message"),
							Messages.getString("message.repository.26", args));

				} catch (Exception e) {
					if (e instanceof InvalidRole_Exception) {
						// アクセス権なしの場合、エラーダイアログを表示する
						MessageDialog.openInformation(
								null,
								Messages.getString("message"),
								Messages.getString("message.accesscontrol.16"));
					} else {
						errMessage = ", " + e.getMessage();
						m_log.warn("action()", e);
					}
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.repository.5") + errMessage);
				}
			} else {
				// 変更の場合
				if (this.isNodeSelected()) {
					Property copy = PropertyUtil.copy(property);
					PropertyUtil.deletePropertyDefine(copy);
					try {
						nodeInfo = NodePropertyUtil.property2node(copy);
						RepositoryEndpointWrapper.modifyNode(nodeInfo);

						// リポジトリキャッシュの更新
						FacilityTree.reflesh();

						MessageDialog.openInformation(
								null,
								Messages.getString("successful"),
								Messages.getString("message.repository.10"));

						result = true;

					} catch (Exception e) {
						if (e instanceof InvalidRole_Exception) {
							// アクセス権なしの場合、エラーダイアログを表示する
							MessageDialog.openInformation(
									null,
									Messages.getString("message"),
									Messages.getString("message.accesscontrol.16"));
						} else {
							errMessage = ", " + e.getMessage();
						}

						MessageDialog.openError(
								null,
								Messages.getString("failed"),
								Messages.getString("message.repository.11") + errMessage);
					}

				} else {
					boolean done = false;
					if(this.isClearDone()){
						// クリアボタンがクリックされていた場合、更新処理を行う。
						done = true;
					}
					else{
						// 確認ダイアログにてOKが選択された場合、更新処理を行う。
						if (MessageDialog.openConfirm(
								null,
								Messages.getString("confirmed"),
								Messages.getString("message.repository.20"))) {
							done = true;
						}
					}

					if(done){
						// スコープ単位で更新
						Property copy = PropertyUtil.copy(property);
						PropertyUtil.deletePropertyDefine(copy);
						try {
							nodeInfo = NodePropertyUtil.property2node(copy);
							RepositoryEndpointWrapper.modifyAllNode(nodeInfo, this.getScopeId());

							// リポジトリキャッシュの更新
							FacilityTree.reflesh();

							MessageDialog.openInformation(
									null,
									Messages.getString("successful"),
									Messages.getString("message.repository.10"));

							result = true;

						} catch (Exception e) {
							if (e instanceof InvalidRole_Exception) {
								// アクセス権なしの場合、エラーダイアログを表示する
								MessageDialog.openInformation(
										null,
										Messages.getString("message"),
										Messages.getString("message.accesscontrol.16"));
							} else {
								errMessage = ", " + e.getMessage();
							}

							MessageDialog.openError(
									null,
									Messages.getString("failed"),
									Messages.getString("message.repository.11") + errMessage);
						}
					}
				}
			}
		}

		return result;
	}

	/**
	 * 変更用ダイアログなのかを返します。
	 * 
	 * @return 変更用ダイアログの場合、true
	 */
	public boolean isModifyDialog() {
		return this.isModifyDialog;
	}

	/**
	 * 「登録方法の選択」にて、ノード単位が選択されているかを返します。
	 * 
	 * @return ノード単位が選択されている場合、true
	 */
	public boolean isNodeSelected() {
		return this.isNodeSelected;
	}

	/**
	 * 一括変更対象スコープのファシリティIDを返します。
	 * 
	 * @return 一括変更対象スコープのファシリティID
	 */
	public String getScopeId() {
		return this.scopeId;
	}

	/**
	 * クリアボタンがクリックされているかを返します。
	 * 
	 * @return クリアボタンがクリックされている場合、true
	 */
	public boolean isClearDone() {
		return this.isClearDone;
	}

	/**
	 * 入力値を保持したデータモデルを生成します。
	 * 
	 * @return データモデル
	 */
	public Property getInputData() {
		return (Property) this.propertySheet.getInput();
	}

	/**
	 * 入力値を保持したデータモデルを設定します。
	 * 
	 * @param property
	 */
	public void setInputData(Property property) {
		propertySheet.setInput(property);
		this.update();
	}

	/**
	 * @return Returns the facilityId.
	 */
	public String getFacilityId() {
		return facilityId;
	}

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		if (isModifyDialog()) {
			return Messages.getString("modify");
		} else {
			return Messages.getString("register");
		}
	}

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}

	/**
	 * NodeInfoを返します。
	 * 
	 * @return NodeInfo
	 */
	public NodeInfo getNodeInfo() {
		return this.nodeInfo;
	}
	
	/**
	 *SNMPを利用してノード情報を生成します。<BR>
	 *
	 * @param pollingData SNMPポーリングのためのIPアドレスなど
	 * @param mode 編集可否モード
	 * @return ノード情報のプロパティ
	 */
	private static Property GetNodePropertyBySNMP(String ipAddress, int port, String community, String version,int mode) {

		try {
			NodeInfo nodeInfo = RepositoryEndpointWrapper.getNodePropertyBySNMP(ipAddress, port, community, version);
			m_log.info("snmp2 " + nodeInfo.getNodeFilesystemInfo().size());
			NodePropertyUtil.setDefaultNode(nodeInfo);
			m_log.info("snmp3 " + nodeInfo.getNodeFilesystemInfo().size());
			return NodePropertyUtil.node2property(nodeInfo, mode, Locale.getDefault());
		} catch (InvalidRole_Exception e) {
			MessageDialog.openInformation(null, Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
		} catch (Exception e) {
			m_log.warn("GetNodePropertyBySNMP(), " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}
		return null;
	}
}