/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.snmptrap.dialog;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.RequiredFieldColorConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.calendar.composite.CalendarIdListComposite;
import com.clustercontrol.composite.CommonTableViewer;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.composite.MonitorBasicScopeComposite;
import com.clustercontrol.monitor.run.dialog.CommonMonitorDialog;
import com.clustercontrol.monitor.util.MonitorSettingEndpointWrapper;
import com.clustercontrol.notify.composite.NotifyInfoComposite;
import com.clustercontrol.notify.util.NotifyGroupIdGenerator;
import com.clustercontrol.snmptrap.bean.MonitorTrapConstant;
import com.clustercontrol.snmptrap.composite.OidListComposite;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.monitor.InvalidRole_Exception;
import com.clustercontrol.ws.monitor.MonitorDuplicate_Exception;
import com.clustercontrol.ws.monitor.MonitorInfo;
import com.clustercontrol.ws.monitor.MonitorTrapValueInfo;
import com.clustercontrol.ws.monitor.TrapCheckInfo;

/**
 * SNMPTRAP監視作成・変更ダイアログクラス<BR>
 * 
 * @version 4.0.0
 * @since 2.1.0
 */
public class SnmpTrapCreateDialog extends CommonMonitorDialog {

	public static final int MAX_COLUMN = 20;
	public static final int MAX_COLUMN_SMALL = 15;
	public static final int WIDTH_TITLE = 6;
	public static final int WIDTH_VALUE = 2;

	// 後でpackするためsizeXはダミーの値。
	private final int sizeX = 750;
	private final int sizeY = 760;

	// ----- instance フィールド ----- //

	/** 入力値を保持するオブジェクト */
	private MonitorInfo inputData = null;

	/** 入力値の正当性を保持するオブジェクト */
	private ValidateResult validateResult = null;

	/** 変更対象の監視項目ID */
	private String monitorId = null;

	/** 監視基本情報 */
	private MonitorBasicScopeComposite monitorBasicComposite = null;

	/** カレンダ設定 */
	CalendarIdListComposite calendarComposite = null;

	/** コミュニティチェックボタン **/
	private Button buttonCommunityCheckOff = null;
	private Button buttonCommunityCheckOn = null;

	/** コミュニティ名 */
	private Text textCommunityName = null;

	/** 文字コード変換ボタン **/
	private Button buttonCharsetConvertOff = null;
	private Button buttonCharsetConvertOn = null;

	/** 変換文字コード名 */
	private Text textCharsetName = null;

	/** OIDテーブル */
	private OidListComposite oidListComposite = null;

	/** テーブルビューア */
	private CommonTableViewer tableViewer = null;

	/** 通知情報 */
	private NotifyInfoComposite notifyInfoComposite = null;


	/** この設定を有効にする */
	private Button confirmValid = null;


	/****/

	// ----- コンストラクタ ----- //

	/**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 * @param monitorType 監視判定タイプ
	 */
	public SnmpTrapCreateDialog(Shell parent) {
		super(parent);
		setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);
	}

	/**
	 * 変更用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 * @param monitorId 変更する監視項目ID
	 * @param monitorType 監視判定タイプ
	 */
	public SnmpTrapCreateDialog(Shell parent, String monitorId) {
		super(parent);
		setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);

		this.monitorId = monitorId;
	}

	// ----- instance メソッド ----- //

	/**
	 * ダイアログの初期サイズを返します。
	 * 
	 * @return 初期サイズ
	 */
	@Override
	protected Point getInitialSize() {
		return new Point(sizeX, sizeY);
	}

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent
	 *            親のインスタンス
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		Shell shell = this.getShell();

		// タイトル
		shell.setText(Messages.getString("dialog.snmptrap.create.modify"));

		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;

		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.numColumns = MAX_COLUMN;
		parent.setLayout(layout);

		// 監視基本情報
		//SNMPトラップでは未登録ノードからのトラップを受け付けるようにするので、
		//第３引数をtrueとする。
		monitorBasicComposite = new MonitorBasicScopeComposite(parent, SWT.NONE ,true);
		gridData = new GridData();
		gridData.horizontalSpan =MAX_COLUMN_SMALL;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		monitorBasicComposite.setLayoutData(gridData);

		/*
		 * カレンダ
		 */
		calendarComposite = new CalendarIdListComposite(parent, SWT.NONE, true);
		gridData = new GridData();
		gridData.horizontalSpan = MAX_COLUMN_SMALL;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		calendarComposite.setLayoutData(gridData);

		/*
		 * トラップ定義グループ
		 */
		// グループ
		Group groupCheckRule = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = MAX_COLUMN;
		groupCheckRule.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = MAX_COLUMN;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		groupCheckRule.setLayoutData(gridData);
		groupCheckRule.setText(Messages.getString("trap.definition"));
		
		/*
		 * コミュニティ
		 */
		Group groupCommunity = new Group(groupCheckRule, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = MAX_COLUMN;
		groupCommunity.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = MAX_COLUMN;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		groupCommunity.setLayoutData(gridData);
		groupCommunity.setText(Messages.getString("community"));

		// ボタン
		this.buttonCommunityCheckOff = new Button(groupCommunity, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.buttonCommunityCheckOff.setLayoutData(gridData);
		this.buttonCommunityCheckOff.setText(Messages.getString("community.no.check"));
		// ボタン
		this.buttonCommunityCheckOn = new Button(groupCommunity, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.buttonCommunityCheckOn.setLayoutData(gridData);
		this.buttonCommunityCheckOn.setText(Messages.getString("community.name") + " : ");
		// テキスト
		this.textCommunityName = new Text(groupCommunity, SWT.BORDER | SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.textCommunityName.setLayoutData(gridData);
		this.textCommunityName.addModifyListener(new ModifyListener() {
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});
		this.buttonCommunityCheckOff.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				update();
				textCommunityName.setEnabled(false);
			}
		});
		this.buttonCommunityCheckOn.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				update();
				textCommunityName.setEnabled(true);
			}
		});

		// 空白
		label = new Label(groupCommunity, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = MAX_COLUMN - WIDTH_TITLE *3;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		/*
		 * 文字コード
		 */
		Group groupCharset = new Group(groupCheckRule, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = MAX_COLUMN;
		groupCharset.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = MAX_COLUMN;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		groupCharset.setLayoutData(gridData);
		groupCharset.setText(Messages.getString("charset.convert"));

		// ボタン
		this.buttonCharsetConvertOff = new Button(groupCharset, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.buttonCharsetConvertOff.setLayoutData(gridData);
		this.buttonCharsetConvertOff.setText(Messages.getString("charset.no.convert"));
		// ボタン
		this.buttonCharsetConvertOn = new Button(groupCharset, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.buttonCharsetConvertOn.setLayoutData(gridData);
		this.buttonCharsetConvertOn.setText(Messages.getString("charset.snmptrap.code") + " : ");
		// テキスト
		this.textCharsetName = new Text(groupCharset, SWT.BORDER | SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.textCharsetName.setLayoutData(gridData);
		this.textCharsetName.addModifyListener(new ModifyListener() {
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});
		this.buttonCharsetConvertOff.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				update();
				textCharsetName.setEnabled(false);
			}
		});
		this.buttonCharsetConvertOn.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				update();
				textCharsetName.setEnabled(true);
			}
		});

		// 空白
		label = new Label(groupCharset, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = MAX_COLUMN - WIDTH_TITLE*3;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		/*
		 * OIDテーブル
		 */
		Group groupOid = new Group(groupCheckRule, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 1;
		groupOid.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = MAX_COLUMN;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		groupOid.setLayoutData(gridData);
		groupOid.setText("OID");

		// テキスト
		this.oidListComposite = new OidListComposite(groupOid, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = MAX_COLUMN;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		this.oidListComposite.setLayoutData(gridData);

		/*
		 * 通知情報の属性グループ
		 */
		// グループ
		Group groupNotifyAttribute = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 1;
		groupNotifyAttribute.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = MAX_COLUMN_SMALL;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupNotifyAttribute.setLayoutData(gridData);
		groupNotifyAttribute.setText(Messages.getString("notify.attribute"));
		
		// 通知情報（通知ID，アプリケーションID，実行）
		this.notifyInfoComposite = new NotifyInfoComposite(groupNotifyAttribute, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 1;
		gridData.horizontalAlignment = GridData.HORIZONTAL_ALIGN_CENTER;
		gridData.grabExcessHorizontalSpace = true;
		notifyInfoComposite.setLayoutData(gridData);

		/*
		 * 有効／無効
		 */
		this.confirmValid = new Button(parent, SWT.CHECK);
		gridData = new GridData();
		gridData.horizontalSpan = MAX_COLUMN_SMALL;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.confirmValid.setLayoutData(gridData);
		this.confirmValid.setText(Messages.getString("setting.valid.confirmed"));

		// ラインを引く
		Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = MAX_COLUMN;
		line.setLayoutData(gridData);

		//ダイアログのサイズ調整（pack:resize to be its preferred size）
		shell.pack();
		shell.setSize(new Point(800, shell.getSize().y));
		
		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);


		// 初期表示
		MonitorInfo info = null;
		if(this.monitorId == null){
			// 作成の場合
			info = new MonitorInfo();
			this.setInfoInitialValue(info);
		} else {
			// 変更の場合、情報取得
			try {
				info = MonitorSettingEndpointWrapper.getMonitor(this.monitorId, HinemosModuleConstant.MONITOR_SNMPTRAP);
			} catch (Exception e) {
				String errMessage = "";
				if (e instanceof InvalidRole_Exception) {
					// アクセス権なしの場合、エラーダイアログを表示する
					MessageDialog.openInformation(
							null,
							Messages.getString("message"),
							Messages.getString("message.accesscontrol.16"));
				} else {
					// 上記以外の例外
					errMessage = ", " + e.getMessage();
				}

				MessageDialog.openInformation(
						null,
						Messages.getString("message"),
						Messages.getString("message.traputil.4") + errMessage);
			}
		}
		this.setInputData(info);

	}

	/**
	 * 入力内容を返します。
	 * 
	 * @return 入力内容を保持した通知情報
	 */
	@Override
	public MonitorInfo getInputData() {
		return this.inputData;
	}

	/**
	 * 各項目に入力値を設定します。
	 * 
	 * @param monitor
	 *            設定値として用いる監視情報
	 */
	@Override
	protected void setInputData(MonitorInfo monitor) {

		// 監視基本情報
		monitorBasicComposite.setInputData(monitor);
		TrapCheckInfo checkInfo = monitor.getTrapCheckInfo();
		if (checkInfo == null) {
			checkInfo = new TrapCheckInfo();
			checkInfo.setCheckMode(0);
			checkInfo.setCommunityCheck(0);
			checkInfo.setCharsetConvert(0);
		}
		List<MonitorTrapValueInfo> monitorTrapValueInfoList = monitor.getTrapValueInfo();
		if (monitorTrapValueInfoList == null) {
			monitorTrapValueInfoList = new ArrayList<MonitorTrapValueInfo>();
		}

		// カレンダID
		if (monitor.getCalendarId() != null) {
			calendarComposite.setText(monitor.getCalendarId());
		}

		// コミュニティ名
		if(checkInfo.getCommunityName() != null){
			textCommunityName.setText(checkInfo.getCommunityName());
		}
		if (checkInfo.getCommunityCheck() == MonitorTrapConstant.COMMUNITY_CHECK_ON) {
			buttonCommunityCheckOn.setSelection(true);
			buttonCommunityCheckOff.setSelection(false);
			textCommunityName.setEnabled(true);
		} else {
			buttonCommunityCheckOn.setSelection(false);
			buttonCommunityCheckOff.setSelection(true);
			textCommunityName.setEnabled(false);
		}

		// 文字コード
		if(checkInfo.getCharsetName() != null){
			textCharsetName.setText(checkInfo.getCharsetName());
		}
		if (checkInfo.getCharsetConvert() == MonitorTrapConstant.CHARSET_CONVERT_ON) {
			buttonCharsetConvertOn.setSelection(true);
			buttonCharsetConvertOff.setSelection(false);
			textCharsetName.setEnabled(true);
		} else {
			buttonCharsetConvertOn.setSelection(false);
			buttonCharsetConvertOff.setSelection(true);
			textCharsetName.setEnabled(false);
		}

		//OID情報
		oidListComposite.setCheckMode(checkInfo.getCheckMode());
		oidListComposite.setMonitorId(monitor.getMonitorId());
		oidListComposite.setOidList(monitor.getTrapValueInfo());
		oidListComposite.update();

		//通知情報の設定
		if(monitor.getNotifyId() != null)
			this.notifyInfoComposite.setNotify(monitor.getNotifyId());

		//アプリケーション
		if (monitor.getApplication() != null) {
			this.notifyInfoComposite.setApplication(monitor.getApplication());
		}


		// 有効／無効
		//if (monitor.getValid() == ValidConstant.TYPE_VALID) {
		if (monitor.getMonitorFlg() == ValidConstant.TYPE_VALID) {
			this.confirmValid.setSelection(true);
		}
	}

	/**
	 * 入力値を用いて通知情報を生成します。
	 * 
	 * @return 入力値を保持した通知情報
	 */
	@Override
	protected MonitorInfo createInputData() {

		// 初期化
		MonitorInfo info = new MonitorInfo();
		info.setMonitorTypeId(HinemosModuleConstant.MONITOR_SNMPTRAP);
		info.setMonitorType(MonitorTypeConstant.TYPE_TRAP);

		// 監視基本情報
		validateResult = monitorBasicComposite.createInputData(info);
		if(validateResult != null){
			return null;
		}

		TrapCheckInfo trapInfo = new TrapCheckInfo();
		trapInfo.setCheckMode(0);
		trapInfo.setCommunityCheck(0);
		trapInfo.setCharsetConvert(0);
		trapInfo.setMonitorId(info.getMonitorId());
		trapInfo.setMonitorTypeId(HinemosModuleConstant.MONITOR_SNMPTRAP);
		info.setTrapCheckInfo(trapInfo);
		List<MonitorTrapValueInfo> monitorTrapValueInfoList = info.getTrapValueInfo();
		monitorTrapValueInfoList = new ArrayList<MonitorTrapValueInfo>();

		// カレンダID
		if (calendarComposite.getText() != null
				&& !"".equals((calendarComposite.getText()).trim())) {
			info.setCalendarId(calendarComposite.getText());
		}

		// コミュニティ名
		if (this.buttonCommunityCheckOn.getSelection()) {
			(info.getTrapCheckInfo()).setCommunityCheck(MonitorTrapConstant.COMMUNITY_CHECK_ON);
		} else {
			(info.getTrapCheckInfo()).setCommunityCheck(MonitorTrapConstant.COMMUNITY_CHECK_OFF);
		}
		if (this.textCommunityName.getText() != null
				&& !"".equals((this.textCommunityName.getText()).trim())) {
			(info.getTrapCheckInfo()).setCommunityName(textCommunityName.getText());
		}

		// 文字コード
		if (this.buttonCharsetConvertOn.getSelection()) {
			(info.getTrapCheckInfo()).setCharsetConvert(MonitorTrapConstant.CHARSET_CONVERT_ON);
		} else {
			(info.getTrapCheckInfo()).setCharsetConvert(MonitorTrapConstant.CHARSET_CONVERT_OFF);
		}
		if (this.textCharsetName.getText() != null
				&& !"".equals((this.textCharsetName.getText()).trim())) {
			(info.getTrapCheckInfo()).setCharsetName(textCharsetName.getText());
		}

		//OIDのチェック方法を設定します。

		(info.getTrapCheckInfo()).setCheckMode(oidListComposite.getCheckMode());

		//OID情報
		List<MonitorTrapValueInfo> monitorTrapValueInfoList_old = info.getTrapValueInfo();
		monitorTrapValueInfoList_old.clear();
		if (oidListComposite.getOidList() != null) {
			monitorTrapValueInfoList_old.addAll(oidListComposite.getOidList());
		}
		for(int i = 0; i < info.getTrapValueInfo().size(); i++){
			MonitorTrapValueInfo oidInfo = info.getTrapValueInfo().get(i);
			if(oidInfo.getMonitorId() == null ||
					oidInfo.getMonitorId().equals(info.getMonitorId())){
				oidInfo.setMonitorId(info.getMonitorId());
			}
		}

		// 通知関連情報とアプリケーションの設定
		// 通知グループIDの設定
		info.setNotifyGroupId(NotifyGroupIdGenerator.createNotifyGroupId(HinemosModuleConstant.MONITOR_SNMPTRAP, info.getMonitorId()));
		validateResult = this.notifyInfoComposite.createInputData(info);
		if (validateResult != null) {
			if(validateResult.getID() == null){	// 通知ID警告用出力
				if(!displayQuestion(validateResult)){
					validateResult = null;
					return null;
				}
			}
			else{	// アプリケーション未入力チェック
				return null;
			}
		}

		// 監視感覚
		info.setRunInterval(0);

		// 有効/無効
		if (this.confirmValid.getSelection()) {
			info.setMonitorFlg(ValidConstant.TYPE_VALID);
		} else {
			info.setMonitorFlg(ValidConstant.TYPE_INVALID);
		}

		return info;
	}

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
	@Override
	protected ValidateResult validate() {
		// 入力値生成
		this.inputData = this.createInputData();

		if (this.inputData != null) { //&& this.validateResult == null) {
			return super.validate();
		} else {
			return validateResult;
		}
	}

	/**
	 * 入力値をマネージャに登録します。
	 * 
	 * @return true：正常、false：異常
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#action()
	 */
	@Override
	protected boolean action() {
		boolean result = false;

		MonitorInfo info = this.inputData;
		if(info != null){
			String[] args = { info.getMonitorId() };
			if(this.monitorId == null){
				// 作成の場合
				try {
					result = MonitorSettingEndpointWrapper.addMonitor(info);

					if(result){
						MessageDialog.openInformation(
								null,
								Messages.getString("successful"),
								Messages.getString("message.monitor.33", args));
					} else {
						MessageDialog.openError(
								null,
								Messages.getString("failed"),
								Messages.getString("message.monitor.34", args));
					}
				} catch (MonitorDuplicate_Exception e) {
					// 監視項目IDが重複している場合、エラーダイアログを表示する
					MessageDialog.openInformation(
							null,
							Messages.getString("message"),
							Messages.getString("message.monitor.53", args));

				} catch (Exception e) {
					String errMessage = "";
					if (e instanceof InvalidRole_Exception) {
						// アクセス権なしの場合、エラーダイアログを表示する
						MessageDialog.openInformation(
								null,
								Messages.getString("message"),
								Messages.getString("message.accesscontrol.16"));
					} else {
						errMessage = ", " + e.getMessage();
					}

					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.monitor.34", args) + errMessage);
				}
			} else {
				// 変更の場合
				String errMessage = "";
				try {
					result = MonitorSettingEndpointWrapper.modifyMonitor(info);
				} catch (InvalidRole_Exception e) {
					// アクセス権なしの場合、エラーダイアログを表示する
					MessageDialog.openInformation(
							null,
							Messages.getString("message"),
							Messages.getString("message.accesscontrol.16"));
				} catch (Exception e) {
					errMessage = ", " + e.getMessage();
				}

				if(result){
					MessageDialog.openInformation(
							null,
							Messages.getString("successful"),
							Messages.getString("message.monitor.35", args));
				}
				else{
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.monitor.36", args) + errMessage);
				}
			}
		}
		return result;
	}

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		return Messages.getString("ok");
	}

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}

	/**
	 * tableViewerを返します。
	 * 
	 * @return tableViewer
	 */
	public CommonTableViewer getTableViewer() {
		return this.tableViewer;
	}

	/**
	 * 必須項目の描画更新
	 */
	@Override
	protected void update() {
		super.update();

		// コミュニティチェックボタンが有効な場合のみ
		if(this.buttonCommunityCheckOn.getSelection() && "".equals(this.textCommunityName.getText())){
			this.textCommunityName.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.textCommunityName.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}

		// 文字コードチェックボタンが有効な場合のみ
		if(this.buttonCharsetConvertOn.getSelection() && "".equals(this.textCharsetName.getText())){
			this.textCharsetName.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.textCharsetName.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
	}
}