/*

Copyright (C) since 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.accesscontrol.ejb.session;

import java.rmi.RemoteException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.UserDuplicate;
import com.clustercontrol.fault.UserNotFound;

import com.clustercontrol.accesscontrol.bean.UserFilterInfo;
import com.clustercontrol.accesscontrol.bean.UserInfo;
import com.clustercontrol.accesscontrol.factory.LoginUserModifier;
import com.clustercontrol.accesscontrol.factory.LoginUserSelector;
import com.clustercontrol.accesscontrol.util.UserValidator;
import com.clustercontrol.accesscontrol.util.VersionUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobUserRelationMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobUserRelationMasterUtil;

/**
 * アクセス管理機能を実現するSession Bean<BR>
 * 
 * @ejb.bean name="AccessController"
 *    jndi-name="AccessController"
 *    type="Stateless"
 *    transaction-type="Container"
 *    view-type="local"
 * 
 * @ejb.transaction
 *    type="Required"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=UserLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=UserRoleLocal"
 */
public abstract class AccessControllerBean implements javax.ejb.SessionBean {

	/** ログ出力のインスタンス */
	private static Log m_log = LogFactory.getLog( AccessControllerBean.class );

	/** コンテキスト情報 */
	private SessionContext m_context;

	/**
	 * コンテキスト情報を設定する。<BR>
	 * 
	 * Session Beanがインスタンスプールに格納される際に行う処理を実装する。<BR>
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	@Override
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}

	/**
	 * ログインチェックの為、本メソッドを使用します。
	 * 
	 * @ejb.interface-method
	 */
	public void checkLogin() {

	}

	/**
	 * ユーザ検索条件に基づき、ユーザ一覧情報を取得する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param filter ユーザ検索条件（ユーザフィルタダイアログの入力値）
	 * @return ユーザ情報のリスト
	 * @throws HinemosUnknown
	 * @throws UserNotFound
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserSelector#getUserList(Property)
	 */
	public ArrayList<UserInfo> getUserInfoList(UserFilterInfo filter) throws UserNotFound, HinemosUnknown {
		try {
			if (filter == null) {
				return LoginUserSelector.getUserInfoList();
			} else {
				return LoginUserSelector.getUserInfoList(filter);
			}
		} catch (UserNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getUserInfoList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * 自身のユーザ情報を取得する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @return 自身のユーザ情報
	 * @throws HinemosUnknown
	 * @throws UserNotFound
	 */
	public UserInfo getOwnUserInfo() throws HinemosUnknown, UserNotFound {
		String loginUser = m_context.getCallerPrincipal().getName();
		m_log.debug("getOwnData() loginUser = " + loginUser);

		try {
			return LoginUserSelector.getUserInfo(loginUser);
		} catch (Exception e) {
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}


	/**
	 * ユーザ情報を取得する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param userId ユーザID
	 * @param mode 取得モード
	 * @param locale ロケール情報
	 * @return ユーザ情報
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.bean.PropertyConstant
	 * @see com.clustercontrol.accesscontrol.factory.UserProperty#getProperty(String, int, Locale)
	 */
	public UserInfo getUserInfo(String userId) throws HinemosUnknown {
		try {
			return LoginUserSelector.getUserInfo(userId);
		} catch (Exception e) {
			m_log.error("getUserInfo " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

	}

	/**
	 * ユーザ認証する<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param username ユーザ名
	 * @param password パスワード
	 * @param role
	 * @return ユーザ情報
	 * @throws InvalidUserPass
	 * @throws HinemosUnknown
	 */
	public void getUserInfoByPassword(String username, String password, ArrayList<String> roleList)
	throws InvalidUserPass, InvalidRole, HinemosUnknown {
		LoginUserSelector.getUserInfoByPassword(username, password, roleList);
	}

	/**
	 * ユーザを追加する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param info ユーザ情報
	 * @throws HinemosUnknown
	 * @throws ParseException
	 * @throws UserDuplicate
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserModifier#addUser(Property, String, String)
	 */
	public void addUserInfo(UserInfo info) throws HinemosUnknown, UserDuplicate, InvalidSetting {
		m_log.warn("user=" + m_context.getCallerPrincipal().getName());

		try {
			// 入力チェック
			UserValidator.validateUserInfo(info);

			/** メイン処理 */
			LoginUserModifier.addUserInfo(info, m_context.getCallerPrincipal().getName());
		} catch (UserDuplicate e) {
			m_context.setRollbackOnly();
			throw e;
		} catch (InvalidSetting e) {
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("addUser() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * ユーザ情報を変更する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param info ユーザ情報
	 * @throws HinemosUnknown
	 * @throws UserNotFound
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserModifier#modifyUser(Property, String)
	 */
	public void modifyUserInfo(UserInfo info) throws HinemosUnknown, UserNotFound, InvalidSetting {
		m_log.warn("user=" + m_context.getCallerPrincipal().getName());

		/** メイン処理 */
		try {
			//入力チェック
			UserValidator.validateUserInfo(info);

			/** メイン処理 */
			LoginUserModifier.modifyUserInfo(info, m_context.getCallerPrincipal().getName());
		} catch (UserNotFound e) {
			throw e;
		} catch (RemoveException e) {
			m_log.warn("modifyUserInfo() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (InvalidSetting e) {
			m_log.warn("modifyUserInfo() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		} catch (CreateException e) {
			m_log.warn("modifyUserInfo() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.warn("modifyUserInfo() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * ユーザ情報を削除する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param userId ユーザID
	 * @throws HinemosUnknown
	 * @throws UserNotFound
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserModifier#deleteUser(String, String)
	 */
	public void deleteUserInfo(String userId) throws HinemosUnknown, UserNotFound {
		/** メイン処理 */
		try {
			LoginUserModifier.deleteUserInfo(userId, m_context.getCallerPrincipal().getName());
		} catch (UserNotFound e) {
			throw e;
		} catch (NamingException e) {
			m_log.warn("deleteUserInfo() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (RemoveException e) {
			m_log.warn("deleteUserInfo() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * 自分自身のパスワードを変更する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param userId ユーザID
	 * @param password パスワード
	 * @throws HinemosUnknown
	 * @throws UserNotFound
	 * 
	 */
	public void changeOwnPassword(String password) throws HinemosUnknown, UserNotFound {
		m_log.debug("changeOwnPassword() password = " + password);

		String loginUser = m_context.getCallerPrincipal().getName();
		m_log.debug("changeOwnPassword() loginUser = " + loginUser);

		try {
			LoginUserModifier.modifyUserPassword(loginUser, password);
		} catch (UserNotFound e) {
			throw e;
		} catch (NamingException e) {
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * パスワードを変更する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param userId ユーザID
	 * @param password パスワード
	 * @throws HinemosUnknown
	 * @throws UserNotFound
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserModifier#modifyUserPassword(String, String)
	 */
	public void changePassword(String userId, String password) throws HinemosUnknown, UserNotFound {
		/** メイン処理 */
		try{
			LoginUserModifier.modifyUserPassword(userId, password);
		} catch (UserNotFound e){
			m_log.warn("changePassword() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		} catch (Exception e){
			m_log.warn("changePassword() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * ログインしているユーザが指定したユーザ権限を持っているかどうかを確認する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param userRole ユーザ権限名
	 * @return ユーザ権限を保持していればtrue, そうでなければfalse
	 * @throws HinemosUnknown
	 */
	public boolean isPermission(String userRole) throws HinemosUnknown {
		/** メイン処理 */
		try{
			return m_context.isCallerInRole(userRole);
		} catch (Exception e){
			m_log.warn("isPermission() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * ログインユーザのユーザ名を取得する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @return ユーザ名
	 * @throws HinemosUnknown
	 * @throws UserNotFound
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserSelector#getUserName(String)
	 */
	public String getUserName() throws HinemosUnknown, UserNotFound {
		try {
			return LoginUserSelector.getUserName(m_context.getCallerPrincipal().getName());
		} catch (UserNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.warn("getUserName() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}

	}

	/**
	 * バージョン番号を取得する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @return バージョン番号
	 */
	public String getVersion() {
		return VersionUtil.getVersion();
	}

	/**
	 * 引数で指定されたユーザがジョブユニットの管理ユーザに設定されているか確認する<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param userId
	 * 	 * @return 引数で指定されたユーザが管理ユーザに設定されているジョブユニットIDのリスト
	 * @throws UserNotFound
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo
	 */
	public ArrayList<String> checkJobManagementUser(String userId) throws UserNotFound, HinemosUnknown {
		ArrayList<String> ret = new ArrayList<String>();

		Collection<JobUserRelationMasterLocal> userList;
		try {
			userList = JobUserRelationMasterUtil.getLocalHome().findByUserId(userId);
			for (JobUserRelationMasterLocal userInfo : userList) {
				ret.add(userInfo.getJobunit_id());
			}
		} catch (FinderException e) {
			m_log.warn("checkJobManagementUser() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new UserNotFound(e.getMessage(), e);
		} catch (Exception e) {
			m_log.warn("checkJobManagementUser() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}

		return ret;
	}
}

