/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.calendar.bean;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.CalendarInvalid;

/**
 * カレンダの実行例外を格納するDTOクラス
 * 
 * @since 4.0
 * @author Tomoya Takahata
 */
public final class CalendarIrregularDTO {

	private static Log log = LogFactory.getLog(CalendarIrregularDTO.class);

	public static enum CalendarIrregularType { ON, OFF }

	private Long beginDate;
	private Long endDate;
	private CalendarIrregularType type;
	private String description;

	private Long registerDate;
	private String registerUser;
	private Long updateDate;
	private String updateUser;

	/**
	 * コンストラクタ(no-arugument for JAXB)
	 */
	public CalendarIrregularDTO() {

	}

	/**
	 * コンストラクタ
	 * @param beginDate 開始日時
	 * @param endDate 終了日時
	 * @param type 例外種別
	 * @param description 実行例外の説明
	 * @param registerDate 登録日時
	 * @param registerUser 登録ユーザ名
	 * @param updateDate 更新日時
	 * @param updateUser 更新ユーザ名
	 * @throws CalendarInvalid 妥当でない値が存在した場合
	 */
	public CalendarIrregularDTO(Long beginDate, Long endDate, CalendarIrregularType type, String description,
			Long registerDate, String registerUser, Long updateDate, String updateUser) throws CalendarInvalid {
		this.beginDate = beginDate;
		this.endDate = endDate;
		this.type = type;
		this.description = description;
		this.registerDate = registerDate;
		this.registerUser = registerUser;
		this.updateDate = updateDate;
		this.updateUser = updateUser;

		validate();
	}

	/**
	 * 開始日時を格納する。<br/>
	 * @param beginDate 開始日時
	 */
	public void setBeginDate(Long beginDate) {
		this.beginDate = beginDate;
	}

	/**
	 * 開始日時を返す。<br/>
	 * @return 開始日時
	 */
	public Long getBeginDate() {
		return beginDate;
	}

	/**
	 * 終了日時を格納する。<br/>
	 * @param endDate 終了日時
	 */
	public void setEndDate(Long endDate) {
		this.endDate = endDate;
	}

	/**
	 * 終了日時を返す。<br/>
	 * @return 終了日時
	 */
	public Long getEndDate() {
		return endDate;
	}

	/**
	 * 実行例外の種別を格納する。<br/>
	 * @param type 実行例外の種別
	 */
	public void setType(CalendarIrregularType type) {
		this.type = type;
	}

	/**
	 * 実行例外の種別を返す。<br/>
	 * @return 実行例外の種別
	 */
	public CalendarIrregularType getType() {
		return type;
	}

	/**
	 * 実行例外の説明を格納する。<br/>
	 * @param description 実行例外の説明
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * 実行例外の説明を返す。<br/>
	 * @return 実行例外の説明
	 */
	public String getDescription() {
		return description;
	}

	/**
	 * 実行例外の登録日時を格納する。<br/>
	 * @param registerDate 実行例外の登録日時
	 */
	public void setRegisterDate(Long registerDate) {
		this.registerDate = registerDate;
	}

	/**
	 * 実行例外の登録日時を返す。<br/>
	 * @return 実行例外の登録日時
	 */
	public Long getRegisterDate() {
		return registerDate;
	}

	/**
	 * 実行例外の登録ユーザ名を格納する。<br/>
	 * @param registerUser 実行例外の登録ユーザ名
	 */
	public void setRegisterUser(String registerUser) {
		this.registerUser = registerUser;
	}

	/**
	 * 実行例外の登録ユーザ名を返す。<br/>
	 * @return 実行例外の登録ユーザ名
	 */
	public String getRegisterUser() {
		return registerUser;
	}

	/**
	 * 実行例外の更新日時を格納する。<br/>
	 * @param updateDate 実行例外の更新日時
	 */
	public void setUpdateDate(Long updateDate) {
		this.updateDate = updateDate;
	}

	/**
	 * 実行例外の更新日時を返す。<br/>
	 * @return 実行例外の更新日時
	 */
	public Long getUpdateDate() {
		return updateDate;
	}

	/**
	 * 実行例外の更新ユーザ名を格納する。<br/>
	 * @param updateUser 実行例外の更新ユーザ名
	 */
	public void setUpdateUser(String updateUser) {
		this.updateUser = updateUser;
	}

	/**
	 * 実行例外の更新ユーザ名を返す。<br/>
	 * @return 実行例外の更新ユーザ名
	 */
	public String getUpdateUser() {
		return updateUser;
	}

	/**
	 * コピーされた実行例外インスタンスを返す。<br/>
	 * @return コピーされた実行例外インスタンス
	 */
	@Override
	public CalendarIrregularDTO clone() {
		CalendarIrregularDTO dto = null;
		try {
			dto = new CalendarIrregularDTO(beginDate, endDate, getType(), getDescription(),
					getRegisterDate(), getRegisterUser(), getUpdateDate(), getUpdateUser());
		} catch (CalendarInvalid e) {
			log.warn("cloning CalendarIrregularDTO is failed. (" + this + ")", e);
		}
		return dto;
	}

	/**
	 * メンバ変数の妥当性を確認する。<br/>
	 * @throws CalendarInvalid 妥当でない値が存在した場合
	 */
	public void validate() throws CalendarInvalid {
		// null check
		if (beginDate == null) {
			throw new CalendarInvalid("beginDate must be defined(not null). (" + this + ")");
		}
		if (endDate == null) {
			throw new CalendarInvalid("endDate must be defined(not null). (" + this + ")");
		}
		if (type == null) {
			throw new CalendarInvalid("type must be defined(not null). (" + this + ")");
		}

		// check consistency
		if (beginDate > endDate) {
			throw new CalendarInvalid("end date must be after beign date. (" + this + ")");
		}
	}

	@Override
	public String toString() {
		return "CalendarIrregularDTO (beginDate = " + beginDate
		+ ", endDate = " + endDate + ", type = " + CalendarIrregularType.ON + ")";
	}
}
